% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/collector.curve.R
\name{collector.curve}
\alias{collector.curve}
\title{Species–area (collector's curve) function}
\usage{
collector.curve(
  formula,
  data,
  x,
  times = 1000,
  replace = FALSE,
  prob = 0.95,
  spar = 0,
  xlab,
  ylab,
  plot = TRUE,
  long = FALSE,
  theme = "theme_classic"
)
}
\arguments{
\item{formula}{An optional formula specifying the relationship between taxa and sample units
(e.g., \code{Taxon ~ Sample}). If provided, the function extracts variables from \code{data}.
A third variable may be included to remove dead individuals (e.g., \code{Taxon ~ Sample - Dead}).}

\item{data}{A data frame containing the variables specified in \code{formula} ('long format').
It must contain one column representing the sample unit labels (e.g., quadrats or points)
and one column representing the taxon names of the individual plants. This argument accepts
the data frame used in the argument \code{x} in \code{\link{phytoparam}}.}

\item{x}{Species-by-sample matrix, with rows representing SUs and columns representing taxa
('wide format'). Can be either an abundance or presence–absence matrix. Ignored if
\code{formula} and \code{data} are used.}

\item{times}{Integer. Number of random permutations used in calculations. Default is 1000.
Larger values (> 1000) yield more stable estimates.}

\item{replace}{Logical. Indicates whether resampling is performed with replacement (\code{TRUE}, bootstrap)
or without replacement (\code{FALSE}, default).}

\item{prob}{Numeric. Probability level used for computing confidence intervals around
species accumulation (default = 0.95).}

\item{spar}{Numeric. Controls the smoothing parameter for plotted confidence intervals
via spline interpolation. Default = 0 (no smoothing).}

\item{xlab}{Character. Label for the x-axis in the plot. Default = \code{"Number of samples"}.}

\item{ylab}{Character. Label for the y-axis in the plot. Default = \code{"Number of species"}.}

\item{plot}{Logical. If \code{TRUE} (default), the species accumulation curve is plotted.}

\item{long}{Logical. If \code{TRUE}, returns detailed results, including the full set of
resampling matrices. Default = \code{FALSE}.}

\item{theme}{Character string specifying the ggplot2 theme to apply
(e.g., \code{"theme_classic"}, \code{"theme_bw"}, \code{"theme_minimal"}).
Defaults to \code{"theme_classic"}.}
}
\value{
If \code{long = FALSE} (default), returns a data frame with columns:
\itemize{
  \item \code{n}: number of sample units.
  \item \code{s}: mean number of species.
  \item \code{lower}: lower confidence interval bound.
  \item \code{upper}: upper confidence interval bound.
}
If \code{long = TRUE}, returns a list with:
\itemize{
  \item \code{matrix.s}: full matrix of species richness per permutation.
  \item \code{s}: the summarized data frame described above.
}
}
\description{
Computes species accumulation (collector's) curves based on sample units (SUs).
The function performs random resampling of the input matrix or data frame to estimate
the expected species richness per number of SUs, with confidence intervals derived
from multiple permutations.
}
\details{
Species accumulation curves are computed by sequentially adding sample units and recording
species richness across permutations. Confidence intervals are estimated from the empirical
distribution of resampled richness values. The plotted confidence intervals are smoothed
using spline interpolation if \code{spar} > 0.

It is recommended to assign the output to an object, as the complete output
(particularly with \code{long = TRUE}) can be large.
}
\note{
With \code{long = TRUE}, the function provides access to the complete set of resampling
results, useful for additional data analyses.
}
\examples{
## Using 'formula' (long format)
## Without smoothing confidence intervals
collector.curve(
  formula = Species ~ Plot - Morta,
  data = quadrat.df,
  times = 1000, long = FALSE, plot = TRUE
)

## Smoothing confidence intervals
collector.curve(
  formula = Species ~ Plot - Morta,
  data = quadrat.df,
  spar = 0.6, times = 1000, long = FALSE, plot = TRUE
)
\donttest{
## Using different plot themes
collector.curve(
  formula = Species ~ Plot - Morta,
  data = quadrat.df,
  times = 1000, long = FALSE, plot = TRUE, theme = "theme_light"
)
collector.curve(
  formula = Species ~ Plot - Morta,
  data = quadrat.df,
  times = 1000, long = FALSE, plot = TRUE, theme = "theme_bw"
)
collector.curve(
  formula = Species ~ Plot - Morta,
  data = quadrat.df,
  times = 1000, long = FALSE, plot = TRUE, theme = "theme_minimal"
)

## Using a matrix (wide format)
data.matrix <- with(
  quadrat.df,
  table(Plot, Species, exclude = "Morta")
)
collector.curve(x = data.matrix, times = 1000)

## Alternatively...
data.matrix <- as.matrix(
  xtabs(~ Plot + Species, data = quadrat.df, exclude = "Morta")
)
collector.curve(x = data.matrix, times = 1000)
}

}
\references{
Magurran, A. E. (1988). \emph{Ecological Diversity and Its Measurement}. Croom Helm.

Magurran, A. E. (2004). \emph{Measuring Biological Diversity}. Blackwell Publishing.
}
\author{
Rodrigo Augusto Santinelo Pereira \email{raspereira@usp.br}

Adriano Sanches Melo
}
