% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/phytoparam.R
\name{phytoparam}
\alias{phytoparam}
\title{Estimate phytosociological parameters and diversity indices}
\usage{
phytoparam(
  x,
  measure.label = NULL,
  h = "h",
  taxon = "taxon",
  family = "family",
  dead = "dead",
  circumference = TRUE,
  su = "quadrat",
  height = TRUE,
  quadrat = TRUE,
  su.size,
  d = "distance",
  shape.factor = 1,
  rm.dead = FALSE,
  check.spelling = TRUE
)
}
\arguments{
\item{x}{A \code{data.frame} containing the community sample data. See 'Details'.}

\item{measure.label}{Name of the column representing the circumference/diameter at
breast height. If omitted the function assumes the default names "cbh" or "dbh"
for circumference or diameter at breast height, respectively (see \code{circumference}).}

\item{h}{Name of the column representing trunk height. Default is "h".}

\item{taxon}{Name of the column representing the sampled taxa. Default is "taxon".
Use UTF-8 encoding; accents and special characters are not allowed.}

\item{family}{Name of the column representing the family names of the sampled taxa.
Default is "family". Used to calculate the number of individuals and number of
species per family. If you do not want these parameters, set \code{family = NA}.
Use \code{UTF-8} encoding; accents and special characters are not allowed.}

\item{dead}{String used to identify dead individuals. Default is "dead".}

\item{circumference}{Logical. If \code{TRUE} (default) circumference at breast height
was measured; otherwise "dbh" is assumed.}

\item{su}{Name of the column representing the sample-unit identifier.
Default is "quadrat" for the quadrat method and "point" for the
point-centered quarter method.}

\item{height}{Logical. If \code{FALSE} (default) trunk volume is not calculated.}

\item{quadrat}{Logical. If \code{TRUE} (default) data were sampled using the quadrat
method; if \code{FALSE}, the point-centered quarter method is assumed.}

\item{su.size}{Numeric scalar giving the quadrat area (\eqn{m^2}); required only if
\code{quadrat = TRUE}.}

\item{d}{Name of the column representing the point-to-tree distance; required only
if \code{quadrat = FALSE}. Default is "distance".}

\item{shape.factor}{Numeric value between in 0 and 1, indicating the trunk shape.
Value \code{1} assumes a perfect cylinder.}

\item{rm.dead}{Logical. If \code{FALSE} (default) phytosociological parameters for dead
individuals are calculated.}

\item{check.spelling}{Logical. If \code{TRUE} (default) taxon names are checked for
misspelling.}
}
\value{
An object of class \code{param} with two or four data frames:
\itemize{
  \item \code{data}: A \code{data.frame} containing the original community sample data,
  added with a column of the individual basal area (ABi).
  \item \code{global}: total parameters and diversity indices. Sampled area in hectares (ha),
  total density in individuals/ha, total dominance in \eqn{m^2} \eqn{ha^{-1}} (basal area) or \eqn{m^3} \eqn{ha^{-1}}
  (volume, when computed), and Shannon--Wiener H' in nats/individual (natural log).
  \item \code{param}: taxon-level table with observed number of individuals (N), absolute/relative density (ADe, RDe),
  absolute/relative frequency (AFr, RFr), absolute/relative dominance (ADo, RDo),
  absolute/relative volume (AVol, RVol), Importance Value Index (IV),
  and Cover Value Index (CV). Absolute parameters per hectare; relative parameters in percentage.
  \item \code{family}: If \code{family != NA}, a table listing the number of individuals and the number of species
  per family is presented.
  \item \code{vars}: A \code{list} containing objects used in the functions \code{\link{AGB}}, \code{\link{stats}} and \code{\link{stratvol}}.
  }
}
\description{
Estimate the phytosociological parameters and the Shannon--Wiener, Pielou, and
Simpson diversity indices, using the quadrat or the point-centered quarter methods.
}
\details{
The function estimates phytosociological parameters for tree communities sampled
by quadrat or point-centered quarter methods (\code{quadrat = TRUE} or \code{FALSE},
respectively).

For the quadrat method, \code{x} must contain columns for sample-unit labels,
taxon names, and "cbh" or "dbh" measurements for each sampled tree. Additionally,
trunk height and family can be included to estimate volume and family-level parameters.

For the point-centered quarter method, \code{x} must contain (in addition to the
mandatory quadrat columns) a column for the distance from the point to each individual.

The "cbh"/"dbh" column accepts multiple-stem notation, e.g. "17.1+8+5.7+6.8".
The plus sign delimits stems. Decimal delimiter may be period or comma; spaces
around "+" are ignored. Column names in \code{x} are coerced to lowercase at runtime,
making matching case-insensitive. If \code{x} contains the default column names, the
arguments \code{h}, \code{taxon}, \code{family}, \code{dead}, \code{su} and \code{d} can be omitted.

Unbiased absolute density for the point-centered quarter method follows
Pollard (1971) and Seber (1982).

\strong{Measurement units:} individual "cbh"/"dbh" in centimeters; trunk height and
point-to-individual distance in meters.
}
\examples{
## Quadrat method
quadrat.param <- phytoparam(
  x = quadrat.df, measure.label = "CBH", taxon = "Species",
  dead = "Morta", family = "Family", circumference = TRUE, su = "Plot",
  height = TRUE, su.size = 25, rm.dead = FALSE
)
summary(quadrat.param)
head(quadrat.param$data)
quadrat.param$global
quadrat.param$family
quadrat.param$param

## Point-centered quarter method
point.param <- phytoparam(
  x = point.df, measure.label = "CBH", taxon = "Species",
  dead = "Morta", family = "Family", circumference = TRUE, su = "Point",
  height = TRUE, quadrat = FALSE, d = "Distance", rm.dead = FALSE
)
summary(point.param)
head(point.param$data)
point.param$global
point.param$family
point.param$param

## Using default column names
point.default <- point.df
colnames(point.default) <- c("point", "family", "taxon", "distance", "cbh", "h")
point.param.default <- phytoparam(
  x = point.default, dead = "morta",
  circumference = TRUE, height = TRUE, quadrat = FALSE
)
summary(point.param.default)
point.param.default$global

## Plotting
plot(quadrat.param)
plot(point.param)
plot(point.param, theme = "theme_light")
plot(point.param, theme = "theme_bw")
plot(point.param, theme = "theme_minimal")

}
\references{
Pollard, J. H. (1971). On distance estimators of density in randomly distributed forests.
\emph{Biometrics}, 27, 991--1002.

Seber, G. A. F. (1982). \emph{The Estimation of Animal Abundance and Related Parameters}.
New York: Macmillan, pp. 41--45.
}
\seealso{
\code{\link{summary.param}}, \code{\link{plot.param}}
}
