#' @title Create Evaluation Criteria Plots
#' @description This function creates plots to compare evaluation criteria of landscape function analysis (LFA), rangeland health and condition.
#' @details
#' The function takes the output from RHC_function and creates evaluation plots for different criteria.
#' @param evaluation.criteria A data frame containing standardized data from the first function.
#' @param selected_columns A vector of column indices specifying which criteria to plot. If NULL, all columns will be plotted.
#' @param ncol Number of columns for arranging the plots. Default is 4.
#' @param plot_RHC A function that takes standardized data and generates evaluation plots for different criteria of landscape function analysis (LFA), rangeland health and condition.
#' @return A list of evaluation criteria(attributes) plots.
#' @examples
#' data(canopy_oc_file)
#' data(trait_file)
#' final_data_st <- prepare_RHC_data(canopy_oc_file, trait_file)
#' evaluation.criteria <- RHC_function(final_data_st)
#' # Plot all columns
#' plots_all <- plot_RHC(evaluation.criteria, ncol = 4)
#' # Plot specific columns
#' selected_columns <- c(4, 8)
#' plots_selected <- plot_RHC(evaluation.criteria, selected_columns, ncol = 2)
#' @importFrom ggplot2 ggplot
#' @importFrom ggplot2 aes
#' @importFrom ggplot2 geom_bar
#' @importFrom ggplot2 labs
#' @importFrom ggplot2 theme_minimal
#' @importFrom ggplot2 theme
#' @importFrom ggplot2 element_text
#' @importFrom ggplot2 margin
#' @importFrom ggplot2 element_blank
#' @importFrom ggplot2 element_line
#' @importFrom ggplot2 geom_text
#' @importFrom ggplot2 coord_cartesian
#' @importFrom ggplot2 guides
#' @importFrom ggplot2 guide_legend
#' @importFrom ggplot2 expand_limits
#' @importFrom gridExtra grid.arrange
#' @name plot_RHC
#' @export
plot_RHC <- function(evaluation.criteria, selected_columns = NULL, ncol = 4) {
  evaluation.criteria.t <- as.matrix(t(evaluation.criteria))

  create_df <- function(scores, source) {
    df <- data.frame(
      Variable = colnames(evaluation.criteria.t),
      Score = as.numeric(scores),
      Source = source
    )
    df$Variable <- factor(df$Variable, levels = rev(unique(df$Variable)))
    return(df)
  }

  create_plot <- function(df, color, title) {
    ggplot(df, aes(x = Score, y = Variable)) +
      geom_bar(stat = "identity", fill = color, width = 0.7) +
      labs(x = "Standardized value", y = "sample", title = title) +
      theme_minimal() +
      theme(
        axis.text.y = element_text(size = 11, hjust = 1, vjust = 0.1, color = "black"),
        axis.text.x = element_text(size = 12, color = "black"),
        axis.title.x = element_text(margin = margin(t = 5), face = "bold"),
        axis.title.y = element_text(face = "bold"),
        panel.grid = element_blank(),
        axis.line = element_line(color = "gray"),
        plot.title = element_text(hjust = 0.5, vjust = 0, face = "bold", size = 14)
      ) +
      geom_text(aes(label = sprintf("%.2f", Score)), vjust = 0.5, color = "black", size = 4, hjust = -0.1) +
      coord_cartesian(xlim = c(0, 1)) +
      expand_limits(x = max(df$Score) * 1.3)
  }

  if (is.null(selected_columns)) {
    selected_columns <- 1:ncol(evaluation.criteria)
  }

  colors <- c("#ff0033", "salmon", "yellow1", "springgreen3", "#00ff00", "cyan2", "deepskyblue", "dodgerblue3", "#E76BF3", "maroon2")

  plots.RHC <- list()
  for (i in selected_columns) {
    plot <- switch(i,
      `1` = create_plot(create_df(evaluation.criteria.t[1, ], "Total Stability (Standardized)"), "#ff0033", paste("Total Stability")),
      `2` = create_plot(create_df(evaluation.criteria.t[2, ], "Total Infiltration (Standardized)"), "salmon", paste("Total Infiltration")),
      `3` = create_plot(create_df(evaluation.criteria.t[3, ], "Total Nutrients (Standardized)"), "yellow1", paste("Total Nutrients")),
      `4` = create_plot(create_df(evaluation.criteria.t[4, ], "LFA (Standardized)"), "springgreen3", paste("LFA")),
      `5` = create_plot(create_df(evaluation.criteria.t[5, ], "Soil.Site.Stability (Standardized)"), "#00ff00", paste("Soil Site Stability")),
      `6` = create_plot(create_df(evaluation.criteria.t[6, ], "Hydrologic.Function (Standardized)"), "cyan2", paste("Hydrologic Function")),
      `7` = create_plot(create_df(evaluation.criteria.t[7, ], "Biotic.Integrity (Standardized)"), "deepskyblue", paste("Biotic Integrity")),
      `8` = create_plot(create_df(evaluation.criteria.t[8, ], "Rangeland.Health (Standardized)"), "dodgerblue3", paste("Rangeland Health")),
      `9` = create_plot(create_df(evaluation.criteria.t[9, ], "condition.4 (Standardized)"), "#E76BF3", paste("condition 4")),
      `10` = create_plot(create_df(evaluation.criteria.t[10, ], "condition.6 (Standardized)"), "maroon2", paste("condition 6"))
    )
    plots.RHC[[length(plots.RHC) + 1]] <- plot
  }
  grid.arrange(grobs = plots.RHC, ncol = ncol)
  return(plots.RHC)
}
