#' Plotting the clustring results
#'
#'
#' This function performs a PCA to reduce the dataset to two dimensions. Then, it draws the points, marks the center of the groups, the exact groups and the obtained groups.
#'
#' @param data Matrix with \code{dim(data)[1]} points of \code{dim(data)[2]} dimensions.
#' @param grouping List with information of the groups obtained using some clustering method. Each component of the list contains a vector with the points that belong to that group. More specifically, the list component i has a vector with the numbers of the row of the matrix \code{data} where the points belonging to the group i are.
#' @param exact_grouping List with the information of the real groups present in the data. Each component of the list contains a vector with the points that belong to that group. More specifically, the list component i has a vector with the numbers of the row of the matrix \code{data} where the points belonging to the group i are.
#' @param centers Matrix. Each row contains the center of each group. The groups are obtained using some clustering methods.
#' @param k Number. Number of groups.
#'
#'
#' @return Returns a plot where it is possible to visualize the he points, the center of the groups, the exact groups (represented in the type of point used to represent the data) and the obtained groups (observed in the geometric froms that join the points).
#'
#'
#'
#'
#' @examples
#'
#'data=iris[,-5]
#'exact_grouping=list(which(iris[,5]=="setosa"),
#'                    which(iris[,5]=="versicolor"),
#'                    which(iris[,5]=="virginica"))
#'
#'grouping=list(c(1:40),c(41:90),c(91:150))
#'k=3
#'centers=rbind(c(1,2,3,4),c(2,3,4,5),c(4,5,6,7))
#'
#'ClustPlot(data, grouping, exact_grouping,centers, k)
#'
#'
#' @export

ClustPlot<-function(data, grouping, exact_grouping, centers, k){


   ObtainedGroup=rep(0,dim(data)[1])
  for(i in 1:dim(data)[1]){
     ObtainedGroup[i]=encontrar_componente(grouping,i)
  }


  ExactGroup=rep(0,dim(data)[1])
  for(i in 1:dim(data)[1]){
    ExactGroup[i]=encontrar_componente(exact_grouping,i)
  }

  pc <- stats::prcomp(data,center = TRUE, scale = TRUE)
  pc.sum<-summary(pc)
  Var.Explained=pc.sum$importance[2,]#Variance information



  colnames(centers)=colnames(data)
  centers_pca=stats::predict(pc, centers)[, 1:2]


  PC1_c = centers_pca[, 1]
  PC2_c = centers_pca[, 2]
  Grupo_c = factor(c(1:k))
  centros_df=data.frame(PC1_c, PC2_c, Grupo_c)


  ind.coord=as.data.frame(factoextra::get_pca_ind(pc)$coord)
  ind.coord$ObtainedGroup  <- factor(ObtainedGroup)
  ind.coord$ExactGroup <-factor(ExactGroup)

  return_out<-ggpubr::ggscatter(
    ind.coord, x = "Dim.1", y = "Dim.2",
    color = "ObtainedGroup", palette = "npg", ellipse = TRUE, ellipse.type = "convex",
    shape = "ExactGroup", size = 1.5,  legend = "right", ggtheme = ggplot2::theme_bw(),
    xlab = paste0("Dim 1 (", Var.Explained[1], "% )" ),
    ylab = paste0("Dim 2 (", Var.Explained[2], "% )" )
  )+ ggplot2::geom_point(data = centros_df, ggplot2::aes(x = PC1_c, y = PC2_c, color = Grupo_c), shape = 8, size = 5)+
    ggplot2::theme(legend.position = "none")


  return(return_out)



}
