#' Step 6 of the Hartigan and Wong algorithm
#'
#' This function implements the Step 6 of the Hartigan and Wong (Hartigan and Wong, 1979) algorithm without imposing that the use of the euclidean distance  and without imposing that the centers of the groups are calculated by averaging the points. This function allows other distances to be used and allows the centers of the groups to be calculated in different ways.
#'
#' @param data Matrix with \code{dim(data)[1]} points of \code{dim(data)[2]} dimensions.
#' @param centers Matrix with \code{dim(centers)[1]} centers of \code{dim(centers)[2]} dimensions.
#' @param grouping List. Each component of the list contains a vector with the points that belong to that group. More specifically, the list component i has a vector with the numbers of the row of the matrix \code{data} where the points belonging to group i are.
#' @param distance Function. This function designs how the distance is going to be calculated. It must have as input two vectors and as output the distance of these vectors.
#' @param centers_function Function. This function designs how the centers of the groups will be calculated. It must have as input \code{data} and \code{grouping} and as output a matrix that has the centers. This matrix will have as many rows as centers.
#' @param Ic12_change Matrix. Contains IC1 and IC2 after the Step 4 is carried out. The first row contains the IC1 of each point. The second column contains the IC2 of each point. IC1 and IC2 are the closets and second closest cluster centers.
#' @param Ic12 Matrix. Contains IC1 and IC2 before the Step 4 is carried out. The first row contains the IC1 of each point. The second column contains the IC2 of each point. IC1 and IC2 are the closets and second closest cluster centers.
#' @param index Number. When a point is reallocated, index becomes zero.
#'
#'
#' @return Returns a list with:
#'
#' \itemize{
#'   \item centers: the information of the centers updated. Matrix with \code{dim(centers)[1]} centers of \code{dim(centers)[2]} dimensions.
#'   \item  IC1andIC2: the information of the IC1 and IC2 updated. Matrix.  The first row contains the IC1 of each point. The second column contains the IC2 of each point. IC1 and IC2 are the closets and second closest cluster centers.
#'   \item  grouping: the information of the groups updated. List. Each component of the list contains a vector with the points that belong to that group. More specifically, the list component i has a vector with the numbers of the row of the matrix \code{data} where the points belonging to group i are.
#'   \item Live_set: Vector. Contains the groups that have been modified during the Step 6.
#'  \item index: number. The information of \code{index} updated.
#'    }
#'
#'
#'
#'
#' @examples
#'
#'set.seed(231)
#'data12=gtools::rdirichlet(10,c(1,1,4,4,20,20))
#'data1=t(data12)
#'k=3
#'seed=5
#'
#'distance<- function(vect1, vect2){
#'  sqrt(sum((vect1-vect2)^2))
#'}
#'
#'centers_function<-function(data, grouping){
#'  center=matrix(0,length(grouping), dim(data)[2])
#'  for (i in 1:(length(grouping))){
#'
#'    if(length(grouping[[i]])==1){
#'      center[i,]=data[grouping[[i]],]
#'    }else{
#'      center[i,]=apply(data[grouping[[i]],],2,mean)
#'    }
#'  }
#'  return(center)
#'}
#'
#'if(!is.null(seed)){
#'  set.seed(seed)
#'}
#'centers <- data1[sample(1:nrow(data1), k), ]
#'
#'#We calculate the distance between each row of the data matrix and the centers
#'Dist_e_cent=matrix(0,dim(data1)[1],dim(centers)[1])
#'for (i in 1:(dim(data1)[1])){
#'  for (j in 1:(dim(centers)[1])){
#'    Dist_e_cent[i,j]=distance(data1[i,],centers[j,])
#'  }
#'}
#'
#'#We obtain the IC1 and IC2 for each taxa
#'Ic12_change=Dist_IC1_IC2(Dist_e_cent)
#'Group=Ic12_change[,1]
#'grouping<-list()
#'for(i in 1:(max(Group))){
#'  grouping[[i]]=which(Group==i)
#'}
#'
#'
#'#Update the clusters centers.
#'centers=centers_function(data1, grouping)
#'
#'Ic12=cbind(c(1,1,3,3,2,2),c(1,2,1,2,3,3))
#'
#'P1=Step6(data1, centers, grouping, distance, centers_function, Ic12_change,Ic12, 0)
#'
#'
#'@references  Hartigan, J. A., & Wong, M. A. (1979). Algorithm AS 136: A k-means clustering algorithm. Journal of the royal statistical society. series c (applied statistics), 28(1), 100-108.
#'
#' @export


Step6<-function(data, centers, grouping, distance, centers_function, Ic12_change,Ic12, index){




  LIVE_SET_original=c()
  count=0
  ii=1

  while (count<=dim(data)[1]){

    count=count+1
    if(any(Ic12_change[ii,]!=Ic12[ii,])){#If IC1 and IC2 have not changed in step 4, we do not take into account the point ii.
      IC1=Ic12_change[ii,1]
      IC2=Ic12_change[ii,2]

      R1=length(grouping[[IC1]])*distance(data[ii,],centers[IC1,])^2/(length(grouping[[IC1]])-1)
      R2=length(grouping[[IC2]])*distance(data[ii,],centers[IC2,])^2/(length(grouping[[IC2]])+1)

            if(R1>=R2 && length(grouping[[IC1]])!=1){#ii changes group if R1>=R2 and ii is not the only point in the group
              count=0
              index=0
              old=IC1#old group
              new=IC2#new group
              Ic12[ii,2]=Ic12_change[ii,2]
              Ic12[ii,1]=Ic12_change[ii,1]
              Ic12_change[ii,2]=old
              Ic12_change[ii,1]=new
              Antiguo_updated=grouping[[old]][grouping[[old]]!=ii]#We removed item ii from the group it was in.
              grouping[[old]]=Antiguo_updated
              grouping[[new]]=add_unique_numbers2(grouping[[new]], ii) #We put the point in the new group
              centers=centers_function(data, grouping)#The group center is updated
              LIVE_SET_original=add_unique_numbers(LIVE_SET_original,old,new )
            }

        }



    if (ii<dim(data)[1]) {ii=ii+1}else{ii=1}


    if(count==dim(data)[1]){Step6_output=list(centers, Ic12_change, grouping, LIVE_SET_original,index)
    names(Step6_output)<-c("centers", "IC1andIC2", "grouping", "Live_set", "index")
    return(Step6_output)}

  }
}
