% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/netRNA.R
\name{netRNA}
\alias{netRNA}
\title{netRNA:Network meta-analysis for gene expression data}
\usage{
netRNA(TE, seTE, treat1, treat2, studlab)
}
\arguments{
\item{TE}{A list containing log fold changes from two individual studies.
Index names of the list should be the gene names; otherwise,
each value of the 'name' column in the output dataframe will correspond to
the position in the list, rather than gene identifiers.}

\item{seTE}{A list containing standard errors of log fold changes from two individual studies.}

\item{treat1}{A vector with Label/Number for first treatment.}

\item{treat2}{A vector with Label/Number for second treatment.}

\item{studlab}{A vector containing study labels}
}
\value{
A list containing the p values for each gene, the fold changes,
the upper and lower bounds for the 95\% CI of the log fold changes, and a summary dataframe with results for each gene.
}
\description{
This function conducts network meta-analysis using gene expression data to make indirect comparisons between different groups. It computes the p values for each gene and the fold changes, and provides a dataframe  containing these results.
}
\details{
The function supports a simple network with three nodes, where one node represents
a control group and the two other nodes represent treatment (or diseased) groups.
While the user provides fold changes and their standard errors of each treatment versus control as input,
the function calculates the fold changes for the indirect comparison between the two treatments.
}
\examples{

\dontrun{
#'#######################
### Data generation ###
#######################
n = 100 ### Sample size per group
G = 100 ### Number of genes

### Basic expression, fold change, batch effects and error
alpha.1 = rnorm(G, 0, 1)
alpha.2 = rnorm(G, 0, 1)
beta.1 = rnorm(G, 0, 1)
beta.2 = rnorm(G, 0, 1)
gamma.1 = rnorm(G, 0, 1)
gamma.2 = rnorm(G, 2, 1)
delta.1 = sqrt(invgamma::rinvgamma(G, 1, 1))
delta.2 = sqrt(invgamma::rinvgamma(G, 1, 2))
sigma.g = rep(1, G)

# Generate gene names
gene_names <- paste("Gene", 1:G, sep = "")

### Data matrices of control and treatment (disease) groups
C.1 = matrix(NA, G, n)
C.2 = matrix(NA, G, n)
T.1 = matrix(NA, G, n)
T.2 = matrix(NA, G, n)

for (j in 1:n) {
 C.1[,j] = alpha.1 + (0 * beta.1) + gamma.1 + (delta.1 * rnorm(1, 0, sigma.g))
 C.2[,j] = alpha.1 + (0 * beta.2) + gamma.2 + (delta.2 * rnorm(1, 0, sigma.g))
 T.1[,j] = alpha.2 + (1 * beta.1) + gamma.1 + (delta.1 * rnorm(1, 0, sigma.g))
 T.2[,j] = alpha.2 + (1 * beta.2) + gamma.2 + (delta.2 * rnorm(1, 0, sigma.g))
}

study1 = cbind(C.1, T.1)
study2 = cbind(C.2, T.2)

# Assign gene names to row names
#rownames(study1) <- gene_names
#rownames(study2) <- gene_names
#############################
### Differential Analysis ###
#############################

if(check_limma()){
### study1: treatment A versus control
group = gl(2, n)
M = model.matrix(~ group)
fit = limma::lmFit(study1, M)
fit = limma::eBayes(fit)
p.S1 = fit$p.value[,2]
fc.S1 = fit$coefficients[,2]
fce.S1 = sqrt(fit$s2.post) * sqrt(fit$cov.coefficients[2,2])

### study2: treatment B versus control
group = gl(2, n)
M = model.matrix(~ group)
fit = limma::lmFit(study2, M)
fit = limma::eBayes(fit)
p.S2 = fit$p.value[,2]
fc.S2 = fit$coefficients[,2]
fce.S2 = sqrt(fit$s2.post) * sqrt(fit$cov.coefficients[2,2])



#############################
### Network meta-analysis ###
#############################
p.net = rep(NA, G)
fc.net = rep(NA, G)
treat1 = c("uninfected", "uninfected")
treat2 = c("ZIKA", "HSV1")
studlab = c("experiment1", "experiment2")
fc.true = beta.2 - beta.1

TEs <- list(fc.S1, fc.S2)
seTEs <- list(fce.S1, fce.S2)
}

# Example usage:
test <- netRNA(TE = TEs, seTE = seTEs, treat1 = treat1, treat2 = treat2, studlab = studlab)
}
}
\references{
Winter, C., Kosch, R., Ludlow, M. et al. Network meta-analysis correlates with analysis of merged independent transcriptome expression data.
\emph{BMC Bioinformatics} \strong{20}, 144 (2019). \doi{https://doi.org/10.1186/s12859-019-2705-9}

Rücker G. (2012). Network meta-analysis, electrical networks and graph theory.
\emph{Research synthesis methods}, \strong{3(4)}, 312–324. \doi{https://doi.org/10.1002/jrsm.1058}
}
\seealso{
For more information, please refer to the package's documentation and the tutorial: \url{https://software.klausjung-lab.de/}.
}
\author{
Klaus Jung, Sergej Ruff
}
