% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dist_spnorm.R
\name{spnorm}
\alias{spnorm}
\alias{dspnorm}
\alias{rspnorm}
\alias{mle.spnorm}
\title{Spherical Normal Distribution}
\usage{
dspnorm(data, mu, lambda, log = FALSE)

rspnorm(n, mu, lambda)

mle.spnorm(data, method = c("Newton", "Halley", "Optimize", "DE"), ...)
}
\arguments{
\item{data}{data vectors in form of either an \eqn{(n\times p)} matrix or a length-\eqn{n} list.  See \code{\link{wrap.sphere}} for descriptions on supported input types.}

\item{mu}{a length-\eqn{p} unit-norm vector of location.}

\item{lambda}{a concentration parameter that is positive.}

\item{log}{a logical; \code{TRUE} to return log-density, \code{FALSE} for densities without logarithm applied.}

\item{n}{the number of samples to be generated.}

\item{method}{an algorithm name for concentration parameter estimation. It should be one of \code{"Newton"},\code{"Halley"},\code{"Optimize"}, and \code{"DE"} (case sensitive).}

\item{...}{extra parameters for computations, including\describe{
\item{maxiter}{maximum number of iterations to be run (default:50).}
\item{eps}{tolerance level for stopping criterion (default: 1e-5).}
}}
}
\value{
\code{dspnorm} gives a vector of evaluated densities given samples. \code{rspnorm} generates 
unit-norm vectors in \eqn{\mathbf{R}^p} wrapped in a list. \code{mle.spnorm} computes MLEs and returns a list 
containing estimates of location (\code{mu}) and concentration (\code{lambda}) parameters.
}
\description{
We provide tools for an isotropic spherical normal (SN) distributions on 
a \eqn{(p-1)}-sphere in \eqn{\mathbf{R}^p} for sampling, density evaluation, and maximum likelihood estimation 
of the parameters where the density is defined as
\deqn{f_{SN}(x; \mu, \lambda) = \frac{1}{Z(\lambda)} \exp \left( -\frac{\lambda}{2} d^2(x,\mu) \right)}
for location and concentration parameters \eqn{\mu} and \eqn{\lambda} respectively and the normalizing constant \eqn{Z(\lambda)}.
}
\examples{
\donttest{
# -------------------------------------------------------------------
#          Example with Spherical Normal Distribution
#
# Given a fixed set of parameters, generate samples and acquire MLEs.
# Especially, we will see the evolution of estimation accuracy.
# -------------------------------------------------------------------
## DEFAULT PARAMETERS
true.mu  = c(1,0,0,0,0)
true.lbd = 5

## GENERATE DATA N=1000
big.data = rspnorm(1000, true.mu, true.lbd)

## ITERATE FROM 50 TO 1000 by 10
idseq = seq(from=50, to=1000, by=10)
nseq  = length(idseq)

hist.mu  = rep(0, nseq)
hist.lbd = rep(0, nseq)

for (i in 1:nseq){
  small.data = big.data[1:idseq[i]]          # data subsetting
  small.MLE  = mle.spnorm(small.data) # compute MLE
  
  hist.mu[i]  = acos(sum(small.MLE$mu*true.mu)) # difference in mu
  hist.lbd[i] = small.MLE$lambda
}

## VISUALIZE
opar <- par(no.readonly=TRUE)
par(mfrow=c(1,2))
plot(idseq, hist.mu,  "b", pch=19, cex=0.5, main="difference in location")
plot(idseq, hist.lbd, "b", pch=19, cex=0.5, main="concentration param")
abline(h=true.lbd, lwd=2, col="red")
par(opar)
}

}
\references{
\insertRef{hauberg_2018_DirectionalStatisticsSpherical}{Riemann}

\insertRef{you_2022_ParameterEstimationModelbased}{Riemann}
}
\concept{distribution}
