% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Simulation.R
\name{GHBMP}
\alias{GHBMP}
\title{Simulation of Gaussian Haar-based multifractional processes}
\usage{
GHBMP(t, H, J = 15, num.cores = availableCores(omit = 1))
}
\arguments{
\item{t}{Time point or time sequence on the interval \eqn{[0,1]}.}

\item{H}{Hurst function which depends on \code{t} \eqn{(H(t))}. See Examples for usage.}

\item{J}{Positive integer. \code{J} is recommended to be greater than \eqn{\log_2(length(t))}. For large \code{J} values could
be rather time consuming. Default is set to 15.}

\item{num.cores}{Number of cores to set up the clusters for parallel computing.}
}
\value{
A data frame of class \code{"mp"} where the first column is time moments \code{t} and second column is simulated values of \eqn{X(t)}.
}
\description{
This function simulates a realisation of a Gaussian Haar-based multifractional process at any
time point or time sequence on the interval \eqn{[0,1]}.
}
\details{
The following formula defined in Ayache, A., Olenko, A. & Samarakoon, N. (2025) was used in simulating Gaussian Haar-based multifractional process.

\deqn{X(t) := \sum_{j=0}^{+\infty}  \sum_{k=0}^{2^{j}-1}\left(\int_{0}^{1} (t-s)_{+}^{H_{j}(k/{2^j})-{1}/{2}} h_{j,k}(s)ds \right)\varepsilon_{j,k},}

where

\deqn{  \int_{0}^{1} (t-s)_{+}^{H_{j,k}-\frac{1}{2}} h_{j,k} (s) ds = 2^{-j H_{j,k}} h^{[H_{j,k}]} (2^jt-k)}

with \eqn{h^{[\lambda]} (x) =  \int_{\mathbb{R}} (x-s)_{+}^{\lambda-\frac{1}{2}} h(s) ds}.
\eqn{h} is the Haar mother wavelet, \eqn{j} and \eqn{k} are positive integers, \eqn{t} is time, \eqn{H} is the Hurst function and
\eqn{\varepsilon_{j,k}} is a sequence of independent \eqn{\mathcal{N}(0,1)} Gaussian random variables.
For simulations, the truncated version of this formula with first summation up to J is used.
}
\note{
See Examples for the usage of constant, time-varying, piecewise or step Hurst functions.
}
\examples{
\donttest{
#Constant Hurst function
t <- seq(0, 1, by = (1/2)^10)
H <- function(t) {return(0.4 + 0*t)}
GHBMP(t, H)

#Linear Hurst function
t <- seq(0, 1, by = (1/2)^10)
H <- function(t) {return(0.2 + 0.45*t)}
GHBMP(t, H)

#Oscillating Hurst function
t <- seq(0, 1, by = (1/2)^10)
H <- function(t) {return(0.5 - 0.4 * sin(6 * 3.14 * t))}
GHBMP(t, H)

#Piecewise Hurst function
t <- seq(0, 1, by = (1/2)^10)
H <- function(x) {
ifelse(x >= 0 & x <= 0.8, 0.375 * x + 0.2,
      ifelse(x > 0.8 & x <= 1,-1.5 * x + 1.7, NA))
}
GHBMP(t, H)
}
}
\references{
Ayache, A., Olenko, A. and Samarakoon, N. (2025).
On Construction, Properties and Simulation of Haar-Based
Multifractional Processes. \doi{doi:10.48550/arXiv.2503.07286}. (submitted).
}
\seealso{
\code{\link{Hurst}}, \code{\link{plot.mp}}, \code{\link{Bm}}, \code{\link{FBm}},
\code{\link{FGn}}, \code{\link{Bbridge}}, \code{\link{FBbridge}}
}
