#' Cross-validation for selecting the tuning parameter of the SAFE model
#'
#' Performs "electoral college" cross-validation for the \code{\link{safe}} function. Unlike a standard
#' cross-validation approach, this method repeats the random partitioning into folds multiple times
#' (controlled by \code{rep}), then selects the tuning parameter \code{lambda} via a majority vote
#' across these repeated cross-validation runs.
#' This function is largely similar [glmnet::cv.glmnet()].
#'
#' @import Matrix
#' @param x A numeric matrix of dimensions \eqn{n \times p}, where \eqn{n} is the number of observations
#'   and \eqn{p} is the number of predictors.
#' @param y A numeric vector of length \eqn{n}, representing the loss values (severity). These are assumed
#'   to follow a Gamma distribution with shape parameter \code{alpha} and scale parameter
#'   \eqn{\theta = \exp(x^\top \gamma) / \alpha}.
#' @param k A numeric vector of length \eqn{n}, representing the number of claims (frequency), assumed to follow
#'   a Poisson distribution with mean \eqn{\mu = x^\top \beta}.
#' @param lambda A user-supplied numeric vector of tuning parameters. The function will compute the solution
#'   for each value in \code{lambda}.
#' @param ind_p A user-provided index or indicator specifying which predictors should share the same sign
#'   across frequency and severity.
#' @param nfolds The number of folds in cross-validation. Default is 5.
#' @param rep The number of repeated cross-validation cycles (default is 24). In each cycle, observations
#'   are randomly assigned to \code{nfolds} folds, cross-validation is performed for all \code{lambda}
#'   values, and a "vote" is cast for the \code{lambda} with the lowest error in that cycle.
#' @param ... Additional arguments passed to \code{\link{safe}}.
#'
#' @details
#' The function computes the average cross-validation error and reports the best \code{lambda} that achieves the 
#' smallest cross-validation error.
#'
#' @return An object of class \code{"safe"}, which is a list containing:
#' \item{\code{lambda.min}}{The tuning parameter \code{lambda} value that won the most votes (i.e.,
#'      achieved the minimum cross-validation error most often).}
#' \item{\code{lambda}}{The full sequence of candidate \code{lambda} values provided to the function.}
#' \item{\code{ncvmat}}{A numeric matrix of dimension \code{rep} * the length of \code{lambda},
#'       containing the cross-validation errors for each \code{lambda} across all \code{rep} runs.}
#' @examples
#' set.seed(1)
#' n <- 100
#' p <- 5
#' x <- matrix(rnorm(n * p), nrow = n, ncol = p)
#' beta_true <- rep(0.1, 5)
#' gamma_true <- c(rep(1, 3), -1, -1)
#' mu <- x %*% beta_true
#' k <- rpois(n, lambda = exp(mu))
#' alpha_val <- 1
#' theta <- exp(x %*% gamma_true) / alpha_val
#' y <- rgamma(n, shape = alpha_val, scale = theta)
#' lambda_seq <- 10^seq(2, -8, length.out = 5)
#' fit <- eccv.safe(x, y, k, lambda=lambda_seq, ind_p = c(1, 1, 1, 0, 0))
#' @keywords cross validation
#' @export

eccv.safe <- function(x, y, k, lambda, ind_p, rep=24, nfolds=5L, ...) {
  x <- as.matrix(x)
  y <- drop(y)
  N <- rep
  x.row <- as.integer(NROW(x))
  if (length(y) != x.row) 
    stop("x and y have different number of observations.")
  if (nfolds < 3) 
    stop("nfolds must be bigger than 3; nfolds=5 recommended.")
  foldidMat <- replicate(N, sample(rep(seq(nfolds), length=x.row)))
  # foldid <- sample(rep(seq(nfolds), length=x.row))
  # outlist <- as.list(seq(nfolds))
  len <- length(lambda)
  np <- dim(x)
  nobs <- as.integer(np[1])
  nvars <- as.integer(np[2])

  fit1 <- ini.safe(x, y, k)
  int_gam <- fit1$gamma
  int_beta <- fit1$beta
  votes <- rep(0, len)
  ncvmat <- matrix(NA, N, len)
  for (r in 1:N){
    foldid <- foldidMat[, r]
    outlist <- as.list(seq(nfolds))
    for (j in seq(nfolds)) {
      which <- foldid == j
      outlist[[j]] <- safe(x=x[!which, , drop=FALSE], y=y[!which], 
        k = k[!which], lambda = lambda,
        ind_p = ind_p, int_gam = int_gam, int_beta = int_beta)
    }
    cvstuff <- cvpath.safe(outlist, x, y, k, lambda, foldid, x.row)
    cvm <- cvstuff$cvm
    loc <- which(cvm == min(cvm), arr.ind = TRUE)[1]
    votes[loc] <- votes[loc] + 1
    ncvmat[r, ] <- cvm
  }
  # print(votes)
  vote_l <- which(votes == max(votes), arr.ind = TRUE)[1]
  # print(vote_l)
  outlist <- list(lambda.min = lambda[vote_l], ncvmat = ncvmat, lambda = lambda)
  class(outlist) <- c("cv.safe")
  outlist
}

cvpath.safe <- function(outlist, x, y, k, lambda, foldid, x.row, ...) {
  nfolds <- max(foldid)
  len <- length(lambda)
  predmat <- matrix(NA, x.row, len)
  nlams <- double(nfolds)
  for (i in seq(nfolds)) {
    whichfold <- foldid == i
    fitobj <- outlist[[i]]
    preds <- predict.safe(fitobj, x[whichfold, , drop = FALSE])
    # nlami <- dim(fitobj$beta)[4]
    predmat[whichfold, seq(len)] <- preds
    # nlams[i] <- nlami
  }
  cvraw <- (y - predmat)^2
  N <- length(y) - apply(is.na(predmat), 2, sum)
  cvm <- apply(cvraw, 2, function(x) mean(x, trim = 0.1, na.rm=TRUE))
  # cat("cvm", cvm, "\n")
  scaled <- scale(cvraw, cvm, FALSE)^2
  cvsd <- sqrt(colMeans(scaled, na.rm = TRUE) / (N - 1))
  # cvm.min <- min(cvm)
  # ratio.loc <- which.min(cvm)
  # # print(ratio.loc)
  # cvsd.min <- cvsd[ratio.loc]
  out <- list(cvm = cvm, cvsd = cvsd)
  out
}