#' Singular Spectrum Analysis Decomposition
#'
#' @importFrom Rssa ssa reconstruct wcor
#' @importFrom utils head
#' @importFrom stats ts
#' @export
#'
SSAdecomp <- function(data, L = 12, corr_thr = 0.97, horizon = 12, verbose = FALSE) {
  data <- ts(data)
  data <- as.vector(data)
  data_trn <- head(data, round(length(data) - horizon))

  # Step 1: SSA decomposition
  s1 <- Rssa::ssa(data_trn, L)
  trend_component <- Rssa::reconstruct(s1, groups = list(1))
  res.trend <- data_trn - trend_component$F1   # safer than residuals()

  # Step 2: Residual SSA
  L1 <- L * floor((length(data_trn) / 2) / L)
  s2 <- Rssa::ssa(res.trend, L1)

  wtcor <- Rssa::wcor(s2)
  num_components <- nrow(wtcor)
  high_correlation_pairs <- list()

  if (verbose) {
    message("Component pairs with correlation > ", corr_thr, ":")
  }

  for (i in 1:(num_components - 1)) {
    for (j in (i + 1):num_components) {
      corr_ij <- wtcor[i, j]
      if (corr_ij > corr_thr) {
        high_correlation_pairs <- append(high_correlation_pairs, list(c(i, j, corr_ij)))
        if (verbose) {
          message("Components ", i, " and ", j, " with correlation: ", signif(corr_ij, 4))
        }
      }
    }
  }

  # Step 3: Reconstruct correlated components
  recons_comp <- list()
  for (pair in high_correlation_pairs) {
    group <- c(pair[1], pair[2])
    reconstructed <- Rssa::reconstruct(s2, groups = list(group))
    recons_comp <- append(recons_comp, list(reconstructed))
  }

  # Final structured result
  final_result <- list(
    trend_component = trend_component,
    recon_components = recons_comp,
    corr_pairs = high_correlation_pairs
  )

  class(final_result) <- "SSAdecomp"
  return(final_result)
}

# Print method for clean console output
#' @export
print.SSAdecomp <- function(x, ...) {
  cat("SSA decomposition object\n")
  cat("Trend component length:", length(x$trend_component$F1), "\n")
  cat("Number of correlated component groups:", length(x$recon_components), "\n")
  if (length(x$corr_pairs) > 0) {
    cat("First correlated pair: ", x$corr_pairs[[1]][1], "-", x$corr_pairs[[1]][2],
        " (corr=", signif(x$corr_pairs[[1]][3], 4), ")\n", sep = "")
  }
  invisible(x)
}
