#' Sample size calculator in each stratum under Stratified Random Sampling
#'
#' @description The SscStr function calculates sample size in each stratum under different allocation methods in Stratified Random Sampling without replacement design which is required for estimation of population mean and proportion based upon the availability of prior information on sizes of the strata, standard deviations of the strata and costs of drawing a sampling unit in the strata.
#'
#' @param Allocation The method of allocation of sample sizes in the strata. It can be Equal, Proportional, Neyman and Optimum. For Equal and Proportional allocation, n and Strata_Size are to be provided whereas for Neyman allocation, n, Strata_Size and Strata_Standard_deviation are required. For, Optimum allocation, n, Strata_Size, Strata_Standard_deviation and Strata_Cost are to be provided.
#' @param n Total sample size that is to be allocated.
#' @param Strata_Size Sizes of the strata in the population.
#' @param Strata_Standard_deviation Standard deviations of the strata in the population.
#' @param Strata_Cost Costs of drawing a sampling unit in the strata.
#'
#' @return A list with the following components:
#' \item{Strata wise allocated sample size}{Allocated sample size in each strata.}
#'
#' @details This function returns the allocated sample size in each stratum under Stratified Random Sampling without replacement design which is required for estimation of population mean and proportion.
#'
#' @references
#' Cochran, W. G. (1977). \emph{Sampling Techniques, 3rd Edition}. New York: John Wiley & Sons, Inc.
#'
#' Singh, D. and Chaudhary, F.S. (1986). \emph{Theory and Analysis of Sample Survey Designs}. New York: John Wiley & Sons, Inc.
#'
#' Sukhatme, P.V., Sukhatme, B.V., Sukhatme, S. and Asok, C. (1984). \emph{Sampling Theory of Surveys with Applications}. Iowa State University Press, Ames and Indian Society of Agricultural Statistics, New Delhi.
#'
#' @examples
#' # Calculates sample size in each stratum under Equal allocation
#' SscStr("Equal", 100, c(500,300,200), NA, NA)
#' # Calculates sample size in each stratum under Proportional allocation
#' SscStr("Proportional", 100, c(500,300,200), NA, NA)
#' # Calculates sample size in each stratum under Neyman's allocation
#' SscStr("Neyman", 100, c(500,300,200), c(10,20,30), NA)
#' # Calculates sample size in each stratum under Optimum allocation
#' SscStr("Optimum", 100, c(500,300,200), c(10,20,30), c(5,10,15))
#' @export
SscStr=function(Allocation, n, Strata_Size, Strata_Standard_deviation, Strata_Cost)
{
  # Calculate sample size for each stratum under equal allocation
  if(Allocation=="Equal")
  {
    size=rep(round((n/length(Strata_Size)),0),length(Strata_Size))
    size[length(Strata_Size)]=n-sum(size[1:length(Strata_Size)-1])
    output <- list("Strata wise allocated sample size"=c(size[1:length(Strata_Size)-1],size[length(Strata_Size)]))
    return(output)
  }
  # Calculate sample size for each stratum under proportional allocation
  if(Allocation=="Proportional")
  {
    size=round((n/sum(Strata_Size))* Strata_Size,0)
    size[length(Strata_Size)]=n-sum(size[1:length(Strata_Size)-1])
    output <- list("Strata wise allocated sample size"=c(size[1:length(Strata_Size)-1],size[length(Strata_Size)]))
    return(output)
  }
  # Calculate sample size for each stratum under Neyman’s allocation
  if(Allocation=="Neyman")
  {
    size=round((n* Strata_Size*Strata_Standard_deviation)/sum(Strata_Size*Strata_Standard_deviation),0)
    size[length(Strata_Size)]=n-sum(size[1:length(Strata_Size)-1])
    output <- list("Strata wise allocated sample size"=c(size[1:length(Strata_Size)-1],size[length(Strata_Size)]))
    return(output)
  }
  # Calculate sample size for each stratum under optimum allocation
  else if(Allocation=="Optimum")
  {
    size=round(((n* Strata_Size*Strata_Standard_deviation)/sqrt(Strata_Cost))/sum((Strata_Size*Strata_Standard_deviation)/sqrt(Strata_Cost)),0)
    size[length(Strata_Size)]=n-sum(size[1:length(Strata_Size)-1])
    output <- list("Strata wise allocated sample size"=c(size[1:length(Strata_Size)-1],size[length(Strata_Size)]))
    return(output)
  }
}
