% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sim_kid.R
\name{sim_kid}
\alias{sim_kid}
\title{Create body size metrics for virtual subjects using anthropometric growth
chart data and a distribution of age}
\usage{
sim_kid(
  num = 1,
  agedistr = "unif",
  agemean = NULL,
  agesd = NULL,
  agemin = NULL,
  agemax = NULL,
  prob_female = 0.5,
  age0isbirth = FALSE,
  age0to2yr_growthchart = "CDC",
  age2to20yr_correlate_htwt = TRUE,
  htwt_percentile_min = NULL,
  htwt_percentile_max = NULL,
  masterseed = NULL
)
}
\arguments{
\item{num}{A positive integer that specifies the number of subjects to
simulate. Defaults to a single subject. For \code{agedistr = "nperage"} the
number of subjects per growth chart age and sex bin.}

\item{agedistr}{A string that specifies the distribution used to create
virtual subject age.
\itemize{
\item \code{unif} (the default): A uniform distribution of age with a range from
\code{agemin} to \code{agemax}.
\item \code{norm}: A truncated normal distribution of age with a mean of \code{agemean},
a standard deviation of \code{agesd}, and a range from \code{agemin} to \code{agemax}.
\item \code{nperage}: An equal number of subjects per growth chart age and sex bin.
}}

\item{agemean}{A positive numeric greater than or equal to \code{agemin} and less
than or equal to \code{agemax} that specifies the mean age when
\code{agedistr = "norm"} is specified.
\itemize{
\item Only used for \code{agedistr = "norm"}.
\item Units of postnatal age in months for \code{age0to2yr_growthchart = "CDC"} or
\code{age0to2yr_growthchart = "WHO"}.
\item Units of gestational age in weeks for \code{age0to2yr_growthchart = "FENTON"}.
}}

\item{agesd}{A numeric greater than or equal to \code{0} that specifies the
standard deviation of age when \code{agedistr = "norm"} is specified.
\itemize{
\item Only used for \code{agedistr = "norm"}.
\item Units of postnatal age in months for \code{age0to2yr_growthchart = "CDC"} or
\code{age0to2yr_growthchart = "WHO"}.
\item Units of gestational age in weeks for \code{age0to2yr_growthchart = "FENTON"}.
}}

\item{agemin}{A numeric that specifies the lower range of age. Defaults to
the maximum allowable range if missing.
\itemize{
\item Must be greater than or equal to \code{0} months of postnatal age for
\code{age0to2yr_growthchart = "CDC"} or \code{age0to2yr_growthchart = "WHO"}.
\item Must be greater than or equal to \code{22} weeks of gestational age for
\code{age0to2yr_growthchart = "FENTON"}.
\item Must be less than or equal to \code{agemax}.
}}

\item{agemax}{A numeric that specifies the upper range of age. Defaults to
the maximum allowable range if missing.
\itemize{
\item Must be less than \code{240} months of postnatal age for
\code{age0to2yr_growthchart = "CDC"} or \code{age0to2yr_growthchart = "WHO"}.
\item Must be less than \code{41} weeks of gestational age for
\code{age0to2yr_growthchart = "FENTON"}.
\item Must be greater than or equal to \code{agemin}.
}}

\item{prob_female}{A numeric value with an inclusive range of \code{0} to \code{1}
that specifies the probability that a given virtual subject is female.
Defaults to \code{0.5}.}

\item{age0isbirth}{A logical that specifies whether age equal to zero
denotes birth.
\itemize{
\item \code{TRUE}: Age of \code{0} is birth.
\item \code{FALSE} (the default): Age of \code{0} is ages from birth to less than one
month.
\item Not applicable nor used for \code{age0to2yr_growthchart = "FENTON"}, for
which postnatal age is always zero.
}}

\item{age0to2yr_growthchart}{A string that specifies which anthropometric
growth charts are used for ages less than or equal to 2 years old.
\itemize{
\item \code{"CDC"} (the default): United States Centers for Disease Control and
Prevention growth charts are used.
\item \code{"WHO"}: World Health Organization growth charts are used.
\item \code{"FENTON"}: Fenton growth charts for preterm newborns are used. This
option is only available when simulating virtual subjects at birth
(postnatal age = 0).
}}

\item{age2to20yr_correlate_htwt}{A logical that specifies whether
correlations, by sex and year of age, are implemented between simulated
height and simulated weight for ages greater than or equal to 2 years old.
\itemize{
\item \code{TRUE} (the default): Correlations are implemented between simulated
height and simulated weight according to an identical
internal-systems-data version of
\link{cdc_ages2to20yr_correlations_by_sex_htcm_wtkg_summarized} located within
the \code{data} folder.
\item \code{FALSE}: Height and weight are simulated independently without any
correlation(s). Note that this will likely result in unrealistic virtual
subjects.
}}

\item{htwt_percentile_min}{A numeric value that specifies the minimum
allowed percentile of simulated height and weight, expressed as a decimal.
\itemize{
\item Must be greater than or equal to \code{0.001}.
\item Must be less than \code{htwt_percentile_max} when
\code{age2to20yr_correlate_htwt = TRUE}. Must be less than or equal to
\code{htwt_percentile_max} when \code{age2to20yr_correlate_htwt = FALSE}.
\item Defaults to \code{0.001} when \code{age0to2yr_growthchart = "CDC"} or
\code{age0to2yr_growthchart = "WHO"}.
\item Defaults to \code{0.01} when \code{age0to2yr_growthchart = "FENTON"} to avoid
non-viable birth weights.
}}

\item{htwt_percentile_max}{A numeric value that specifies the maximum
allowed percentile of simulated height and weight, expressed as a decimal.
\itemize{
\item Must be less than or equal to \code{0.999}.
\item Must be greater than \code{htwt_percentile_min} when
\code{age2to20yr_correlate_htwt = TRUE}. Must be greater than or equal to
\code{htwt_percentile_min} when \code{age2to20yr_correlate_htwt = FALSE}..
\item Defaults to \code{0.999} when \code{age0to2yr_growthchart = "CDC"} or
\code{age0to2yr_growthchart = "WHO"}.
\item Defaults to \code{0.99} when \code{age0to2yr_growthchart = "FENTON"} to avoid
non-viable birth weights.
}}

\item{masterseed}{An integer ranging from \code{1} to \code{.Machine$integer.max} that
sets an overall seed for the simulation to ensure reproducibility of the
results. Defaults to no seed.}
}
\value{
A data frame with the number of rows equal to \code{num}
(except for \code{agedistr = "nperage"}) and columns of:
\itemize{
\item \code{ID}: An integer ranging from \code{1} to \code{num} that serves as a virtual
subject identifier.
\item \code{SEXF}: An integer of value \code{0} for male or \code{1} for female.
\item \code{AGEMO}: Postnatal age in months.
\item \code{AGE}: Postnatal age in years.
\item \code{GAWK}: Gestational age in weeks.
\item \code{WTKG}: Body weight in kilograms, rounded to 2 decimal places.
\item \code{HTCM}: Body height in centimeters, rounded to the nearest centimeter.
\item \code{BMI}: Body mass index in kilograms per meter squared, rounded to 1
decimal place.
\item \code{BSA1}: Body surface area in meters squared, rounded to 2 decimal
places; calculated using the Mosteller equation.
\item \code{BSA2}: Body surface area in meters squared, rounded to 2 decimal
places; calculated using the Gehan and George equation.
\item \code{BSA3}: Body surface area in meters squared, rounded to 2 decimal
places; calculated using the DuBois equation.
\item \code{ZWTKG}: The z-score of weight-for-height for ages 0 to 2 years,
weight-for-age for ages greater than 2 years, and
weight-for-gestational-age for newborns when using Fenton growth charts.
\item \code{ZHTCM}: The z-score of height-for-age.
\item \code{PWTKG}: The percentile of weight corresponding to the respective
z-score.
\item \code{PHTCM}: The percentile of height corresponding to the respective
z-score.
\item \code{CHART}: The anthropometric growth chart used.
An error will be returned if the simulation fails.
}
}
\description{
Body size metrics (height, weight, BMI, and BSA) are created for a
population of virtual subjects.The body size metrics reflect the
anthropometric growth chart distribution(s) and correlations
(ex. height vs weight) according to virtual subject age and sex. The assumed
distribution of age (uniform or truncated normal) and probability that a
given subject is female are specified by the user. For ages greater than 2
years, CDC growth charts are used. For ages birth to 2 years, either CDC
(the default) or WHO growth charts can be used. Note that while CDC growth
charts are used to prevent a jump discontinuity at 2 years, WHO growth
charts are recommended for ages 0 to 2 years. For birth only (postnatal age
of zero), Fenton growth charts for preterm can be used according to a
distribution of gestational age.
Note: when using Fenton growth charts, only body weight will be simulated.
}
\details{
Equations and methods involved during the creation of virtual
subjects.
}
\section{Calculation of simulated body height and weight}{


The equation for simulated body height in cm (\code{HTCM}) or weight in kg
(\code{WTKG}) is:
if L (rounded to 6 decimal places) is equal to 0, then \verb{= M*exp(S*Z)};
otherwise \verb{= M*(1+L*S*Z)^(1/L))} (1).

Where L, M, and S are obtained, using the independent variables of sex
(\code{SEXF}) and age bucket (\code{AGEGRP}), from identical internal-systems-data
versions of the combined anthropometric growth chart datasets (\link{kid0} and
\link{htwt0} located within the \code{data} folder).
And where Z, the z-score respective to either the height or weight
distribution, is randomly sampled for each virtual subject.

(1) \url{https://www.cdc.gov/growthcharts/cdc-data-files.htm}
}

\section{Simulation of z-scores for variability in height and weight}{


For ages 0 to 2 years, correlations between height and weight are always
implemented. This is done by simulating height using length-for-age growth
charts (see \link{kid0} located within the \code{data} folder) and then simulating
weight using weight-for-height growth charts (see \link{htwt0} located within the
\code{data} folder).
For ages greater than 2 years, correlations between height and weight were
repeatedly optimized (see
\link{cdc_ages2to20yr_correlations_by_sex_htcm_wtkg_allreplicates} located within
the \code{data} folder) and then summarized to the mean (see
\link{cdc_ages2to20yr_correlations_by_sex_htcm_wtkg_summarized} located within
the \code{data} folder).
For ages greater than 2 years, the user can override the default behavior
that includes correlations (as per an identical internal-systems-data
version of \link{cdc_ages2to20yr_correlations_by_sex_htcm_wtkg_summarized})
between simulated height and weight using the \code{age2to20yr_correlate_htwt}
input.

For ages 0 to 2 years and for ages greater than 2 years when simulating
without correlations between height and weight:
The z-scores are obtained independently for height and weight and for each
virtual subject via random sampling from a truncated standard normal
distribution using \code{msm::rtnorm()}.

For ages greater than 2 years when simulating with correlations between
height and weight:
The z-scores are obtained simultaneously for height and weight and for each
virtual subject via random sampling from a truncated multivariate standard
normal distribution using \code{tmvtnorm::rtmvnorm()}.
}

\section{Calculation of body mass index}{


The equation for body mass index in kilograms per meter squared is
\code{BMI = WTKG/((HTCM/100)^2)}.
}

\section{Calculation of body surface area}{


The Mosteller equation (1) for body surface area in meters squared is
\code{BSA1 = sqrt(WTKG*HTCM/3600)}.

The Gehan and George equation (2) for body surface area in meters squared is
\code{BSA2 = 0.0235*(WTKG^0.51456)*(HTCM^0.42246)}.

The DuBois equation (3) for body surface area in meters squared is
\code{BSA3 = 0.007184*(WTKG^0.425)*(HTCM^0.725)}.

(1) Mosteller RD. Simplified calculation of body-surface area. N Engl J Med.
1987 Oct 22;317(17):1098. <doi: 10.1056/NEJM198710223171717.> PMID: 3657876.
(2) Gehan EA, George SL. Estimation of human body surface area from height
and weight. Cancer Chemother Rep. 1970 Aug;54(4):225-35. PMID: 5527019.
(3) Du Bois D, Du Bois EF. A formula to estimate the approximate surface
area if height and weight be known. 1916. Nutrition. 1989
Sep-Oct;5(5):303-11; discussion 312-3. PMID: 2520314.
}

\examples{
# Simulate 1 subject with an age randomly sampled from a uniform 
# distribution of ages ranging
#    from 0 to 20 years using CDC growth charts.
df_kids <- sim_kid()

# Simulate 10 female 3 year old subjects with a seed set for reproducibility.
df_kids <- sim_kid(
  num = 10,
  agedistr = "norm", agemean = 36, agesd = 0,
  prob_female = 1, masterseed = 513
)

# Simulate 10 subjects (approximately 50\% female) with ages ranging from 
# 1 year to 2 years
# according to a uniform distribution of age using WHO growth charts.
df_kids <- sim_kid(
  num = 10,
  agedistr = "unif", agemin = 12, agemax = 24,
  age0to2yr_growthchart = "WHO"
)

# Simulate 1 subject per age bin and per sex using CDC growth charts
df_kids <- sim_kid(agedistr = "nperage")
}
