#' @title Get permutation number
#' @description Given a permutation, this function returns its position in the Lehmer code order.
#' @param permutation A permutation, as a vector.
#' @return The position of the permutation according to the Lehmer code order, as an integer.
#' @details The Lehmer code makes use of the fact that there are \eqn{n!} permutations of a sequence of \eqn{n} numbers.
#' If a permutation \eqn{\sigma} is specified by the sequence \eqn{(\sigma_{i})_{i=1}^{n}}, its Lehmer code is the
#' sequence \eqn{L(\sigma)=(L(\sigma)_{i})_{i=1}^{n}}, where \eqn{L(\sigma)_i=|\{j>i:\sigma_j<\sigma_i\}|}.
#'
#' The position of permutation \eqn{\sigma} according to the Lehmer code order is
#' \deqn{L_{\sigma}=1 + \sum_{i=1}^{n} (n-i)! L(\sigma)_i}.
#' @examples
#' getpermutationnumber(c(1, 2, 5, 4, 3))
#' @seealso \link{getpermutation}
#' @export

getpermutationnumber <- function(permutation) { # Principio de la función

  ##################################
  ### Comprobación datos entrada ###
  ##################################

  if (sum(permutation <= 0) > 0 || sum(permutation %% 1 != 0) > 0) {
    stop("'permutation' must be a positive vector.")
  }

  if (sum(unique(permutation) != permutation) > 0) {
    stop("A number cannot appear in 'permutation' more than once.")
  }

  if (max(permutation) != length(permutation)) {
    stop("'permutation' might be missing at least one number.")
  }

  ################################
  ##### Creación de variables######
  ################################

  n <- length(permutation)

  ################################
  ###### Cuerpo de la función######
  ################################

  d <- c(permutation[1] - 1, rep(0, n - 1))

  for (ii in 2:n - 1)
  {
    cantos <- sum(permutation[ii] > permutation[1:ii - 1])
    d[ii] <- permutation[ii] - cantos - 1
  }

  interno <- rep(0, n)
  for (ii in 1:n)
  {
    interno[ii] <- factorial(n - ii)
  }

  num <- as.numeric(d %*% interno + 1)

  ################################
  ###### Salidas de la función#####
  ################################

  return(num)
} # Fin de la función
