AfromFam <- function(Fam, n, tol = 100*.Machine$double.eps) {
  # AfromFam Independent coalitions from those in family Fam
  # Fam = vector of independent coalitions Fam=[S1 ... Sp]
  # n = number of players
  # tol = tolerance level (by default, 100*eps)

  # There are nc=2^n-1 coalitions that can be formed with n players
  # For each coalition S=1:nC we check if S is independent from Fam
  # f(S)=1 S is independent
  # f(S)=0 S is a linear combination of coalitions in Fam
  # Therefore f is a (0,1)-vector of size 1x(2^n-1)
  # To simplify its storage we divide f into chunks of size 31 (close to the threshold for binary numbers)
  # So, we identify f with a 1xr vector, where r=floor(nC/52)+1.
  # Each coordinate of this compact version of f is
  # the binary number associated with the coordinates of the real vector f in the
  # corresponding chunck
  # AfromFam(c(1,2,3,5),4)
  # AfromFam(c(12,122,33,50,45),8)

  # Si no nos dan la tolerancia, fijar tol = 100 * .Machine$double.eps


  nC <- 2^n - 1 # Número de coaliciones
  nFam <- length(Fam) # Cardinal de la familia Fam
  ###############################
  # En primer lugar calculamos los vectores u^j
  # que me permiten obtener por recurrencia todas las coaliciones
  # Vectores u^j =
  #  0 de la posición 1 hasta la j-1;
  #  1 en la posición j;
  # -1 de la posición j+1 hasta la n

  # Construcción de los vectores u^j
  u <- matrix(0, n, n)
  for (i in 1:n) {
    u[i, n - i + 1] <- 1
    if (n - i + 2 <= n) {
      u[i, (n - i + 2):n] <- -1
    }
  }
  # Escribimos la (0-1)-matriz con coeficientes
  # correspondientes a las coaliciones en Fam.
  # Si nFam<n completamos con filas de zeros
  # Construcción de la matriz Fam en binario
  Famb <- Fam
  Fam <- matrix(0, nFam, n)
  for (ii in 1:nFam) {
    for (jj in 1:n) {
      if (bitwAnd(bitwShiftR(Famb[ii], jj - 1),1) != 0) { # Si el jugador jj está en la coalicion Famb(ii)
        Fam[ii, n - jj + 1] <- 1 # Pongo un uno en esa posición contando desde la derecha
      }
    }
  }

  # % Podíamos hacer, equivalentemente,
  # for ii=1:nFam
  # Fam(ii,:)=bitget(Famb(ii),n:-1:1);
  # end
  ######
  # Triangulamos Fam. SUT es una matriz cuadrada de orden nxn

  # Triangulación superior de la matriz Fam
  H <- triangularup(Fam, tol)$SUT # Esta función debe implementarse

  ######### Ahora, "arreglamos" la diagonal de la matriz H
  # (según la definicion del paper) y creamos el vector h
  #########
  # Ajustar la diagonal de H
  # print(H$T)

  h <- 0
  for (ii in 1:n) {
    if (H[ii, ii] == 0) {
      h <- bitwOr(h,bitwShiftL(1,ii - 1))
      H[ii, ii] <- 1
    }
  }
  #  Calculamos la solución (mu) de los sistemas de ecuaciones
  #                        xH=u^j para j=1:n

  # Resolución de sistemas lineales
  mu <- u %*% solve(H) # Alternativamente, mu=u*inv(H)
  #print(mu)

  # Según el artículo, si alpha es la solución de xH=a,
  # el (0-1)-vector a NO es una fila de A
  # sí y sólo si (alpha_i)(h_i)=0 para todo i=1,...,n
  # Ciertamente, como H es triangular superior y las primeras j coordenadas
  # de u^j son nulas, se tiene que las primeras j coordenadas de mu^j son
  # nulas.
  #
  # Recorremos todos los (0,1)-vectores (los generamos por recurrencia a partir de los u^j) y comprobamos si cada vector es o no una fila de A.
  # Recordemos que guardaremos esa información en forma compacta en un vector
  #f de tamaño 1xr.
  #
  # En cada paso, el valor k será:
  # o bien n+1 si alpha*h=0
  # o bien el menor índice i=1,...,n para el cual alpha_ih_i\not=0
  #
  ##########################

  bloque <- 31 #tamaño de cada bloque de información
  r <- floor(nC / bloque) + 1 # Number of chuncks
  f <- rep(0,r) #  f es en realidad una versión reducida del autentico vector f de longitud 2^n-1.

  # Empezamos por el primer vector en el orden lexicográfico
  # a=(0 0 ... 0 1)
  alpha <- mu[1,] # El primer vector alpha=mu^1=(0 0 ... 0 1)

  # Si alpha es la solución de xH=(0 0 ... 0 1)
  # entonces como H es triangular superior, alpha=(0 0 ... 0 1). Luego
  # a NO es una fila de A
  # sí y sólo si (alpha_n)(h_n)=0, es decir,
  # si hn=0.
  if (bitwAnd(bitwShiftR(h, n - 1),1) == 1) {
    k <- n
    f[1] <- 1 # a=(0 0 ... 0 1) es una fila da A
  } else {
    k <- n + 1
  }

  # El Bucle sobre todas las coaliciones
  for (c in 1:(nC - 1)) {
    #Comienza bucle en coaliciones
    # Escribo en binario de derecha a izquierda
    # Busco la posición d del primer cero (desde la posición más a la derecha)
    # Es decir, el primer jugador que no pertenece a c
    ##################################################
    # Encontrar el primer jugador no en c
    cc <- c
    d <- 0
    ii <- 1
    while (d == 0) {
      if (cc %% 2 == 0) {
        d <- ii
      }
      ii <- ii + 1
      cc <- floor(cc / 2)
    }
    # Posición del jugador d en el vector binario c
    dd <- n - d + 1
    #  Fin del cálculo del valor d, primer jugador que no está en c
    #######################################################
    # Each coalition is obtained form the previous one
    # by adding the corresponding vector u^d; a^(c+1)=a^c+u^d
    # Each solution alpha of xH=a is obtained from the previous one
    # by adding the corresponding mu^d (the solution of muH=u^d).
    # Actualizar alpha
    alpha <- alpha + mu[d,]
    #  Comprobamos si (c+1)=c+u^d está en A. Recalculamos el valor k.
    # Observemos que, si el último valor de k satisface k<dd, entonces k es
    # el menor índice para el cual en la coalición previa,  alpha_k*h_k\not=0.
    # Como el nuevo ALPHA=alpha+u^d, el índice mínimo k para este paso es el
    # mismo que el previo, ya que las componentes de u^d*h son nulas hasta la
    # dd.
    # Luego sólo recalculo si k>=dd.
    if (k >= dd) {
      jj <- dd
      while (jj <= n && (abs(alpha[jj]) < tol || bitwAnd(bitwShiftR(h, jj - 1),1) == 0)) {
        # Si alpha_j*h_j=0, sigo con la siguiente coordenada
        jj <- jj + 1
      }
      k <- jj # % O bien salí del bucle while por la segunda condición y k<=n
      # o bien salí porque llegué a la última coordenada n, en cuyo
      # caso, k=n+1;
    }
    # Si k=n+1 entonces alpha*h=0 y la coalición c no está en A
    # En caso contrario, c+a está en A
    #############
    # Calculamos el valor de f(c+1). Si k<=n entonces f(c+1)=1, en caso
    # contrario f(c+1)=0. Ahora, guardamos ese valor en forma compacta.
    # Determinamos el bloque y la posición en dicho bloque
    Pos <- (c + 1) %% bloque
    # Ojo, cuando llego a la última posición de un bloque
    if (Pos == 0) {
      Pos <- bloque
      chunck <- (c + 1) / bloque
    } else {
      chunck <- (c + 1 - Pos) / bloque + 1
    }
    ########
    if (k <= n) {
      f[chunck] <- bitwOr(f[chunck],bitwShiftL(1, Pos-1))
      #print(f[chunck])
    }
  } # Fin del bucle en las coaliciones
  return(f)
}

CoalitionsInd <- function(F1, F2 = NULL, n = NULL) {
  # COALITIONSIND selects the independent coalitions in a collection


  # F1=c(S1, S2, ... Sp), F2=c(R1,...,Rq) Familias de coaliciones de n jugadores
  # Sn y Rn son los numeros binarios de las coaliciones
  # F=c(T1 ... Tt) son las coaliciones independientes en la union de F1 y F2.

  # Ejemplo. CoalitionsInd(c(7,3),c(3,5,13),4)
  # Calculamos la union Fam de F1 y F2
  if (is.null(F2)) { # La familia F2 es la coalicion vacia
    Fam <- sort(unique(F1))
  } else {
    Fam <- sort(unique(c(F1, F2)))
  }

  numC <- length(Fam)  # Numero de coaliciones en la familia Fam

  if (is.null(n)) { # si no nos dan el numero de jugadores, Calculamos el jugador mas grande en la familia Fam
    n <- floor(log2(Fam[1])) + 1 # mayor jugador en Fam(1)
    for (ii in 2:numC) {
      n <- max(n, floor(log2(Fam[ii])) + 1) # maximo entre el anterior valor  y el mayor jugador en el siguiente miembro de la familia Fam
    }
  }

  # Calculamos la matriz de coaliciones completa
  M <- matrix(0, nrow = numC, ncol = n)
  #for (ii in 1:numC) {
  # M[ii, ] <- as.integer(intToBits(Fam[ii])[1:n])
  #}

  # bitwAnd(bitwShiftR(a,pos-1),1),  bitget(Fam[ii], pos)) borrar línea

  for (ii in 1:numC) {
    M[ii, ] <- sapply(1:n, function(pos) bitwAnd(bitwShiftR(Fam[ii],pos-1),1)  )
  }



  # Triangulamos y obtenemos las filas independientes
  pivot <- triangularup(M)$pivot
  # Las filas independientes ordenadas
  Fam <- Fam[sort(pivot[pivot != 0])]
  # Codificamos en binario
  #IND=sum(2.^Fam)

  return(Fam)
}

pasoptimo <- function(x, y, t, v, f, tol = 100*.Machine$double.eps) {
  # PASOPTIMO: Calcula el tamaño de paso óptimo en el algoritmo del prenucleolo DK


  # INPUTS
  # x,t = current feasible solution
  # y = improving direction
  # v = the coalitional game
  # f = the compact-version vector that defines matrix A
  # tol= tolerance level (by default, 100*eps)
  #
  # OUTPUT
  # lambda = min { (aix+t-bi)/(1-aiy) : fi=1 and aiy<1 }
  # jj = Smallest index where the minimun is attained
  #
  # Si a es un (0-1)-vector (los coeficientes de una coalición S)
  # y x es una asignaxión cualquiera
  # entonces ax=x(S).
  # Ya sabemos que x(S) se puede calcular por recurrencia
  # v=c(0,0,1,0,1,1,2)
  # F=2^3-1
  # x=c(2/3,2/3,2/3)
  # y=c(0,0,0)
  # t=0


  # Si no nos dan la tolerancia, fijar tol = 100 * .Machine$double.eps
  nC <- length(v)  # Número de coaliciones
  n <- length(x)   # Número de jugadores

  ########################
  # Vectores de incrementos
  ########################
  increx <- numeric(n)
  increy <- numeric(n)
  # Vector de incrementos de x:
  # x1, x2-x1, x3-x2-x1, x4-x3-x2-x1,...

  increx[1]=x[1]
  increy[1]=y[1]
  for (ii in 2:n) {
    aux <- x[ii] # xi
    for (jj in 1:(ii-1)) {
      aux <- aux - x[jj]  # xi - (x1 + ... + x(i-1))
    }
    increx[ii] <- aux

    #  Vector de incrementos de y:
    # y1, y2-y1, y3-y2-y1, y4-y3-y2-y1,...
    aux <- y[ii] # y(i)
    for (jj in 1:(ii-1)) {
      aux <- aux - y[jj]
    }
    increy[ii] <- aux # yi - (y1 + ... + y(i-1))
  }

  # Alternativamente:
  # cumx=cumsum(x) % Sumas acumuladas de x
  # cumy=cumsum(y) % Sumas acumuladas de y
  # increx=[x(1) x(2:end)-cumx(1:end-1)] % Vector de incrementos de x
  # increy=[y(1) y(2:end)-cumy(1:end-1)] % Vector de incrementos de y
  #
  # Ahora x(S)=x(Sant)+increx(p), y(S)=y(Sant)+increy(p)
  # siendo p el menor jugador en S
  #

  # Primer vector S = (0 0 ... 0 1)
  ax <- x[1]
  ay <- y[1]
  #####
  # bitwAnd(bitwShiftR(a,pos-1),1), bitwAnd(bitwShiftR(f[chunck], Pos-1),1)
  ## lo que estaba ((bitget(f[chunck], Pos) == 1 ), borrar
  chunck <- 1
  Pos <- 1
  bloques <- 31
  #####
  if ((bitwAnd(bitwShiftR(f[chunck], Pos-1),1) )  && ay < 1 - tol) {
    jj <- 1
    lambda <- (ax + t - v[1]) / (1 - ay)
  } else {
    jj <- 0
    lambda <- Inf
  }

  # Bucle para las demás coaliciones (por recurrencia)
  for (c in 1:(nC - 1)) {
    # La coalicion efectiva en este bucle es S=c+1
    # Encontramos la posicion de f(c+1)
    Pos <- Pos + 1
    if (Pos > bloques) {
      Pos <- 1
      chunck <- chunck + 1
    }

    # Alernativamente:
    # Pos=mod(c+1,bloque);
    # if Pos==0
    # Pos=52;chunck=(c+1)/bloque;
    # else
    # chunck=(c+1-Pos)/bloque+1;
    # end

    # Primer jugador no en S: m
    m <- 0
    ii <- 0
    cc <- c
    while (m == 0) {
      ii <- ii + 1
      if (cc %% 2 == 0) {
        m <- ii
      }
      cc <- floor(cc / 2)
    }

    ax <- ax + increx[m]  # Valor x(S)
    ay <- ay + increy[m]  # Valor y(S)

    #Aplico las definiciones:
    # lambda = min { (aix+t-bi)/(1-aiy) : fi=1 and aiy<1 }
    # jj = Smallest index where the minimun is attained

    # Calculo del mínimo lambda

    if  ((bitwAnd(bitwShiftR(f[chunck], Pos-1),1)) && ay < 1 - tol) { #  Si se dan las condiciones del minimo
      L <- (ax + t - v[c + 1]) / (1 - ay) # Calculo el valor correspondiente
      if (L < lambda) { # Si el nuevo valor es estrictamente menor que el último lamba
        lambda <- L # Reemplazo el nuevo valor mínimo de lambda
        jj <- c + 1 # Reemplazo el nuevo índice donde se alcanza
      }
    }
  } # Fin del bucle en las coaliciones

  return(list(lambda = lambda, jj = jj))
}

sistemaFamA <- function(Fam, A, n, tol = 100*.Machine$double.eps) {
  # sistemaFamA Solves the linear system zF+uA=0, u1=1

  # INPUT
  # Fam = Family of coalitions
  # A = Subfamily of independent coalitions from A (A has maximum rank)
  # n = Number of players
  # tol = Tolerance level (default: 100eps)
  #
  # OUTPUT
  # y = solution of the linear system zFam+uA=0, u1=1. Here, y=(z u).
  # exit =
  # -1 ----- (Caso 1) Óptimo
  #  0 ----- (Caso 3) No es óptimo (y no hay que modificar A)
  #  k ----- (Caso 2) No es optimo y k es la fila de A que hay que suprimir
  #
  # If exit=-1, i.e., y=(z u) is a solution with u>=0, then
  # sat = vector of coalitions for which the component uj>0
  # sat = Inf,  otherwise.
  # sistemaFamA(7,3,3)

  ######
  # # Si no nos dan la tolerancia, fijar tol = 100 * .Machine$double.eps
  ######




  nFam <- length(Fam)  # Number of coalitions in Fam
  nA <- length(A)  # Number of coalitions in A
  m <- nFam + nA     # Total number of coalitions



  # Control de errores
  # if m>n
  #   warning('Más filas que columnas')
  # end
  # A partir de los vectores Fam y A construiremos las (0-1)-matrices:
  # Fam de orden nFamxn
  # A de orden nAxn
  # Consideremos el sistema zFam+uA=0, u1=1.
  # z es 1xnFam
  # u es 1xnA
  # El 1 en u1 es un vector columna nAx1
  # El segundo termino de u1=1 es el escalar 1 (1x1)
  # Luego el sistema zFam+uA=0, u1=1; puede escribirse como
  #        [z u]*[Fam 0;A 1]= [0 1]      (S1)
  # Si denotamos por C la matriz [Fam 0;A 1] y por y=[z u]
  # entonces, el sistema (S1) es equivalente a
  #           y*C=[0 1]                 (S2)
  # Aquí, C es mx(n+1) mientras que y es 1xm.
  # Resolver (S2) equivale a resolver
  #           C'*y'=[0;1]                (S3)
  # La matriz ampliada de (S3) es
  #          M'=[C' | [0;1]] de orden (n+1)x(m+1).
  # En definitiva, la traspuesta de la matriz ampliada M' es:
  #
  #              M=[Fam 0;A 1;0 1] de orden (m+1)x(n+1)

  # Construimos la matriz M
  M <- matrix(0, nrow = m + 1, ncol = n + 1)

  ###################
  # Ultima columna; nFam ceros (ya colocados)  y nB unos
  M[(nFam + 1):m, n + 1] <- 1

  for (ii in 1:nFam) {
    for (jj in 1:n) {M[ii,jj] <- bitwAnd(bitwShiftR(Fam[ii],jj-1),1)}
    # M[ii,(1:n) ] <-  bitget(Fam[ii], (1:n))
  }

  for (ii in 1:nA) {
    for (jj in 1:n) {M[nFam + ii,jj] <- bitwAnd(bitwShiftR(A[ii],jj-1),1)}
    # M[nFam + ii, (1:n)] <-  bitget(A[ii], (1:n))
  }

  # La última fila
  M[m + 1, n + 1] <- 1

  # Resolvemos el sistema C' * y' = [0;1]
  result <- solvels(t(M), tol)
  y <- result$solution
  flag <- result$flag

  #  % flag =  1  Sistema compatible determinado
  # flag =  0  Sistema compatible indeterminado
  # flag = -1  Sistema incompatible
  #################################
  ## CASUISTICA
  #################################
  # Este sistema puede ser:
  # CASO 1: Compatible determinado, con solución y=(z u)
  # tal que u>=0
  # CASO 2:  Compatible determinado, con solución y=(z u)
  # tal que ui<0 para algún i
  # CASO 3:  Incompatible
  #

  # Control de errores
  #  1) Nunca debería darme sistema compatible indeterminado
  if (flag == 0) {
    stop("Compatible indeterminate system.")
  }
  #  2) Si el sistema es compatible determinado, compruebo la longitud del vector solucion

  if (flag == 1 && length(y) != (nFam + nA)) {
    stop("The length of 'y' is not nFam + nA.")
  }
  ############################
  # Casos
  ############################
  if (flag == 1 && sum(y[(nFam + 1):length(y)] >= -tol) == (m - nFam)) { # Case 1
    # Todas las coordenadas "finales" de la solución y son mayores o iguales que 0
    # (en realidad, superiores a -tol)
    # disp('Caso 1')
    exit <- -1 # (x,t) Solución óptima
    #  Utilizamos las coordenadas finales estrictamente positivas de
    # y=(z u), o sea, aquellas tales que u_j>0,
    # para saber que restricciones de A se saturan en el óptimo
    sat <- A[y[(nFam + 1):length(y)] > tol]
  } else if (flag == 1) { # Case 2
    jj <- which(y[(nFam + 1):length(y)] < -tol)[1] # Smallest index such that  u_j<0
    M[nFam + jj, ] <- rep(0, n + 1) # OJO!!! Eliminamos la fila jj+1,  porque hemos colocado en primer lugar la coalicion [1 ... 1]
    # M[jj + 1, ] <- rep(0, n + 1)
    M[m + 1, n + 1] <- 0 #  Eliminamos la ultima ecuación (la traspuesta de u1=1).

    # Resolvemos el sistema [Fam;B]y=[0;1]
    result <- solvels(M, tol)
    y <- result$solution
    flag <- result$flag
    #  Control de error
    # El sistema debería ser compatible
    if (flag == -1) {
      stop("System [Fam;B]y=[0;1] is incompatible.")
    }

    sat <- Inf
    exit <- jj
    #  % la fila de A que debemos suprimir (aquí es jj)
    # Porque en la siguiente iterada ya consideramos la Fam aparte de la
    # matriz A.
  } else { # CASE 3
    M[m + 1, n + 1] <- 0 #Eliminamos la ultima ecuación (la traspuesta de u1=1).
    # Resolvemos el sistema [Fam;B]y=[0;1]

    result <- solvels(M, tol)
    y <- result$solution
    flag <- result$flag
    #Eliminamos la ultima ecuación (la traspuesta de u1=1).
    # Resolvemos el sistema [Fam;B]y=[0;1]
    if (flag == -1) {
      stop("System [Fam;B]y=[0;1] is incompatible.")
    }

    sat <- Inf
    exit <- 0 # El caso más habitual (creo)
  }

  return(list(y = y, exit = exit, sat = sat))
}
