#' @title Strategically equivalent check
#' @description This function checks if two games are strategically equivalent.
#' @param v A characteristic function, as a vector.
#' @param w A characteristic function, as a vector.
#' @param binary A logical value. By default, \code{binary=FALSE}. Should be set to \code{TRUE} if \code{v} and \code{w} are introduced in binary order instead of lexicographic order.
#' @param parameters A logical value. By default, \code{parameters=FALSE}.
#' @return \code{TRUE} if \code{v} and \code{w} are strategically equivalent, \code{FALSE} otherwise. If \code{parameters=TRUE}, whenever \code{v} and \code{w} are strategically equivalent, the function also returns \code{k} (a positive integer) and \code{a} (the characteristic function of an additive game, as a vector in binary order if \code{binary=TRUE} and in lexicographic order otherwise) such that \eqn{\code{v} = \code{k} \code{w} + \code{a}}.
#' @details Games \eqn{v\in G^N} and \eqn{w\in G^N} are strategically equivalent if there exist \eqn{k>0} and an additive game \eqn{a\in G^N} such that \eqn{v(S)=k w(S)+a(S)} for all \eqn{S\in 2^N}.
#' @examples
#' w <- c(1000, 0, 0, 2000, 3000, 2000, 4000)
#' v <- 4.5 * w + additivegame(c(4, 6, 1), binary = TRUE)
#' strategicallyequivalentcheck(v, w, binary = TRUE, parameters = TRUE)
#' @seealso \link{additivegame}, \link{normalizedgame}, \link{zeronormalizedgame}
#' @export

strategicallyequivalentcheck <- function(v, w, binary = FALSE, parameters = FALSE) { # Principio de la función

  ################################
  ### Comprobación datos entrada###
  ################################

  nC <- length(v) # Número de coaliciones.
  n <- log(nC + 1) / log(2) # Número de jugadores.
  if (n > floor(n)) {
    stop("'v' must have length 2^n-1 for some n.")
  }

  nC2 <- length(w) # Número de coaliciones de w.
  n2 <- log(nC2 + 1) / log(2) # Número de jugadores.
  if (n2 > floor(nC2)) {
    stop("'w' must have length 2^n-1 for some n.")
  }
  if (nC != nC2) {
    stop("'v' and 'w' must have the same length.")
  }
  if (binary == FALSE) { # Si los juegos se introdujeron en lexicográfico, los pasamos a binario.
    v <- lex2bin(v)
    w <- lex2bin(w)
  }

  ################################
  ##### Creación de variables######
  ################################

  checkR <- TRUE # Por defecto, mientras no encuentre un contraejemplo, checkR=TRUE
  k <- NULL # Parámetro k.
  a <- NULL # Vector del juego aditivo.
  tol <- 12 # OJO: aquí la tolerancia es un entero (número de cifras decimales)

  ################################
  ###### Cuerpo de la función######
  ################################

  # 0-normalización de v y w.
  v0 <- v
  w0 <- w
  for (S in 1:nC)
  {
    # Calculamos los jugadores pertenecientes a cada coalición y su valor en v.
    num <- S
    J <- floor(log2(num)) + 1 # Jugador más grande
    Sjug <- c()
    for (ii in 1:J) # Algoritmo de paso de un número en decimal a binario.
    {
      if (num %% 2 == 1) {
        Sjug <- c(Sjug, ii)
      }
      num <- num %/% 2
    }
    v0[S] <- v[S] - sum(v[2.^(Sjug - 1)])
    w0[S] <- w[S] - sum(w[2.^(Sjug - 1)])
  }

  if (sum((w0 != 0) == (v0 != 0)) < nC) { # Si el número de coaliciones iguales a 0 es diferente en los dos juegos, entonces no son estratégicamente equivalentes.
    checkR <- FALSE
  } else {
    if (length(unique(round(v0[v0 != 0] / w0[w0 != 0], tol))) > 1) { # Si al dividir obtenemos más de un índice, no son estratégicamente equivalentes.
      # Añadimos round porque puede dar problemas con muchos decimales.
      checkR <- FALSE
    }
  }

  ################################
  ###### Salidas de la función#####
  ################################


  if (parameters == FALSE) { # Si el usuario sólo pide una salida, es decir, instance=FALSE
    return(check = checkR)
  } else { # Si instance=TRUE, la función devuelve también el número positivo k y la función característica a.
    if (checkR == TRUE) { # Si son estratégicamente equivalentes, calculamos a y k.
      if (sum(v0 == 0) == nC & sum(w0 == 0) == nC) { # Caso trivial
        k <- 1
      } else {
        k <- unique(v0[v0 != 0] / w0[w0 != 0])[1] # Parametro k.
      }
      a <- v - k * w # Juego aditivo.
      # a=a[2.^(seq(1,n)-1)] #Vector de valores de coaliciones individuales del juego aditivo.
    }
    if (binary == FALSE) { # Devolvemos k y a (en orden lexicográfico).
      return(list(check = checkR, k = k, a = bin2lex(a)))
    } else { # Devolvemos k y a (en orden binario).
      return(list(check = checkR, k = k, a = a))
    }
  }
} # Fin de la función
