#' Calculate Entropy for Amino Acid Sequences
#'
#' This function calculates the entropy of a set of amino acid sequences. 
#' It is used to evaluate the diversity of sequences in a given alignment.
#'
#' @param alignment An `AAStringSet` object containing the aligned amino acid sequences.
#' @return A numeric vector representing the entropy values for each position in the alignment.
#' @details The entropy is calculated based on the frequency of amino acids at each position in the alignment.
#' Higher entropy indicates greater variability at that position.
#' @examples
#' # Example usage:
#' library(Biostrings)
#' sequences <- AAStringSet(c("ACDEFGHIK", "ACDEFGHIK", "ACDEFGHIK"))
#' entropy_values <- entropy_RR(sequences)
#' print(entropy_values)
#' @export

entropy_RR <- function (alignment)
{

    alignment = as.matrix(alignment)
  aa <- c("V", "I", "L", "M", "F", "W", "Y", "S", "T", "N",
          "Q", "H", "K", "R", "D", "E", "A", "G", "P", "C", "-" )
  composition <- table(alignment)
  unk <- composition[!names(composition) %in% aa]
  if (length(unk) > 0) {
    for (i in 1:length(unk)) alignment[alignment == names(unk[i])] = "-"
  }
  len <- ncol(alignment)
  freq.21 <- matrix(0, nrow = 21, ncol = ncol(alignment), dimnames = list(aa,
                                                                          seq(1:len)))
  for (i in 1:len) {
    freq.21[names(summary((as.factor(toupper(alignment[,
                                                       i]))))), i] <- (summary(as.factor(toupper(alignment[,
                                                                                                           i])))/length(alignment[, i]))
  }
  entropy.21 <- vector(length = len)
  for (i in 1:len) {

    entropy.21[i] <- -1 * sum(freq.21[freq.21[, i] != 0,i] * log(freq.21[freq.21[, i] != 0, i], base = 21))
  }

 entropy.21
}




