#' Creates Ada-plot for the provided data.
#'
#' @description Ada-plot identifies the characteristics of the distribution such as symmetry, skewness, and outliers of the data set.
#' @usage adaplot(X, title = "Ada-plot", xlab = "x", lcol = "black", rcol = "grey60", ...)
#' @param X an \eqn{n} by \eqn{1} matrix, equivalently, a column vector of length \eqn{n}, where \eqn{n} is the number of observations.
#' @param title title of the plot, \emph{Ada-plot} by default.
#' @param xlab \eqn{x}-axis label, \eqn{x} by default.
#' @param lcol color of the points which correspond to data on or below the sample average, \emph{black} by default.
#' @param rcol color of the points which correspond to data above the sample average, \emph{grey60} by default.
#' @param ... other graphical parameters.
#'
#' @import ggplot2
#' @return Ada-plot
#' @references Wijesuriya, U. A. (2025a). Ada-plot and Uda-plot as Alternatives for
#'             Ad-plot and Ud-plot.
#'             \emph{Communications in Statistics-Theory and Methods}, \doi{10.1080/03610926.2025.2558108}.
#' @references Wijesuriya, U. A. (2025b). Ad-plot and Ud-plot for Determining Distributional Characteristics
#'             and Normality.
#'             \emph{Communications in Statistics-Theory and Methods}, \doi{10.1080/03610926.2024.2440583}.
#' @references Wijesuriya, U. A. (2025c). adplots: Ad-plot and Ud-plot for Visualizing Distributional
#'             Properties and Normality.
#'             \emph{R Package Version 0.1.0}, \doi{10.32614/CRAN.package.adplots}.
#' @examples
#'    set.seed(0)
#'    X1 <- matrix(rnorm(50, mean = 2, sd = 5))
#'    adaplot(X1)
#'
#'    X2 <- matrix(rf(50, df1 = 10, df2 = 5))
#'    adaplot(X2)
#'
#'    X3 <- matrix(rbeta(50, shape1 = 10, shape2 = 2))
#'    adaplot(X3, title = "", lcol = "blue", rcol = "red")
#' @export
adaplot <- function(X, title = "Ada-plot", xlab = "x", lcol = "black", rcol = "grey60",...){
  D <- sort(X)                                     # Sorts data in ascending order
  n <- length(D)                                   # Computes sample size
  xbar <- mean(D)                                  # Computes sample average
  abdev <- abs(D-xbar)                             # Computes absolute deviation
  A <- matrix(NA, nrow = n, ncol = 1)              # Defines an NA column vector size n
  for (j in 1:n){
    A[j]=round((1/n)*sum(abdev[abdev<=abs(xbar-D[j])]),10) # Empirical Centralized Cumulative Average Deviation function
  }
  B=min(A)+max(A)-A
  df <- data.frame(D, B)                           # Creates a dataframe
  g <- ggplot(df, aes(x = D))+
    geom_point(aes(x = D, y = B), col = ifelse(D <= xbar, lcol, rcol))+
    labs(title = title)+
    xlab(xlab)+
    ylab(parse(text=paste0('A[',n,'](abs(x-',round(xbar,2),'))')))+
    theme(axis.text = element_text(size = 12), axis.title = element_text(size = 12))
  return(g)
}

