\encoding{UTF-8}


\name{CreateInputsPert}
\alias{CreateInputsPert}
\alias{[.InputsPert}
\alias{plot.InputsPert}


\title{Generation of ensemble model inputs for data assimilation}


\usage{
CreateInputsPert(FUN_MOD, DatesR, Precip = NULL,
                 PotEvap = NULL, TempMean = NULL,
                 ZInputs = NULL, HypsoData = NULL, NLayers = 5,
                 NbMbr = 50, Seed = NULL)

\method{[}{InputsPert}(x, i)

\method{plot}{InputsPert}(x, which = "all", main = NULL,
     ColPrecip = "royalblue", ColPotEvap = "green3",
     ask = prod(par("mfcol")) < length(which) && dev.interactive(), ...)
}


\arguments{
\item{FUN_MOD}{[function] hydrological model function (e.g. \code{\link[airGR]{RunModel_GR5J}}, \code{\link[airGR]{RunModel_CemaNeigeGR5J}})}

\item{DatesR}{[POSIXct] vector of dates}

\item{Precip}{(optional) [numeric] time series of total precipitation to perturb [mm/d]}

\item{PotEvap}{(optional) [numeric] time series of potential evapotranspiration to perturb [mm/d]}

\item{TempMean}{(optional) [numeric] time series of mean air temperature [°C] (not perturbed), required to create the CemaNeige module inputs}

\item{ZInputs}{(optional) [numeric] real giving the mean elevation of the Precip and Temp series (before extrapolation) [m], possibly used to create the CemaNeige module inputs}

\item{HypsoData}{(optional) [numeric] vector of 101 reals: min, q01 to q99 and max of catchment elevation distribution [m], if not defined a single elevation is used for CemaNeige}

\item{NLayers}{(optional) [numeric] integer giving the number of elevation layers requested [-], required to create CemaNeige module inputs, default=5}

\item{NbMbr}{(optional) [numeric] number of ensemble members (minimum of 20 recommanded for the EnKF scheme and of 30 for the PF scheme)}

\item{Seed}{(optional) [numeric] seed of random number generator}

\item{x}{[InputsPert] containing the vector of dates (\emph{POSIXt}) and the time series of numeric values list perturbed}

\item{i}{[integer] of the indices to subset a time series or [character] names of the elements to extract}

\item{which}{(optional) [character] choice of plots (e.g. \code{"Precip"} or \code{"PotEvap"}, default = \code{"all"})}

\item{main}{(optional) [character] an overall title for the plot (see \code{\link[graphics]{title}})}

\item{ColPrecip, ColPotEvap}{(optional) [character] color to be used for perturbed precipitation and perturbed potential evapotransipration (in any format that \code{\link[grDevices]{col2rgb}} accepts)}

\item{ask}{(optional) [logical] if \code{TRUE}, the user is asked before each plot, see \code{\link[graphics]{par}(ask = .)}}

\item{...}{other parameters to be passed through to plotting functions}
}


\value{
[list] of class \code{'InputsModel'} containing the ensembles of perturbed model inputs required to perform data assimilation:
  \item{DatesR }{[POSIXlt] vector of dates}
  \item{Precip }{[numeric] matrix (dim(NbTime, NbMbr)) of ensemblist time series of perturbed total precipitation [mm/d] (with NbTime the length of the period; generated only if the precipitation time series is provided as a function argument)}
  \item{PotEvap}{[numeric] matrix (dim(NbTime, NbMbr)) of ensemblist time series of perturbed potential evapotranspiration [mm/d] (generated only if the time series of potential evapotranspiration is provided as a function argument)}
  \item{NbMbr  }{[integer] atomic vector of number of ensemble members}

On the graphical outputs: \cr
  - solid line: medians of the input values \cr
  - polygon: minima and maxima of the input values \cr
}


\description{
Function which perturbs the model inputs to generate probabilistic meteorological forcings required to perform ensemble-based data assimilation.
}


\details{
The function generates an ensemble of precipitation or/and potential evapotranspiration time series, from data provided as function argument/s. The mean air temperature required to create CemaNeige inputs is not perturbed.
\cr\cr
Probabilistic forcing/s is/are generated by stochastically perturbing the meteorological variable/s through a multiplicative stochastic noise, according to the methodology proposed by Clark et al. (2008).
\cr\cr
The random perturbations are provided through a first-order autoregressive model relying on a fractional error parameter equal to 0.65 and a temporal decorrelation length of 1 day for rainfall and 2 days for potential evapotranspiration.
\cr\cr
In order to ensure reproducible results, \emph{Seed} can be set to fix the randomness in the generation of perturbations.
\cr\cr
For further details, see the references section.
\cr\cr
Nota: The function can be applied when using GR4J, GR5J and GR6J models (i.e. daily model time step), with or withouth the CemaNeige module.
}


\examples{
library(airGRdatassim)

## loading catchment data
data(L0123001, package = "airGR")

## preparation of the InputsModel object
InputsModel <- CreateInputsModel(FUN_MOD = RunModel_GR5J, DatesR = BasinObs$DatesR,
                                 Precip = BasinObs$P, PotEvap = BasinObs$E)

## run period selection
IndRun <- seq(which(format(BasinObs$DatesR, format = "\%Y-\%m-\%d")=="2006-01-01"),
              which(format(BasinObs$DatesR, format = "\%Y-\%m-\%d")=="2006-01-31"))

## preparation of perturbed meteorological ensemble
InputsPert <- CreateInputsPert(FUN_MOD = RunModel_GR5J,
                               DatesR = BasinObs$DatesR,
                               Precip = BasinObs$P,
                               PotEvap = BasinObs$E,
                               NbMbr = 100L)
str(InputsPert)

## results preview
oldpar <- par(mfrow = c(2, 1))
plot(InputsPert)
par(oldpar)

## results preview on a subset on one perturbed variable
oldpar <- par(mfrow = c(1, 1))
plot(InputsPert[IndRun], which = "PotEvap")
par(oldpar)
}


\author{
Gaia Piazzi, Olivier Delaigue
}


\references{
- Clark, M. P., Rupp, D. E., Woods, R. A., Zheng, X., Ibbitt, R. P., Slater, A. G. et al. (2008). Hydrological data assimilation with the ensemble Kalman filter: Use of streamflow observations to update states in a distributed hydrological model. Advances in Water Resources, 31(10), 1309-1324, \doi{10.1016/j.advwatres.2008.06.005}
\cr\cr
- Piazzi, G., Thirel, G., Perrin, C. and Delaigue, O. (2021). Sequential data assimilation for streamflow forecasting: assessing the sensitivity to uncertainties and updated variables of a conceptual hydrological model at basin scale. Water Resources Research, 57, \doi{10.1029/2020WR028390}.
}


\seealso{
\code{\link{RunModel_DA}}
}
