#######################################################
#	apc package
#	Bent Nielsen, 17 September 2016, version 1.2.3
#	Data examples
#######################################################
#	Copyright 2014-2016 Bent Nielsen
#	Nuffield College, OX1 1NF, UK
#	bent.nielsen@nuffield.ox.ac.uk
#
#	This program is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.
#######################################################

#############################################################################################
#	SPECIFIC DATA SETS
#############################################################################################

###############################
#	JAPANESE BREAST CANCER DATA
###############################
data.Japanese.breast.cancer	<- function()
#	BN, 24 apr 2015	(17 oct 2013)
#	An example with A,P,C effects
#
#	Taken from table I of
#	Clayton, D. and Schifflers, E. (1987b)
#	Models for temperoral variation in cancer rates. II: age-period-cohort models.
#	Statistics in Medicine 6, 469-481.
#
#	Original Table caption:
#	age-specific mortality rates (per 100,000 person-years observation) of breast cancer in Japan,
#	during the period 1955-1979. Numbers of cases on which rates are based are in parentheses
#	(source: WHO mortality data base).  
{	#	data.Japanese.breast.cancer
v.rates		<- c( 0.44, 0.38, 0.46, 0.55, 0.68,
			 	  1.69, 1.69, 1.75, 2.31, 2.52,
				  4.01, 3.90, 4.11, 4.44, 4.80,
				  6.59, 6.57, 6.81, 7.79, 8.27,
				  8.51, 9.61, 9.96,11.68,12.51,
				 10.49,10.80,12.36,14.59,16.56,
				 11.36,11.51,12.98,14.97,17.79,
				 12.03,10.67,12.67,14.46,16.42,
				 12.55,12.03,12.10,13.81,16.46,
				 15.81,13.87,12.65,14.00,15.60,
				 17.97,15.62,15.83,15.71,16.52)
v.cases		<- c(   88,   78,  101,  127,  179,
				   299,  330,  363,  509,  588,
				   596,  680,  798,  923, 1056,
				   874,  962, 1171, 1497, 1716,
				  1022, 1247, 1429, 1987, 2398,
				  1035, 1258, 1560, 2079, 2794,
				   970, 1087, 1446, 1828, 2465,
				   820,  861, 1126, 1549, 1962,
				   678,  738,  878, 1140, 1683,
				   640,  628,  656,  900, 1162,
				   497,  463,  536,  644,  865)				 
col.names	<- paste(as.character(seq(from=1955,length=5,by=5)),"-",
			 		 as.character(seq(from=1955,length=5,by=5)+4),sep="")
row.names	<- paste(as.character(seq(from=25  ,length=11,by=5)),"-",
			 		 as.character(seq(from=25  ,length=11,by=5)+4),sep="")
rates	<- matrix(data=v.rates,nrow=11, ncol=5,byrow=TRUE,dimnames=list(row.names,col.names))
cases	<- matrix(data=v.cases,nrow=11, ncol=5,byrow=TRUE,dimnames=list(row.names,col.names))
return(apc.data.list(
			response	=cases			,
			dose		=cases/rates	,
			data.format	="AP"			,
			age1		=25				,
			per1		=1955			,
			unit		=5				,
			label		="Japanese breast cancer"))
}	#	data.Japanese.breast.cancer

##################################
#	ITALIAN BLADDER CANCER DATA
##################################
data.Italian.bladder.cancer	<- function()
#	BN, 24 apr 2015 (17 oct 2013)
#	An example with A,C effects
#
#	Taken from table IV of
#	Clayton, D. and Schifflers, E. (1987a)
#	Models for temperoral variation in cancer rates. I: age-period and age-cohort models.
#	Statistics in Medicine 6, 449-467.
#
#	Original Table caption:
#	age-specific incidence rates (per 100,000 person-years observation) of bladder cancer in
#	Italian males during the period 1955-1979. Numerators are in parentheses
#	(source of data: WHO mortality database).  
{	#	data.Italian.bladder.cancer
v.rates		<- c( 0.03, 0.03, 0.01, 0.04,  0.12,
				  0.17, 0.18, 0.12, 0.08,  0.09,
				  0.32, 0.31, 0.35, 0.42,  0.32,
				  1.04, 1.05, 0.91, 1.04,  1.27,
				  2.86, 2.52, 2.61, 3.04,  3.16,
				  6.64, 7.03, 6.43, 6.46,  8.47,
				 12.71,13.39,14.59,14.64, 16.38,
				 20.11,23.98,26.69,27.55, 28.53,
				 24.40,33.16,42.12,47.77, 50.37,
				 32.81,42.31,52.87,66.01, 74.64,
				 45.54,47.94,62.05,84.65,104.21)
v.cases		<- c(   3,   3,   1,   4,  12,
  				   16,  17,  11,   8,   8,
				   24,  29,  33,  39,  30,
				   79,  76,  82,  95, 115,
				  234, 185, 183, 267, 285,
				  458, 552, 450, 431, 723,
				  720, 867,1069, 974,1004,
				  890,1230,1550,1840,1811,
				  891,1266,1829,2395,3028,
				  920,1243,1584,2292,3176,
				  831, 937,1285,1787,2659)
col.names	<- paste(as.character(seq(from=1955,length=5,by=5)),"-",
			 		 as.character(seq(from=1955,length=5,by=5)+4),sep="")
row.names	<- paste(as.character(seq(from=25  ,length=11,by=5)),"-",
			 		 as.character(seq(from=25  ,length=11,by=5)+4),sep="")
rates	<- matrix(data=v.rates,nrow=11, ncol=5,byrow=TRUE,dimnames=list(row.names,col.names))
cases	<- matrix(data=v.cases,nrow=11, ncol=5,byrow=TRUE,dimnames=list(row.names,col.names))
return(apc.data.list(
			response	=cases			,
			dose		=cases/rates	,
			data.format	="AP"			,
			age1		=25				,
			per1		=1955			,
			unit		=5				,
			label		="Italian bladder cancer"))
}	#	data.Italian.bladder.cancer

##################################
#	BELGIAN LUNG CANCER DATA
##################################
data.Belgian.lung.cancer	<- function(unbalanced=FALSE)
#	BN, 17 oct 2013
#	An example with A,drift effects
#
#	Taken from table VIII of
#	Clayton, D. and Schifflers, E. (1987a)
#	Models for temperoral variation in cancer rates. I: age-period and age-cohort models.
#	Statistics in Medicine 6, 449-467.
#
#	Original Table caption:
#	age-specific mortality rates (per 100,000 person-years observation) of lung cancer in
#	Belgian females during the period 1955-1978. Numerators are shown in parentheses
#	(source of data: WHO mortality database).
#
#	NOTE	The data are unbalanced since the last column only covers 4 years.  This is not used.
#	In:		unbalanced		logical.  If true unbalanced version includind last column
{	#	data.Belgian.lung.cancer
v.rates		<- c( 0.19, 0.13, 0.50, 0.19, 0.70,
				  0.66, 0.98, 0.72, 0.71, 0.57,
				  0.78, 1.32, 1.47, 1.64, 1.32,
				  2.67, 3.16, 2.53, 3.38, 3.93,
				  4.84, 5.60, 4.93, 6.05, 6.83,
				  6.60, 8.50, 7.65,10.59,10.42,
				 10.36,12.00,12.68,14.34,17.95,
				 14.76,16.37,18.00,17.60,23.91,
				 20.53,22.60,24.90,24.33,32.70,
				 26.24,27.70,30.47,36.94,38.47,
				 33.47,33.61,36.77,43.69,45.20)
v.cases		<- c(  3,  2,  7,  3, 10,
				  11, 16, 11, 10,  7,
				  11, 22, 24, 25, 15,
				  36, 44, 42, 53, 48,
				  77, 74, 68, 99, 88,
				 106,131, 99,142,134,
				 157,184,189,180,177,
				 193,232,262,249,239,
				 219,267,323,325,343,
				 223,250,308,412,358,
				 198,214,253,338,312)
col.names	<- c("1955-1959","1960-1964","1965-1969","1970-1974","1975-1978")
row.names	<- paste(as.character(seq(from=25  ,length=11,by=5)),"-",
			 		 as.character(seq(from=25  ,length=11,by=5)+4),sep="")
rates	<- matrix(data=v.rates,nrow=11, ncol=5,byrow=TRUE,dimnames=list(row.names,col.names))
cases	<- matrix(data=v.cases,nrow=11, ncol=5,byrow=TRUE,dimnames=list(row.names,col.names))
if(unbalanced==FALSE)
	return(apc.data.list(
			response	=cases[,(1:4)]					,
			dose		=cases[,(1:4)]/rates[,(1:4)]	,
			data.format	="AP"							,
			age1		=25								,
			per1		=1955							,
			unit		=5								))
if(unbalanced==TRUE)
	return(apc.data.list(
			response	=cases			,
			dose		=cases/rates	,
			data.format	="AP"			,
			unit		=5				,
			label		="Belgian lung cancer"))
}	#	data.Belgian.lung.cancer

###################################
##	PROSTATE CANCER FOR NONWHITES IN THE US
###################################
data.US.prostate.cancer	<- function()
##	BN, 28 apr 2015
##	An example with over-dispersion
##
##	Taken from table 2 of
##	Holford, T.R. (1983)
##	The estimation of age, period and cohort effects for vital rates.
##	Biometrics 39, 311-324.
##
##	Original Table caption:
##	Number of prostate cancer deathrs and midperiod population for nonwhites in the
##	U.S. by age and period
##	Sources:
##	Cancer deaths: National Center for Health Statistics, 1937-1973
##	Population 1935-60: Grove and Hetzel, 1968
##	Population 1960-69: Bureau of the Census, 1974		
##	Population measured in 1000s
##
{	#	data.US.prostate.cancer
v.deaths	<- c( 177, 271, 312, 382, 321, 305, 308,
				  262, 350, 552, 620, 714, 649, 738,
				  360, 479, 644, 949, 932,1292,1327,
				  409, 544, 812,1150,1668,1958,2153,
				  328, 509, 763,1097,1593,2039,2433,
				  222, 359, 584, 845,1192,1638,2068,
				  108, 178, 285, 475, 742, 992,1374)

v.population<- c( 301, 317, 353, 395, 426, 473, 498,
				  212, 248, 279, 301, 358, 411, 443,
				  159, 194, 222, 222, 258, 304, 341,
				  132, 144, 169, 210, 230, 264, 297,
				   76,  94, 110, 125, 149, 180, 197,
				   37,  47,  59,  71,  91, 108, 118,
				   19,  22,  32,  39,  44,  56,  66)
col.names	<- paste(as.character(seq(from=1935,length=7,by=5)),"-",
			 		 as.character(seq(from=1935,length=7,by=5)+4),sep="")
row.names	<- paste(as.character(seq(from=50  ,length=7,by=5)),"-",
			 		 as.character(seq(from=50  ,length=7,by=5)+4),sep="")
response	<- matrix(data=v.deaths		,nrow=7, ncol=7,byrow=TRUE,dimnames=list(row.names,col.names))
dose		<- matrix(data=v.population	,nrow=7, ncol=7,byrow=TRUE,dimnames=list(row.names,col.names))
return(apc.data.list(
			response	=response		,
			dose		=dose			,
			data.format	="AP"			,
			age1		=50				,
			per1		=1935			,
			unit		=5				,
			label		="US prostate cancer"))
}	#	data.US.prostate.cancer

##################################
#	UK Asbestos data
##################################
data.asbestos	<- function(all.age.groups=FALSE)
#	BN, 17 oct 2013
#
#	Taken from
#	Martinez Miranda, Nielsen and Nielsen (2013)
#	Inference and forecasting in the age-period-cohort model with unknown exposure with
#	an application to mesothelioma mortality.
#	To appear in Journal of the Royal Statistical Society series A
#	
#	update of data from the Health and Safety Executive
{	#	data.asbestos
v.cases	<-c(0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,1	,1	,0	,2	,0	,1	,2	,0	,0	,1	,0	,0	,2	,1	,1	,1	,1	,4	,1	,1	,4	,5	,3	,5	,3	,3	,6	,3	,2	,3	,4	,1	,4	,1	,0	,2	,1	,0	,1	,0	,0	,0	,0	,2	,0	,0	,0	,1	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,1	,1	,0	,0	,1	,0	,0	,0	,0	,0	,1	,2	,0	,3	,0	,1	,3	,4	,1	,1	,2	,6	,1	,1	,3	,3	,5	,3	,4	,1	,5	,3	,8	,3	,4	,4	,5	,3	,1	,3	,2	,2	,3	,1	,0	,3	,1	,4	,2	,0	,1	,1	,3	,1	,0	,0	,1	,0	,1	,1	,0	,0	,0	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,1	,3	,2	,2	,3	,0	,1	,1	,1	,0	,3	,1	,4	,6	,8	,3	,6	,3	,10	,6	,7	,6	,4	,2	,6	,5	,8	,4	,1	,0	,2	,1	,3	,1	,1	,1	,2	,1	,0	,0	,0	,0	,1	,0	,1	,0	,0	,0	,1	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,2	,1	,0	,1	,0	,0	,1	,1	,2	,1	,0	,1	,1	,4	,2	,1	,2	,5	,4	,6	,5	,10	,3	,4	,11	,10	,5	,9	,1	,5	,4	,7	,6	,3	,2	,4	,6	,5	,0	,1	,0	,0	,1	,2	,2	,1	,0	,1	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,2	,0	,0	,1	,1	,1	,1	,1	,3	,0	,1	,0	,3	,3	,4	,3	,2	,4	,3	,5	,5	,1	,1	,10	,7	,4	,7	,5	,2	,5	,13	,1	,5	,3	,4	,0	,6	,5	,1	,4	,2	,2	,1	,1	,1	,2	,1	,0	,0	,1	,1	,1	,0	,0	,0	,0	,0	,0	,0,
			0	,0	,1	,0	,0	,0	,0	,0	,1	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,2	,1	,2	,1	,6	,1	,2	,5	,2	,4	,4	,2	,3	,4	,6	,9	,7	,8	,8	,3	,6	,6	,5	,2	,6	,7	,4	,10	,5	,5	,3	,5	,6	,2	,1	,1	,2	,0	,3	,0	,1	,1	,0	,1	,1	,0	,2	,0	,0	,0	,0	,0	,0	,1,
			1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,2	,0	,1	,0	,1	,0	,0	,1	,1	,1	,1	,1	,1	,0	,4	,4	,1	,5	,5	,6	,1	,6	,5	,2	,6	,1	,5	,8	,5	,9	,9	,6	,7	,8	,5	,3	,7	,9	,7	,4	,8	,2	,5	,4	,2	,1	,4	,2	,0	,1	,0	,1	,1	,1	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,2	,0	,0	,0	,2	,0	,1	,3	,4	,3	,5	,2	,6	,2	,5	,3	,4	,4	,11	,3	,5	,10	,10	,3	,6	,11	,7	,8	,6	,6	,4	,9	,10	,7	,5	,2	,3	,2	,0	,4	,0	,0	,2	,2	,0	,2	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,1	,1	,1	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,2	,1	,2	,2	,1	,2	,6	,2	,3	,12	,7	,5	,3	,4	,3	,4	,3	,8	,8	,6	,11	,11	,9	,11	,11	,4	,6	,10	,5	,7	,6	,9	,3	,3	,3	,3	,5	,0	,4	,2	,3	,1	,1	,0	,0	,0	,1	,0	,1	,0	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,1	,0	,1	,1	,2	,0	,0	,2	,2	,2	,0	,1	,0	,3	,3	,3	,7	,5	,4	,5	,9	,5	,8	,9	,5	,7	,5	,14	,13	,5	,11	,9	,7	,10	,8	,9	,9	,12	,8	,2	,11	,7	,7	,3	,0	,4	,3	,3	,1	,2	,3	,1	,0	,2	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,1	,0	,0	,2	,0	,3	,2	,2	,1	,3	,3	,5	,6	,1	,5	,7	,5	,6	,5	,6	,5	,11	,9	,4	,10	,4	,9	,9	,9	,14	,13	,10	,7	,6	,8	,10	,10	,8	,7	,7	,9	,8	,2	,4	,2	,2	,1	,3	,2	,1	,1	,1	,1	,0	,0	,0	,0	,1	,0	,0	,0,
			0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,1	,1	,0	,2	,1	,0	,0	,1	,1	,2	,2	,2	,2	,1	,5	,3	,7	,5	,5	,9	,8	,9	,13	,11	,9	,8	,8	,12	,11	,9	,12	,6	,23	,5	,17	,11	,8	,4	,5	,8	,13	,12	,12	,9	,8	,3	,5	,4	,6	,3	,1	,0	,1	,1	,1	,0	,0	,2	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,1	,0	,1	,1	,1	,2	,1	,0	,1	,3	,3	,1	,0	,5	,0	,4	,8	,4	,7	,10	,10	,9	,9	,12	,11	,10	,10	,8	,8	,6	,8	,14	,10	,13	,13	,15	,15	,10	,13	,15	,8	,12	,8	,11	,6	,6	,6	,3	,1	,2	,2	,2	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,2	,1	,0	,2	,0	,0	,3	,1	,2	,1	,1	,6	,2	,7	,3	,5	,5	,6	,11	,11	,13	,8	,8	,13	,12	,17	,9	,15	,8	,6	,10	,13	,17	,16	,14	,12	,11	,10	,9	,12	,8	,4	,9	,5	,7	,7	,4	,0	,1	,1	,2	,1	,3	,1	,0	,0	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,1	,1	,1	,1	,2	,4	,3	,3	,1	,1	,4	,4	,3	,6	,3	,4	,10	,3	,9	,10	,17	,12	,13	,14	,18	,17	,11	,14	,18	,12	,12	,16	,14	,12	,12	,11	,12	,5	,14	,9	,7	,11	,12	,3	,7	,7	,5	,3	,1	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,1	,2	,0	,1	,2	,1	,4	,3	,3	,4	,4	,2	,3	,5	,5	,5	,3	,4	,4	,11	,10	,7	,14	,5	,18	,13	,15	,12	,22	,11	,13	,10	,15	,21	,12	,14	,14	,16	,22	,15	,6	,14	,6	,11	,8	,5	,4	,2	,1	,3	,3	,2	,1	,1	,0	,1	,0	,0	,0	,0	,0	,0,
			0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,3	,0	,1	,2	,1	,5	,6	,3	,6	,4	,4	,5	,10	,7	,9	,13	,10	,12	,16	,14	,21	,21	,18	,12	,16	,11	,11	,5	,20	,24	,14	,21	,11	,15	,20	,14	,17	,11	,9	,7	,7	,9	,6	,12	,2	,3	,1	,3	,0	,2	,0	,0	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,2	,1	,3	,1	,2	,2	,3	,2	,6	,5	,2	,6	,9	,8	,11	,9	,4	,9	,13	,14	,20	,10	,22	,26	,12	,25	,22	,19	,14	,19	,11	,21	,20	,14	,18	,15	,14	,19	,11	,7	,12	,11	,12	,12	,12	,4	,9	,5	,3	,1	,2	,2	,1	,0	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,1	,0	,0	,4	,1	,0	,0	,2	,3	,4	,3	,4	,6	,4	,8	,7	,2	,2	,3	,11	,15	,14	,18	,14	,12	,24	,16	,26	,27	,16	,16	,23	,8	,8	,11	,14	,16	,24	,18	,24	,17	,12	,7	,22	,12	,8	,7	,6	,8	,4	,8	,3	,3	,1	,2	,0	,0	,2	,0	,1	,0	,0,
			0	,0	,1	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,1	,0	,0	,0	,3	,2	,1	,2	,2	,2	,5	,4	,7	,9	,4	,4	,4	,6	,10	,9	,16	,9	,12	,13	,20	,24	,18	,19	,27	,25	,19	,25	,25	,21	,16	,23	,19	,25	,20	,13	,20	,18	,15	,14	,14	,12	,6	,9	,9	,9	,2	,4	,2	,1	,2	,0	,0	,1	,0	,0	,0	,1,
			0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,1	,3	,3	,3	,1	,3	,6	,4	,5	,5	,4	,7	,5	,16	,11	,9	,19	,11	,12	,18	,16	,17	,22	,30	,27	,27	,28	,25	,29	,20	,37	,23	,16	,19	,13	,16	,16	,30	,21	,20	,21	,15	,10	,18	,7	,13	,7	,6	,4	,3	,2	,1	,1	,0	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,3	,1	,3	,4	,3	,3	,4	,4	,9	,10	,8	,8	,14	,9	,16	,6	,11	,11	,19	,21	,22	,29	,24	,21	,16	,27	,30	,31	,26	,36	,35	,26	,24	,20	,34	,23	,24	,19	,17	,19	,18	,17	,12	,6	,5	,4	,5	,7	,2	,2	,3	,2	,2	,2	,0	,0	,0	,1,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,1	,0	,2	,3	,2	,2	,2	,1	,5	,4	,2	,3	,8	,6	,3	,7	,14	,11	,10	,13	,12	,19	,18	,19	,23	,21	,39	,24	,33	,22	,25	,29	,29	,38	,30	,29	,17	,25	,15	,12	,30	,27	,23	,18	,15	,15	,16	,12	,7	,7	,9	,4	,4	,3	,2	,0	,2	,0	,0	,1	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,2	,0	,6	,3	,8	,3	,3	,5	,9	,9	,10	,11	,13	,15	,13	,15	,16	,14	,23	,14	,21	,18	,27	,23	,30	,30	,29	,21	,35	,22	,31	,34	,25	,20	,32	,20	,21	,19	,22	,18	,15	,16	,12	,6	,7	,11	,8	,4	,3	,2	,0	,0	,1	,0	,0	,0	,2,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,1	,1	,1	,1	,3	,2	,10	,6	,4	,6	,10	,12	,10	,11	,13	,13	,13	,20	,23	,17	,24	,22	,26	,18	,25	,40	,28	,29	,42	,37	,35	,33	,42	,39	,30	,23	,25	,25	,19	,22	,16	,19	,14	,15	,14	,6	,11	,5	,3	,1	,2	,1	,2	,1	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,2	,2	,3	,1	,1	,4	,2	,10	,2	,3	,7	,14	,13	,14	,12	,16	,21	,23	,22	,15	,21	,22	,30	,26	,30	,32	,18	,40	,27	,37	,37	,30	,34	,46	,29	,32	,34	,22	,34	,24	,38	,28	,22	,17	,14	,8	,8	,9	,5	,7	,5	,3	,2	,0	,2	,2	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,1	,1	,0	,1	,0	,0	,1	,3	,2	,5	,2	,4	,5	,6	,6	,13	,12	,19	,9	,7	,21	,19	,18	,22	,29	,18	,17	,17	,27	,29	,37	,29	,27	,30	,43	,42	,37	,38	,50	,41	,46	,26	,26	,29	,30	,26	,15	,22	,22	,17	,19	,11	,6	,6	,6	,2	,4	,0	,1	,2	,1	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,2	,3	,0	,6	,8	,10	,19	,19	,13	,24	,17	,21	,16	,23	,23	,27	,23	,26	,33	,26	,47	,49	,38	,52	,39	,40	,40	,43	,34	,35	,40	,35	,36	,25	,27	,25	,25	,23	,16	,15	,16	,9	,7	,7	,9	,8	,4	,4	,1	,2	,0	,0	,1,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,1	,0	,0	,0	,0	,1	,1	,2	,8	,3	,4	,8	,6	,11	,11	,12	,15	,18	,13	,22	,22	,25	,31	,26	,35	,28	,29	,27	,31	,45	,51	,48	,40	,44	,55	,54	,32	,43	,47	,52	,30	,30	,26	,29	,26	,15	,19	,13	,6	,7	,11	,3	,3	,5	,6	,5	,0	,2	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,1	,0	,1	,0	,0	,2	,2	,1	,3	,0	,1	,5	,6	,9	,17	,13	,20	,11	,18	,15	,24	,22	,23	,30	,25	,25	,24	,38	,31	,35	,31	,51	,42	,47	,44	,55	,48	,48	,39	,44	,41	,40	,32	,17	,21	,28	,22	,20	,11	,21	,16	,13	,6	,10	,6	,2	,4	,0	,1	,0	,2,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,2	,1	,0	,2	,3	,2	,4	,6	,5	,7	,10	,13	,15	,13	,22	,24	,21	,16	,20	,28	,30	,30	,32	,47	,29	,34	,37	,37	,43	,53	,46	,49	,49	,38	,38	,51	,36	,61	,34	,22	,21	,23	,26	,18	,19	,20	,12	,15	,3	,7	,9	,2	,1	,0	,2	,1	,0,
			0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,1	,0	,2	,0	,0	,0	,1	,1	,2	,0	,0	,2	,3	,4	,2	,7	,13	,10	,15	,15	,17	,24	,27	,24	,23	,26	,28	,24	,26	,42	,30	,29	,40	,40	,53	,46	,44	,54	,42	,50	,69	,50	,49	,38	,64	,44	,39	,30	,31	,23	,28	,23	,26	,13	,10	,7	,7	,3	,5	,4	,4	,3	,0,
			0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,2	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,1	,2	,2	,2	,2	,6	,4	,7	,9	,13	,19	,8	,17	,25	,29	,33	,36	,35	,41	,45	,39	,34	,40	,42	,43	,43	,51	,50	,42	,40	,45	,62	,56	,71	,54	,52	,49	,45	,27	,21	,26	,24	,21	,18	,12	,13	,9	,7	,5	,3	,1	,0	,1	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,1	,0	,0	,1	,0	,0	,0	,5	,1	,7	,5	,4	,8	,9	,7	,15	,23	,18	,17	,21	,26	,29	,28	,38	,43	,38	,42	,57	,42	,35	,45	,50	,52	,61	,51	,66	,54	,57	,43	,50	,61	,54	,40	,38	,22	,20	,20	,22	,25	,18	,10	,9	,5	,4	,2	,1	,2	,0	,1,
			0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,2	,2	,0	,1	,2	,1	,6	,3	,3	,4	,5	,11	,10	,20	,13	,25	,25	,32	,30	,35	,30	,40	,36	,43	,41	,55	,41	,47	,50	,54	,51	,70	,49	,57	,57	,62	,63	,72	,57	,59	,58	,55	,52	,24	,18	,22	,24	,21	,20	,7	,8	,6	,3	,4	,1	,2	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,2	,0	,0	,0	,0	,1	,1	,1	,3	,2	,2	,9	,9	,13	,16	,21	,27	,29	,35	,34	,28	,35	,39	,37	,45	,74	,52	,41	,57	,48	,61	,51	,79	,65	,77	,62	,73	,60	,51	,45	,49	,48	,34	,39	,20	,17	,15	,19	,8	,12	,12	,8	,4	,4	,3	,0	,4,
			0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,2	,0	,0	,3	,3	,6	,4	,7	,8	,10	,13	,15	,30	,24	,26	,24	,46	,40	,37	,49	,47	,32	,53	,41	,65	,59	,58	,48	,66	,53	,53	,53	,68	,55	,64	,67	,60	,53	,47	,41	,42	,22	,25	,17	,21	,12	,11	,6	,4	,3	,4	,0	,2,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,1	,0	,0	,1	,0	,2	,4	,3	,3	,2	,6	,5	,2	,7	,6	,14	,7	,17	,26	,26	,25	,40	,30	,49	,41	,47	,59	,56	,49	,50	,64	,68	,64	,54	,74	,62	,69	,61	,59	,56	,72	,67	,53	,46	,37	,52	,39	,22	,14	,12	,13	,13	,10	,5	,7	,4	,1	,4,
			0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,1	,0	,0	,2	,0	,0	,2	,1	,1	,3	,1	,2	,3	,2	,3	,5	,7	,11	,13	,14	,13	,19	,18	,37	,32	,37	,38	,38	,55	,49	,57	,48	,54	,74	,62	,57	,57	,68	,61	,73	,60	,66	,69	,54	,61	,74	,61	,53	,42	,44	,33	,32	,21	,22	,13	,6	,10	,5	,9	,0	,1	,1,
			0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,2	,0	,0	,1	,1	,1	,3	,4	,5	,3	,2	,9	,3	,11	,16	,4	,18	,28	,31	,33	,31	,31	,40	,58	,60	,52	,59	,63	,62	,78	,52	,57	,67	,59	,75	,81	,69	,71	,64	,56	,56	,55	,47	,39	,41	,29	,37	,19	,15	,8	,5	,10	,7	,2	,6	,5,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,2	,1	,0	,1	,0	,0	,4	,0	,0	,3	,1	,6	,2	,3	,5	,4	,9	,8	,16	,14	,27	,52	,49	,46	,39	,46	,41	,65	,45	,62	,76	,80	,58	,61	,51	,75	,67	,62	,85	,84	,65	,70	,55	,60	,58	,45	,47	,38	,36	,31	,9	,11	,10	,6	,3	,5	,3	,9)
col.names	<- c("5-9","10-14","15-19","20-24",as.character(seq(25,94)),"95+")
cases	<- matrix(data=v.cases,nrow=41, ncol=75, byrow=TRUE,dimnames=list(NULL,col.names))
if(all.age.groups==FALSE)
	return(apc.data.list(
				response		=cases[,seq(5,69)]	,
				data.format		="PA"			,
				age1			=25				,
				per1			=1967			,
				unit			=1				))
if(all.age.groups==TRUE)
	return(apc.data.list(
				response		=cases			,
				data.format		="PA"			,
				label			="mesothelioma, UK"))
}	#	data.asbestos						

##################################
#	UK Asbestos data updated to 2013
##################################
data.asbestos.2013	<- function(all.age.groups=FALSE)
#	BN, 30 Apr 2016
#
#	Taken from
#	Martinez-Miranda, Nielsen and Nielsen (2016)
#	A simple benchmark for mesothelioma projection for Great Britain.
#	Occupational and Environmental Medicine 73, 561-563.
#
#	Asbestos induced mesothelioma mortality for men
#	update of data from the Health and Safety Executive
{	#	data.asbestos.2013
v.cases	<-c(0	,0	,0	,0	,0	,1	,1	,0	,0	,1	,0	,0	,0	,0	,0	,1	,2	,0	,3	,0	,1	,3	,4	,1	,1	,2	,6	,1	,1	,3	,3	,5	,3	,4	,1	,5	,3	,8	,3	,4	,4	,5	,3	,1	,3	,2	,2	,3	,1	,0	,3	,1	,4	,2	,0	,1	,1	,3	,1	,0	,0	,1	,0	,1	,1	,0	,0	,0	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,1	,3	,2	,2	,3	,0	,1	,1	,1	,0	,3	,1	,4	,6	,8	,3	,6	,3	,10	,6	,7	,6	,4	,2	,6	,5	,8	,4	,1	,0	,2	,1	,3	,1	,1	,1	,2	,1	,0	,0	,0	,0	,1	,0	,1	,0	,0	,0	,1	,0	,0	,0	,0,
			0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,2	,1	,0	,1	,0	,0	,1	,1	,2	,1	,0	,1	,1	,4	,2	,1	,2	,5	,4	,6	,5	,10	,3	,4	,11	,10	,5	,9	,1	,5	,4	,7	,6	,3	,2	,4	,6	,5	,0	,1	,0	,0	,1	,2	,2	,1	,0	,1	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,2	,0	,0	,1	,1	,1	,1	,1	,3	,0	,1	,0	,3	,3	,4	,3	,2	,4	,3	,5	,5	,1	,1	,10	,7	,4	,7	,5	,2	,5	,13	,1	,5	,3	,4	,0	,6	,5	,1	,4	,2	,2	,1	,1	,1	,2	,1	,0	,0	,1	,1	,1	,0	,0	,0	,0	,0	,0	,0,
			1	,0	,0	,0	,0	,0	,1	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,2	,1	,2	,1	,6	,1	,2	,5	,2	,4	,4	,2	,3	,4	,6	,9	,7	,8	,8	,3	,6	,6	,5	,2	,6	,7	,4	,10	,5	,5	,3	,5	,6	,2	,1	,1	,2	,0	,3	,0	,1	,1	,0	,1	,1	,0	,2	,0	,0	,0	,0	,0	,0	,1,
			1	,0	,0	,0	,0	,0	,0	,0	,0	,2	,0	,1	,0	,1	,0	,0	,1	,1	,1	,1	,1	,1	,0	,4	,4	,1	,5	,5	,6	,1	,6	,5	,2	,6	,1	,5	,8	,5	,9	,9	,6	,7	,8	,5	,3	,7	,9	,7	,4	,8	,2	,5	,4	,2	,1	,4	,2	,0	,1	,0	,1	,1	,1	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,2	,0	,0	,0	,2	,0	,1	,3	,4	,3	,5	,2	,6	,2	,5	,3	,4	,4	,11	,3	,5	,10	,10	,3	,6	,11	,7	,8	,6	,6	,4	,9	,10	,7	,5	,2	,3	,2	,0	,4	,0	,0	,2	,2	,0	,2	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0,
			0	,0	,1	,1	,1	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,2	,1	,2	,2	,1	,2	,6	,2	,3	,12	,7	,5	,3	,4	,3	,4	,3	,8	,8	,6	,11	,11	,9	,11	,11	,4	,6	,10	,5	,7	,6	,9	,3	,3	,3	,3	,5	,0	,4	,2	,3	,1	,1	,0	,0	,0	,1	,0	,1	,0	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,1	,0	,0	,1	,0	,1	,1	,2	,0	,0	,2	,2	,2	,0	,1	,0	,3	,3	,3	,7	,5	,4	,5	,9	,5	,8	,9	,5	,7	,5	,14	,13	,5	,11	,9	,7	,10	,8	,9	,9	,12	,8	,2	,11	,7	,7	,3	,0	,4	,3	,3	,1	,2	,3	,1	,0	,2	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,1	,0	,0	,2	,0	,3	,2	,2	,1	,3	,3	,5	,6	,1	,5	,7	,5	,6	,5	,6	,5	,11	,9	,4	,10	,4	,9	,9	,9	,14	,13	,10	,7	,6	,8	,10	,10	,8	,7	,7	,9	,8	,2	,4	,2	,2	,1	,3	,2	,1	,1	,1	,1	,0	,0	,0	,0	,1	,0	,0	,0,
			0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,1	,1	,0	,2	,1	,0	,0	,1	,1	,2	,2	,2	,2	,1	,5	,3	,7	,5	,5	,9	,8	,9	,13	,11	,9	,8	,8	,12	,11	,9	,12	,6	,23	,5	,17	,11	,8	,4	,5	,8	,13	,12	,12	,9	,8	,3	,5	,4	,6	,3	,1	,0	,1	,1	,1	,0	,0	,2	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,1	,0	,1	,1	,1	,2	,1	,0	,1	,3	,3	,1	,0	,5	,0	,4	,8	,4	,7	,10	,10	,9	,9	,12	,11	,10	,10	,8	,8	,6	,8	,14	,10	,13	,13	,15	,15	,10	,13	,15	,8	,12	,8	,11	,6	,6	,6	,3	,1	,2	,2	,2	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,2	,1	,0	,2	,0	,0	,3	,1	,2	,1	,1	,6	,2	,7	,3	,5	,5	,6	,11	,11	,13	,8	,8	,13	,12	,17	,9	,15	,8	,6	,10	,13	,17	,16	,14	,12	,11	,10	,9	,12	,8	,4	,9	,5	,7	,7	,4	,0	,1	,1	,2	,1	,3	,1	,0	,0	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,1	,1	,1	,1	,2	,4	,3	,3	,1	,1	,4	,4	,3	,6	,3	,4	,10	,3	,9	,10	,17	,12	,13	,14	,18	,17	,11	,14	,18	,12	,12	,16	,14	,12	,12	,11	,12	,5	,14	,9	,7	,11	,12	,3	,7	,7	,5	,3	,1	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,1	,2	,0	,1	,2	,1	,4	,3	,3	,4	,4	,2	,3	,5	,5	,5	,3	,4	,4	,11	,10	,7	,14	,5	,18	,13	,15	,12	,22	,11	,13	,10	,15	,21	,12	,14	,14	,16	,22	,15	,6	,14	,6	,11	,8	,5	,4	,2	,1	,3	,3	,2	,1	,1	,0	,1	,0	,0	,0	,0	,0	,0,
			0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,3	,0	,1	,2	,1	,5	,6	,3	,6	,4	,4	,5	,10	,7	,9	,13	,10	,12	,16	,14	,21	,21	,18	,12	,16	,11	,11	,5	,20	,24	,14	,21	,11	,15	,20	,14	,17	,11	,9	,7	,7	,9	,6	,12	,2	,3	,1	,3	,0	,2	,0	,0	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,2	,1	,3	,1	,2	,2	,3	,2	,6	,5	,2	,6	,9	,8	,11	,9	,4	,9	,13	,14	,20	,10	,22	,26	,12	,25	,22	,19	,14	,19	,11	,21	,20	,14	,18	,15	,14	,19	,11	,7	,12	,11	,12	,12	,12	,4	,9	,5	,3	,1	,2	,2	,1	,0	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,1	,0	,0	,4	,1	,0	,0	,2	,3	,4	,3	,4	,6	,4	,8	,7	,2	,2	,3	,11	,15	,14	,18	,14	,12	,24	,16	,26	,27	,16	,16	,23	,8	,8	,11	,14	,16	,24	,18	,24	,17	,12	,7	,22	,12	,8	,7	,6	,8	,4	,8	,3	,3	,1	,2	,0	,0	,2	,0	,1	,0	,0,
			1	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,1	,0	,0	,0	,3	,2	,1	,2	,2	,2	,5	,4	,7	,9	,4	,4	,4	,6	,10	,9	,16	,9	,12	,13	,20	,24	,18	,19	,27	,25	,19	,25	,25	,21	,16	,23	,19	,25	,20	,13	,20	,18	,15	,14	,14	,12	,6	,9	,9	,9	,2	,4	,2	,1	,2	,0	,0	,1	,0	,0	,0	,1,
			0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,1	,3	,3	,3	,1	,3	,6	,4	,5	,5	,4	,7	,5	,16	,11	,9	,19	,11	,12	,18	,16	,17	,22	,30	,27	,27	,28	,25	,29	,20	,37	,23	,16	,19	,13	,16	,16	,30	,21	,20	,21	,15	,10	,18	,7	,13	,7	,6	,4	,3	,2	,1	,1	,0	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,3	,1	,3	,4	,3	,3	,4	,4	,9	,10	,8	,8	,14	,9	,16	,6	,11	,11	,19	,21	,22	,29	,24	,21	,16	,27	,30	,31	,26	,36	,35	,26	,24	,20	,34	,23	,24	,19	,17	,19	,18	,17	,12	,6	,5	,4	,5	,7	,2	,2	,3	,2	,2	,2	,0	,0	,0	,1,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,1	,0	,2	,3	,2	,2	,2	,1	,5	,4	,2	,3	,8	,6	,3	,7	,14	,11	,10	,13	,12	,19	,18	,19	,23	,21	,39	,24	,33	,22	,25	,29	,29	,38	,30	,29	,17	,25	,15	,12	,30	,27	,23	,18	,15	,15	,16	,12	,7	,7	,9	,4	,4	,3	,2	,0	,2	,0	,0	,1	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,2	,0	,6	,3	,8	,3	,3	,5	,9	,9	,10	,11	,13	,15	,13	,15	,16	,14	,23	,14	,21	,18	,27	,23	,30	,30	,29	,21	,35	,22	,31	,34	,25	,20	,32	,20	,21	,19	,22	,18	,15	,16	,12	,6	,7	,11	,8	,4	,3	,2	,0	,0	,1	,0	,0	,0	,2,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,1	,1	,1	,1	,3	,2	,10	,6	,4	,6	,10	,12	,10	,11	,13	,13	,13	,20	,23	,17	,24	,22	,26	,18	,25	,40	,28	,29	,42	,37	,35	,33	,42	,39	,30	,23	,25	,25	,19	,22	,16	,19	,14	,15	,14	,6	,11	,5	,3	,1	,2	,1	,2	,1	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,2	,2	,3	,1	,1	,4	,2	,10	,2	,3	,7	,14	,13	,14	,12	,16	,21	,23	,22	,15	,21	,22	,30	,26	,30	,32	,18	,40	,27	,37	,37	,30	,34	,46	,29	,32	,34	,22	,34	,24	,38	,28	,22	,17	,14	,8	,8	,9	,5	,7	,5	,3	,2	,0	,2	,2	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,1	,1	,0	,1	,0	,0	,1	,3	,2	,5	,2	,4	,5	,6	,6	,13	,12	,19	,9	,7	,21	,19	,18	,22	,29	,18	,17	,17	,27	,29	,37	,29	,27	,30	,43	,42	,37	,38	,50	,41	,46	,26	,26	,29	,30	,26	,15	,22	,22	,17	,19	,11	,6	,6	,6	,2	,4	,0	,1	,2	,1	,0	,0,
			0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,2	,3	,0	,6	,8	,10	,19	,19	,13	,24	,17	,21	,16	,23	,23	,27	,23	,26	,33	,26	,47	,49	,38	,52	,39	,40	,40	,43	,34	,35	,40	,35	,36	,25	,27	,25	,25	,23	,16	,15	,16	,9	,7	,7	,9	,8	,4	,4	,1	,2	,0	,0	,1,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,1	,0	,0	,0	,0	,1	,1	,2	,8	,3	,4	,8	,6	,11	,11	,12	,15	,18	,13	,22	,22	,25	,31	,26	,35	,28	,29	,27	,31	,45	,51	,48	,40	,44	,55	,54	,32	,43	,47	,52	,30	,30	,26	,29	,26	,15	,19	,13	,6	,7	,11	,3	,3	,5	,6	,5	,0	,2	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,1	,0	,1	,0	,0	,2	,2	,1	,3	,0	,1	,5	,6	,9	,17	,13	,20	,11	,18	,15	,24	,22	,23	,30	,25	,25	,24	,38	,31	,35	,31	,51	,42	,47	,44	,55	,48	,48	,39	,44	,41	,40	,32	,17	,21	,28	,22	,20	,11	,21	,16	,13	,6	,10	,6	,2	,4	,0	,1	,0	,2,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,2	,1	,0	,2	,3	,2	,4	,6	,5	,7	,10	,13	,15	,13	,22	,24	,21	,16	,20	,28	,30	,30	,32	,47	,29	,34	,37	,37	,43	,53	,46	,49	,49	,38	,38	,51	,36	,61	,34	,22	,21	,23	,26	,18	,19	,20	,12	,15	,3	,7	,9	,2	,1	,0	,2	,1	,0,
			0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,1	,0	,2	,0	,0	,0	,1	,1	,2	,0	,0	,2	,3	,4	,2	,7	,13	,10	,15	,15	,17	,24	,27	,24	,23	,26	,28	,24	,26	,42	,30	,29	,40	,40	,53	,46	,44	,54	,42	,50	,69	,50	,49	,38	,64	,44	,39	,30	,31	,23	,28	,23	,26	,13	,10	,7	,7	,3	,5	,4	,4	,3	,0,
			0	,0	,0	,0	,1	,0	,0	,0	,2	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,1	,2	,2	,2	,2	,6	,4	,7	,9	,13	,19	,8	,17	,25	,29	,33	,36	,35	,41	,45	,39	,34	,40	,42	,43	,43	,51	,50	,42	,40	,45	,62	,56	,71	,54	,52	,49	,45	,27	,21	,26	,24	,21	,18	,12	,13	,9	,7	,5	,3	,1	,0	,1	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,1	,0	,0	,1	,0	,0	,0	,5	,1	,7	,5	,4	,8	,9	,7	,15	,23	,18	,17	,21	,26	,29	,28	,38	,43	,38	,42	,57	,42	,35	,45	,50	,52	,61	,51	,66	,54	,57	,43	,50	,61	,54	,40	,38	,22	,20	,20	,22	,25	,18	,10	,9	,5	,4	,2	,1	,2	,0	,1,
			0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,2	,2	,0	,1	,2	,1	,6	,3	,3	,4	,5	,11	,10	,20	,13	,25	,25	,32	,30	,35	,30	,40	,36	,43	,41	,55	,41	,47	,50	,54	,51	,70	,49	,57	,57	,62	,63	,72	,57	,59	,58	,55	,52	,24	,18	,22	,24	,21	,20	,7	,8	,5	,3	,4	,1	,2	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,2	,0	,0	,0	,0	,1	,1	,1	,3	,2	,2	,9	,9	,13	,16	,21	,27	,29	,35	,34	,28	,35	,39	,37	,45	,74	,52	,41	,57	,48	,61	,51	,79	,65	,77	,62	,73	,60	,51	,45	,49	,48	,34	,39	,20	,17	,15	,19	,8	,12	,12	,8	,4	,4	,3	,0	,4,
			1	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,2	,0	,0	,3	,3	,6	,4	,7	,8	,10	,13	,15	,30	,24	,26	,24	,46	,40	,37	,48	,47	,32	,53	,41	,65	,59	,58	,48	,66	,53	,53	,53	,68	,55	,64	,67	,60	,53	,47	,41	,43	,22	,25	,17	,21	,12	,11	,6	,4	,3	,4	,0	,2,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,1	,0	,0	,1	,0	,2	,4	,3	,3	,2	,6	,5	,2	,7	,6	,14	,7	,17	,26	,26	,25	,40	,30	,49	,41	,47	,59	,56	,49	,50	,64	,68	,64	,54	,74	,62	,69	,61	,59	,56	,72	,67	,53	,46	,37	,52	,39	,22	,14	,12	,13	,13	,10	,5	,7	,4	,1	,4,
			1	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,1	,0	,0	,2	,0	,0	,2	,1	,1	,3	,1	,2	,3	,2	,3	,5	,7	,11	,13	,14	,13	,19	,18	,37	,32	,37	,38	,38	,55	,49	,57	,48	,54	,74	,62	,57	,57	,69	,61	,74	,60	,66	,69	,54	,61	,74	,61	,53	,42	,44	,33	,32	,21	,22	,13	,6	,10	,5	,9	,0	,1	,1,
			0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,2	,0	,0	,1	,1	,1	,3	,4	,5	,3	,2	,9	,3	,11	,16	,4	,18	,28	,31	,33	,31	,31	,40	,58	,60	,52	,60	,63	,62	,78	,52	,57	,67	,59	,75	,81	,69	,71	,64	,56	,57	,55	,47	,39	,41	,29	,37	,19	,15	,8	,5	,10	,7	,2	,6	,5,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,2	,1	,0	,1	,0	,0	,4	,0	,0	,3	,1	,6	,2	,3	,6	,4	,9	,8	,16	,14	,27	,51	,49	,46	,39	,47	,42	,66	,45	,63	,77	,81	,58	,62	,51	,76	,67	,62	,86	,85	,65	,70	,55	,60	,60	,45	,50	,38	,36	,32	,10	,11	,10	,6	,3	,5	,3	,9,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,2	,1	,1	,1	,0	,3	,3	,2	,4	,0	,4	,7	,9	,7	,10	,9	,23	,26	,26	,34	,52	,37	,40	,62	,55	,59	,60	,56	,81	,61	,83	,86	,73	,69	,83	,86	,70	,71	,63	,56	,73	,62	,47	,39	,47	,44	,31	,16	,8	,11	,4	,4	,6	,3	,8,
			0	,0	,0	,0	,0	,0	,0	,0	,1	,1	,0	,0	,0	,0	,0	,0	,0	,0	,1	,1	,1	,1	,0	,2	,1	,4	,0	,4	,5	,8	,4	,10	,11	,17	,17	,22	,22	,26	,40	,36	,48	,48	,63	,50	,63	,52	,64	,80	,66	,93	,86	,72	,89	,67	,88	,87	,73	,61	,71	,77	,63	,43	,44	,40	,35	,33	,23	,13	,6	,1	,4	,3	,4,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,2	,0	,0	,0	,1	,2	,0	,1	,0	,0	,1	,0	,3	,6	,5	,2	,3	,10	,6	,5	,14	,18	,14	,21	,29	,39	,36	,60	,52	,51	,64	,56	,72	,70	,78	,66	,81	,73	,82	,77	,67	,79	,103,81	,74	,67	,63	,47	,61	,44	,39	,42	,31	,26	,9	,6	,4	,5	,1	,5,
			0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,2	,0	,1	,0	,1	,1	,2	,2	,2	,0	,2	,3	,5	,7	,7	,13	,11	,13	,18	,24	,26	,29	,30	,50	,65	,50	,46	,64	,56	,70	,82	,65	,63	,79	,74	,76	,77	,83	,75	,86	,80	,83	,80	,64	,46	,45	,32	,41	,26	,30	,21	,14	,8	,4	,1	,8,
			0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,2	,1	,0	,0	,0	,1	,3	,2	,4	,0	,6	,7	,9	,7	,18	,8	,22	,18	,18	,26	,47	,52	,66	,65	,78	,64	,78	,83	,74	,81	,101,94	,89	,91	,92	,83	,79	,100,98	,69	,81	,56	,47	,56	,35	,31	,26	,26	,17	,8	,9	,3	,4,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,2	,1	,3	,1	,3	,3	,4	,0	,1	,5	,1	,5	,10	,7	,11	,14	,12	,22	,25	,29	,36	,51	,56	,68	,68	,75	,79	,66	,96	,88	,72	,94	,101,101,79	,76	,90	,92	,83	,82	,74	,57	,49	,47	,45	,38	,27	,28	,14	,13	,8	,5	,6)
col.names	<- c("0-19","20-24",as.character(seq(25,94)),"95+")
cases	<- matrix(data=v.cases,nrow=46, ncol=73, byrow=TRUE,dimnames=list(NULL,col.names))
if(all.age.groups==FALSE)
	return(apc.data.list(
				response		=cases[,seq(3,67)]	,
				data.format		="PA"			,
				age1			=25				,
				per1			=1968			,
				unit			=1				,
				label			="men, UK mesothelioma 2013 update"))
if(all.age.groups==TRUE)
	return(apc.data.list(
				response		=cases			,
				data.format		="PA"			,
				label			="men, UK mesothelioma 2013 update, all age groups"))
}	#	data.asbestos.2013


########################
#	men
data.asbestos.2013.men	<- function(all.age.groups=FALSE)
#	BN, 17 Sep 2016
{	#	data.asbestos.2013.men
	return(data.asbestos.2013(all.age.groups))
}	#	data.asbestos.2013.men

########################
#	women
data.asbestos.2013.women	<- function(all.age.groups=FALSE)
#	BN, 17 Sep 2016
#	Asbestos induced mesothelioma mortality for women
#	update of data from the Health and Safety Executive
{	#	data.asbestos.2013
v.cases	<-c(0	,2	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,2	,0	,0	,0	,1	,1	,2	,1	,2	,1	,1	,2	,1	,0	,0	,1	,2	,0	,0	,1	,0	,0	,0	,4	,1	,4	,1	,1	,1	,1	,1	,0	,1	,0	,1	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0,
			1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,1	,1	,0	,0	,0	,0	,0	,0	,0	,1	,1	,0	,1	,0	,1	,0	,0	,0	,1	,1	,3	,0	,2	,3	,2	,3	,2	,1	,1	,0	,1	,0	,2	,0	,2	,0	,1	,1	,0	,1	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0,
			0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,1	,2	,1	,1	,2	,1	,1	,0	,1	,1	,0	,2	,1	,0	,4	,2	,1	,2	,0	,1	,2	,1	,1	,1	,2	,0	,2	,1	,2	,4	,0	,0	,0	,1	,1	,1	,1	,1	,1	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,1	,2	,1	,1	,1	,1	,1	,3	,1	,1	,0	,4	,1	,1	,2	,0	,0	,2	,0	,3	,0	,2	,0	,1	,1	,0	,0	,0	,3	,0	,0	,1	,0	,1	,0	,1	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,1	,0	,0	,0	,1	,2	,0	,1	,0	,0	,0	,1	,1	,1	,0	,0	,0	,2	,2	,1	,2	,1	,2	,1	,2	,1	,2	,1	,4	,1	,0	,0	,4	,0	,2	,0	,0	,0	,0	,1	,1	,0	,0	,0	,0	,1	,0	,1	,0	,1	,1	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,1	,0	,0	,0	,0	,1	,2	,2	,0	,1	,1	,1	,2	,0	,0	,2	,2	,2	,2	,0	,1	,3	,1	,1	,0	,1	,5	,1	,2	,0	,1	,1	,0	,0	,2	,0	,0	,1	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,1	,1	,0	,0	,0	,0	,0	,1	,1	,0	,0	,0	,3	,1	,1	,2	,2	,3	,0	,3	,2	,1	,2	,0	,2	,1	,1	,0	,4	,5	,3	,3	,3	,1	,1	,1	,2	,1	,1	,1	,1	,0	,1	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0,
			1	,0	,0	,0	,0	,1	,0	,1	,1	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,1	,1	,0	,0	,0	,1	,0	,2	,2	,0	,0	,0	,0	,2	,1	,1	,5	,1	,0	,3	,1	,1	,2	,5	,1	,2	,1	,4	,2	,2	,0	,1	,1	,0	,0	,0	,2	,0	,1	,1	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0,
			1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,1	,1	,3	,1	,2	,2	,0	,2	,3	,2	,0	,0	,2	,0	,2	,5	,3	,1	,4	,2	,1	,5	,0	,3	,1	,2	,1	,1	,1	,0	,0	,0	,1	,0	,0	,0	,0	,1	,0	,1	,0	,0	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,1	,0	,0	,0	,1	,1	,0	,0	,0	,0	,1	,0	,0	,0	,0	,1	,2	,2	,3	,1	,1	,1	,5	,2	,1	,1	,2	,0	,4	,2	,3	,2	,1	,1	,1	,1	,6	,0	,3	,0	,2	,1	,1	,1	,2	,0	,0	,1	,0	,0	,0	,0	,1	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,1	,1	,0	,0	,1	,0	,0	,1	,0	,0	,3	,1	,2	,1	,2	,2	,3	,4	,1	,3	,4	,4	,3	,2	,2	,0	,1	,2	,5	,1	,2	,2	,2	,0	,2	,0	,2	,0	,0	,2	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0,
			0	,0	,1	,0	,0	,0	,1	,0	,2	,0	,0	,0	,0	,0	,1	,0	,0	,0	,1	,1	,0	,1	,0	,1	,0	,1	,0	,1	,3	,1	,0	,2	,3	,2	,4	,3	,0	,3	,2	,5	,6	,2	,5	,2	,6	,2	,4	,0	,2	,3	,6	,2	,1	,2	,0	,3	,1	,1	,0	,2	,1	,1	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,1,
			0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,2	,0	,3	,0	,0	,1	,0	,0	,0	,2	,0	,0	,0	,0	,5	,2	,2	,3	,1	,5	,1	,3	,3	,1	,4	,3	,1	,5	,4	,2	,8	,5	,1	,8	,4	,2	,3	,3	,1	,1	,3	,3	,1	,0	,1	,0	,1	,1	,0	,0	,0	,0	,1	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,1	,1	,1	,0	,2	,0	,0	,1	,1	,0	,0	,0	,1	,1	,3	,1	,5	,1	,2	,2	,0	,3	,1	,2	,4	,3	,2	,0	,4	,0	,6	,2	,1	,3	,1	,2	,2	,4	,0	,2	,2	,0	,3	,1	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,1	,1	,1	,0	,0	,1	,0	,0	,0	,0	,1	,1	,1	,0	,1	,1	,3	,1	,2	,2	,4	,3	,4	,3	,4	,5	,3	,4	,1	,0	,8	,3	,3	,0	,5	,4	,4	,3	,3	,1	,0	,2	,0	,0	,1	,1	,0	,1	,1	,1	,0	,0	,0	,2	,0	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,1	,1	,1	,0	,0	,2	,0	,1	,0	,1	,1	,1	,1	,2	,2	,1	,5	,2	,2	,1	,3	,2	,5	,4	,1	,3	,3	,6	,3	,5	,3	,4	,3	,4	,0	,3	,2	,4	,1	,2	,1	,1	,1	,1	,1	,1	,0	,0	,0	,0	,0	,1	,0	,0,
			0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,1	,0	,1	,1	,1	,0	,0	,0	,0	,0	,2	,1	,1	,0	,0	,1	,0	,1	,0	,0	,1	,1	,1	,5	,4	,3	,3	,6	,5	,3	,6	,3	,3	,4	,3	,5	,1	,3	,3	,0	,2	,2	,1	,0	,1	,1	,1	,1	,1	,0	,0	,0	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,1	,0	,0	,0	,0	,0	,1	,0	,0	,0	,1	,0	,1	,1	,2	,0	,0	,0	,0	,0	,1	,0	,2	,1	,1	,1	,2	,7	,3	,0	,2	,5	,5	,7	,0	,5	,0	,5	,2	,2	,5	,1	,3	,3	,2	,2	,2	,0	,1	,0	,1	,0	,1	,1	,0	,0	,1	,0	,2	,0	,0	,1	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,1	,0	,0	,1	,2	,1	,0	,0	,1	,1	,2	,1	,0	,0	,0	,2	,2	,0	,1	,0	,0	,2	,2	,1	,4	,3	,2	,3	,4	,4	,4	,2	,5	,1	,6	,2	,3	,5	,5	,3	,1	,5	,3	,4	,0	,3	,2	,0	,1	,1	,2	,2	,0	,0	,0	,0	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,1	,1	,0	,2	,1	,0	,1	,0	,0	,0	,0	,1	,1	,1	,3	,4	,3	,2	,2	,7	,1	,4	,1	,4	,2	,5	,3	,4	,4	,3	,8	,5	,4	,2	,5	,2	,2	,4	,0	,4	,0	,2	,2	,1	,0	,0	,0	,1	,0	,0	,1	,1	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,2	,0	,1	,0	,0	,0	,0	,2	,1	,4	,0	,2	,1	,0	,1	,1	,3	,5	,3	,4	,4	,1	,4	,6	,5	,3	,7	,5	,3	,9	,3	,4	,1	,3	,5	,3	,4	,1	,3	,1	,2	,1	,0	,2	,0	,0	,1	,1	,1	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,1	,0	,3	,1	,1	,2	,1	,1	,0	,2	,2	,1	,3	,5	,3	,1	,3	,2	,2	,3	,8	,4	,4	,2	,7	,3	,14	,4	,3	,2	,6	,5	,4	,4	,9	,1	,4	,2	,2	,1	,0	,3	,1	,1	,0	,0	,0	,1	,1	,0	,0	,0	,1,
			0	,0	,0	,0	,0	,0	,0	,1	,1	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,1	,1	,1	,0	,3	,1	,0	,1	,1	,3	,1	,1	,2	,2	,2	,2	,4	,0	,4	,1	,4	,8	,3	,7	,4	,6	,1	,4	,3	,6	,1	,5	,6	,2	,7	,1	,3	,1	,2	,1	,4	,0	,1	,2	,0	,0	,0	,1	,1	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,1	,0	,1	,0	,0	,2	,2	,1	,2	,3	,3	,0	,2	,4	,2	,1	,1	,2	,0	,4	,1	,1	,1	,5	,4	,1	,2	,4	,4	,9	,7	,7	,9	,5	,9	,6	,10	,4	,7	,6	,2	,1	,3	,1	,2	,2	,3	,1	,1	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0,
			0	,1	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,2	,1	,0	,0	,3	,1	,0	,2	,1	,0	,2	,1	,2	,3	,1	,0	,2	,2	,0	,5	,4	,2	,5	,6	,5	,8	,5	,6	,8	,5	,7	,5	,6	,3	,4	,5	,4	,4	,1	,5	,2	,3	,0	,1	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,1	,1	,0	,0	,0	,0	,3	,3	,0	,0	,1	,3	,2	,1	,3	,2	,3	,1	,2	,3	,1	,3	,3	,2	,2	,5	,0	,5	,5	,9	,5	,5	,4	,9	,9	,6	,6	,5	,4	,5	,3	,5	,1	,5	,1	,3	,0	,1	,0	,0	,1	,0	,1	,0	,0	,1	,1,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,2	,0	,0	,0	,2	,2	,3	,2	,1	,2	,0	,1	,2	,2	,4	,3	,1	,1	,1	,5	,5	,5	,6	,5	,2	,4	,6	,8	,9	,3	,9	,7	,4	,3	,4	,8	,7	,6	,2	,1	,0	,3	,4	,3	,1	,2	,0	,0	,1	,0	,0	,0	,0	,0,
			1	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,1	,0	,0	,0	,1	,1	,0	,1	,2	,2	,1	,1	,1	,7	,2	,2	,7	,3	,5	,3	,8	,1	,3	,1	,3	,5	,2	,4	,6	,6	,4	,4	,5	,13	,8	,6	,6	,6	,5	,4	,6	,6	,4	,3	,4	,4	,1	,2	,2	,0	,2	,0	,0	,1	,0	,1	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,1	,1	,1	,1	,0	,1	,2	,2	,1	,4	,2	,1	,4	,4	,2	,2	,1	,2	,2	,4	,4	,2	,3	,6	,9	,4	,7	,5	,5	,9	,8	,6	,6	,3	,2	,4	,7	,6	,2	,5	,3	,4	,1	,3	,1	,0	,1	,0	,0	,1	,1,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,1	,1	,0	,1	,1	,0	,1	,1	,2	,2	,2	,3	,1	,5	,3	,2	,10	,4	,1	,1	,4	,7	,5	,4	,7	,4	,6	,7	,6	,8	,8	,13	,8	,8	,6	,5	,4	,3	,4	,4	,0	,4	,5	,2	,0	,0	,0	,2	,0	,0	,1	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,1	,1	,1	,2	,1	,3	,2	,2	,1	,2	,3	,2	,1	,4	,4	,4	,7	,3	,5	,2	,5	,2	,5	,7	,5	,9	,7	,9	,8	,5	,7	,10	,8	,10	,7	,5	,4	,5	,6	,5	,9	,3	,5	,1	,1	,1	,1	,0	,1	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,1	,0	,0	,2	,1	,1	,2	,2	,4	,2	,2	,4	,4	,7	,3	,2	,8	,7	,3	,6	,7	,7	,4	,5	,3	,10	,11	,3	,10	,7	,12	,13	,6	,10	,10	,10	,7	,7	,3	,4	,7	,3	,3	,1	,1	,2	,0	,0	,0	,0	,0	,1,
			0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,2	,1	,0	,0	,1	,0	,0	,4	,4	,1	,3	,1	,3	,7	,7	,3	,7	,3	,7	,4	,6	,4	,5	,6	,6	,2	,7	,13	,5	,9	,8	,6	,3	,11	,10	,12	,8	,7	,6	,5	,8	,3	,7	,5	,1	,5	,0	,1	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,1	,0	,3	,0	,1	,2	,1	,1	,1	,2	,0	,6	,6	,4	,8	,8	,2	,5	,5	,5	,4	,8	,5	,5	,7	,8	,4	,9	,8	,7	,12	,6	,4	,14	,7	,12	,5	,10	,7	,14	,9	,5	,10	,5	,3	,5	,2	,5	,2	,2	,1	,1	,1	,1,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,1	,0	,0	,0	,1	,0	,0	,0	,0	,2	,2	,0	,1	,3	,1	,4	,5	,4	,8	,6	,7	,2	,7	,10	,4	,4	,10	,3	,9	,12	,7	,6	,3	,14	,10	,12	,9	,13	,7	,8	,11	,14	,6	,12	,10	,6	,4	,7	,3	,4	,3	,5	,0	,1	,1	,2	,0	,1,
			0	,0	,0	,0	,0	,0	,1	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,3	,1	,1	,3	,0	,3	,3	,0	,2	,2	,3	,2	,3	,2	,8	,4	,7	,3	,9	,7	,13	,14	,6	,11	,6	,11	,5	,4	,9	,8	,10	,6	,9	,15	,10	,10	,14	,12	,9	,11	,5	,3	,2	,3	,5	,4	,1	,0	,1	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,1	,0	,0	,0	,2	,1	,2	,1	,1	,2	,1	,1	,0	,6	,3	,0	,13	,9	,9	,5	,11	,10	,5	,12	,10	,5	,8	,5	,8	,7	,13	,6	,6	,16	,8	,13	,13	,11	,14	,11	,8	,10	,4	,5	,7	,5	,0	,2	,2	,2	,2	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,1	,0	,3	,0	,2	,4	,1	,1	,4	,4	,2	,5	,4	,6	,4	,7	,6	,4	,5	,7	,9	,9	,13	,9	,3	,11	,4	,8	,13	,9	,10	,11	,4	,13	,14	,10	,9	,7	,12	,8	,10	,6	,3	,4	,1	,0	,3	,2	,2	,2	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,1	,1	,0	,1	,1	,0	,0	,0	,2	,3	,0	,1	,4	,2	,3	,4	,9	,12	,5	,11	,12	,11	,8	,10	,10	,9	,11	,11	,11	,8	,8	,9	,9	,10	,5	,8	,14	,8	,9	,10	,6	,15	,17	,7	,8	,3	,3	,1	,0	,2	,0	,1	,0	,1,
			0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,1	,1	,0	,1	,1	,0	,1	,1	,0	,1	,1	,1	,1	,2	,1	,2	,3	,2	,4	,4	,6	,6	,8	,6	,8	,9	,11	,7	,2	,15	,7	,11	,14	,6	,9	,13	,14	,17	,19	,13	,15	,11	,10	,5	,7	,13	,10	,11	,12	,5	,5	,3	,5	,1	,1	,0	,2	,1,
			0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,2	,2	,0	,1	,3	,1	,2	,2	,1	,3	,3	,2	,4	,6	,12	,6	,8	,11	,10	,11	,11	,9	,13	,14	,15	,13	,12	,13	,16	,12	,11	,10	,16	,18	,11	,23	,12	,11	,5	,7	,12	,5	,8	,6	,2	,3	,4	,1	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,1	,0	,0	,0	,0	,0	,0	,1	,0	,1	,1	,1	,0	,2	,3	,0	,1	,2	,2	,2	,0	,4	,10	,6	,11	,4	,10	,10	,15	,9	,13	,8	,9	,11	,14	,15	,9	,15	,13	,8	,16	,19	,15	,9	,12	,10	,12	,13	,7	,10	,10	,16	,11	,9	,1	,3	,2	,1	,2	,1,
			1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,1	,1	,0	,2	,1	,1	,3	,4	,2	,1	,4	,0	,7	,7	,3	,9	,4	,6	,11	,10	,7	,10	,14	,16	,12	,8	,13	,18	,19	,17	,12	,15	,9	,12	,12	,9	,19	,14	,4	,14	,12	,10	,19	,7	,8	,6	,2	,2	,2	,2	,1	,2,
			0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,1	,0	,1	,0	,2	,0	,2	,1	,1	,2	,2	,1	,5	,1	,2	,6	,5	,5	,6	,3	,6	,5	,4	,12	,12	,10	,14	,14	,9	,9	,15	,14	,16	,12	,12	,13	,17	,16	,13	,12	,11	,7	,10	,9	,8	,9	,10	,8	,3	,3	,4	,2	,0	,1,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,2	,4	,0	,1	,1	,1	,2	,2	,2	,4	,1	,4	,2	,6	,5	,9	,7	,12	,13	,10	,12	,5	,7	,12	,12	,15	,15	,14	,15	,15	,20	,23	,15	,19	,15	,17	,10	,13	,15	,11	,11	,6	,7	,6	,4	,1	,3	,0	,4,
			0	,0	,0	,0	,0	,0	,1	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,2	,1	,0	,0	,2	,7	,2	,1	,1	,5	,1	,1	,2	,6	,5	,4	,3	,3	,4	,7	,14	,15	,13	,10	,17	,13	,15	,19	,11	,18	,12	,16	,23	,11	,18	,18	,18	,12	,5	,14	,14	,13	,5	,9	,5	,0	,7	,3	,3	,3	,2)
col.names	<- c("0-19","20-24",as.character(seq(25,94)),"95+")
cases	<- matrix(data=v.cases,nrow=46, ncol=73, byrow=TRUE,dimnames=list(NULL,col.names))
if(all.age.groups==FALSE)
	return(apc.data.list(
				response		=cases[,seq(3,67)]	,
				data.format		="PA"			,
				age1			=25				,
				per1			=1968			,
				unit			=1				,
				label			="women, UK mesothelioma 2013 update"))
if(all.age.groups==TRUE)
	return(apc.data.list(
				response		=cases			,
				data.format		="PA"			,
				label			="women, UK mesothelioma 2013 update, all age groups"))
}	#	data.asbestos.2013.women


###############################
#	MOTOR DATA
###############################
data.loss.VNJ	<- function()
#	BN, 24 Apr 2015 (6 Feb 2015)
#	A Chain-Ladder with A,C effects
#
#	Taken from tables 1,2 of
#	Verrall R, Nielsen JP, Jessen AH (2010)
#	Prediction of RBNS and IBNR claims using claim amounts and claim counts
#	ASTIN Bulletin 40, 871-887
#
#	Also analysed in 
#
#	Martinez Miranda MD, Nielsen B, Nielsen JP and Verrall R (2011)
#	Cash flow simulation for a model of outstanding liabilities based on claim amounts and claim numbers
#	ASTIN Bulletin 41, 107-129
#
#	Kuang D, Nielsen B, Nielsen JP (2015)
#	The geometric chain-ladder
#	Scandinavian Acturial Journal, to appear
#
#	Data from Codan, Danish subsiduary of Royal & Sun Alliance 
#	Portfolio of motor policies: third party liability
#	Units in years
#	X Paid run-off triangle
#	N Number of reported claims
{	#	data.loss.VNJ
#	dimension
k		<- 10
#	Number of reported claims
Nvec	<- c(	6238  , 831   , 49    , 7     , 1 , 1 , 2 , 1 , 2 , 3 ,
				7773  , 1381  , 23    , 4     , 1 , 3 , 1 , 1 , 3 ,  
				10306 , 1093  , 17    , 5     , 2 , 0 , 2 , 2 ,    
				9639  , 995   , 17    , 6     , 1 , 5 , 4 ,    
				9511  , 1386  , 39    , 4     , 6 , 5 ,     
				10023 , 1342  , 31    , 16    , 9 ,  
				9834  , 1424  , 59    , 24    ,   
				10899 , 1503  , 84    ,       
				11954 , 1704  ,
				10989 )
#	X	as a vector
Xvec	<- c(   451288 , 339519 , 333371 , 144988 , 93243  , 45511  , 25217 , 20406 , 31482 , 1729 ,
			    448627 , 512882 , 168467 , 130674 , 56044  , 33397  , 56071 , 26522 , 14346 ,     
			    693574 , 497737 , 202272 , 120753 , 125046 , 37154  , 27608 , 17864 ,       
			    652043 , 546406 , 244474 , 200896 , 106802 , 106753 , 63688 ,       
			    566082 , 503970 , 217838 , 145181 , 165519 , 91313  ,      
			    606606 , 562543 , 227374 , 153551 , 132743 ,        
			    536976 , 472525 , 154205 , 150564 ,        
			    554833 , 590880 , 300964 ,          
			    537238 , 701111 ,           
			    684944 )     
return(c(apc.data.list(
			response	=vector.2.triangle(Xvec,k)	,
			data.format	="CL"						,
			time.adjust =1							,
			label		="loss VNJ"					),			
			list(
			counts		=vector.2.triangle(Nvec,k)	)))
}	#	data.loss.VNJ

###############################
#	LOSS TRIANGLE DATA
###############################
data.loss.BZ	<- function()
#	BN, 24 Apr 2014 (7 Feb 2015)
#	A Loss Triangle with A,P,C effects
#
#	Taken from table 3.5 of
#   Barnett, G. and Zehnwirth, B. (2000). Best estimates for reserves.
#	Proc. Casualty Actuar. Soc. 87, 245--321.
#
#	Also analysed in 
#
#	Kuang D, Nielsen B and Nielsen JP (2011)
#	Forecasting in an extended chain-ladder-type model
#	Journal of Risk and Insurance 78, 345-359
#
#	BZ write: "loss development array with a major trend change between payment years 1984 and 1985"
#	Time in years
#	X Paid run-off triangle
{	#	data.loss.BZ
#	dimension
k		<- 11
#	X	as a vector
Xvec	<- c(   153638,	188412,	134534,	 87456,	 60348,	42404,	31238,	21252,	16622,	14440,	12200,
			    178536,	226412,	158894,	104686,	 71448,	47990,	35576,	24818,	22662,	18000,        
			    210172,	259168,	188388,	123074,	 83380,	56086,	38496,	33768,	27400,                
			    211448,	253482,	183370,	131040,	 78994,	60232,	45568,	38000,                        
			    219810,	266304,	194650,	120098,	 87582,	62750,	51000,                                
			    205654,	252746,	177506,	129522,	 96786,	82400,                                        
			    197716,	255408,	194648,	142328,	105600,                                               
			    239784,	329242,	264802,	190400,                                                       
			    326304,	471744,	375400,                                                               
			    420778,	590400,                                                                       
				496200)
Exposure	<- c(     2.2,    2.4,	   2.2,	   2.0,	   1.9,	  1.6,    1.6,    1.8,    2.2,    2.5,    2.6)				
return(c(apc.data.list(
			response	=vector.2.triangle(Xvec,k)	,
			data.format	="CL"						,
			coh1		=1977						,
			time.adjust =1							,
			label		="loss BZ"					),
			exposure	=Exposure					))
}	#	data.loss.BZ

###############################
#	LOSS TRIANGLE DATA
###############################
data.loss.TA	<- function()
#	BN, 24 Apr 2015 (18 Mar 2015)
#	A Loss Triangle with A,C effects and over-dispersion
#
#	Attributed to
#	Taylor and Ashe
#
##	Analysed in
#
#	Verrall, R.J. (1991)
#	On the estimation of reserves from loglinear models
#	Insurance: Mathematics and Economics 10, 75-80
#
#	England, P., Verrall, R.J. (1999)
#	Analytic and bootstrap estimates of prediction errors in claims reserving
#	Insurance: Mathematics and Economics 25, 281-293
#
#	X Paid run-off triangle
{	#	data.loss.BZ
#	dimension
k		<- 10
#	X	as a vector
Xvec	<- c(	357848,	766940, 610542, 482940, 527326, 574398, 146342, 139950, 227229,  67948,
				352118, 884021, 933894,1183289, 445745, 320996, 527804, 266172, 425046,
				290507,1001799, 926219,1016654, 750816, 146923, 495992, 280405,
				310608,1108250, 776189,1562400, 272482, 352053, 206286,
				443160, 693190, 991983, 769488, 504851, 470639,
				396132, 937085, 847498, 805037, 705960,
				440832, 847631,1131398,1063269,
				359480,1061648,1443370,
				376686, 986608,
				344014)
return(apc.data.list(
			response	=vector.2.triangle(Xvec,k)	,
			data.format	="CL"						,
			time.adjust =1							,
			label		="loss TA"					))
}	#	data.loss.TA

################################
##	LOSS TRIANGLE DATA
################################
#data.loss.Greek	<- function()
##	BN, 29 jan 2018
##	A Loss Triangle with A,C effects, over-dispersion
##	Paid and incurred
##
##	Used and analysed in
##
##	Margraf, C. and Nielsen, B.
##	A likelihood approach to Bornhuetter-Ferguson Analysis.
##	mimeo, Nuffield College
##
##	amounts in Euros
##	
##	Paid run-off triangle, cumulative
##	Incurred run-off triangle, cumulative
#{	#	data.loss.Greek
##	dimension
#k		<- 9
##	Paid and incurred as vectors
#Paid.vec<- c(	34492471, 47124007,   55244404,   59817460,   62550940,   66042036,   69311560,   70992659,  72265079,
#				39467733, 54003286,   61349336,   69986825,   76412887,   81768759,   86684598,   90726054,          
#				38928855, 57087550,   65905902,   77128507,   84158380,   92436441,   97838371,                     
#				34202332, 50932726,   60560484,   68566905,   76409739,   82082804,                                
#				35657409, 52397264,   59849582,   66698806,   72724524,                                           
#				25404394, 37040589,   42371049,   50709319,                                                      
#				21268516, 31311410,   35973015,                                                                 
#				17404447, 27786399,                                                                             
#				17676374)                                                                                       
#Incu.vec<- c(	54018141, 56699807,   60273204,   61112600,   63729660,   67142341,   69733859,   71980196,   72738376,
#				68706483, 70534436,   70254136,   75919965,   77900147,   83401774,   88690144,   92171660,           
#				64613205, 72600950,   76163387,   82388057,   87424383,   96246891,  102854340,                       
#				58071632, 66701421,   69420629,   75280537,   81978240,   89923269,                                   
#				60368719, 67868349,   72528239,   80726223,   85339588,                                               
#				47282519, 56488940,   60896832,   65900623,                                                           
#				49905225, 54801141,   60026903,                                                                       
#				48425940, 52652928,                                                                                   
#				47449977)
##	Paid and incurred as matrices				
#Paid.mat.cum	<- vector.2.triangle(Paid.vec,k)					
#Incu.mat.cum	<- vector.2.triangle(Incu.vec,k)
##	Get incrementat triangles
#Paid.mat.inc	<- Paid.mat.cum
#Incu.mat.inc	<- Incu.mat.cum
#for(col in k:2)
#{
#	Paid.mat.inc[,col]	<- Paid.mat.inc[,col]-Paid.mat.inc[,col-1]
#	Incu.mat.inc[,col]	<- Incu.mat.inc[,col]-Incu.mat.inc[,col-1]
#}
#
#
#return(c(apc.data.list(
#			response		=Paid.mat.inc			,
#			data.format		="CL"					,
#			coh1			=2005					,
#			time.adjust 	=0						,
#			label			="loss Greek"			),
#		list(
#			paid			=Paid.mat.inc			,
#			incurred		=Incu.mat.inc			,
#			response.cum	=Paid.mat.cum			,
#			incurred.cum	=Incu.mat.cum			)
#			))
#}	#	data.loss.Greek


###############################
#	LOSS TRIANGLE DATA
###############################
data.loss.XL	<- function()
#	BN, 5 feb 2018
#	A Loss Triangle with A,C effects, log normal
#	Paid
#
#	Used and analysed in
#
#	Kuang, D. and Nielsen, B.
#	Generalized log-normal chain-ladder.
#	mimeo, Nuffield College
#
#	amounts in 1000 USD
#	
#	Paid run-off triangle, cumulative
#	Incurred run-off triangle, cumulative
{	#	data.loss.XL
#	dimension
k		<- 20
#	Paid and incurred as vectors
Paid.vec<- c(  2185 ,  13908 , 44704 , 56445 , 67313 , 62830 , 72619 , 42511 , 32246 , 51257 , 11774 , 21726 , 10926 , 4763  , 3580  , 4777  , 1070 ,  1807  , 824   , 1288 ,            
			   3004 ,  17478 , 49564 , 55090 , 75119 , 66759 , 76212 , 62311 , 31510 , 15483 , 23970 , 8321  , 15027 , 3247  , 8756  , 14364 , 3967 ,  3858  , 4643  ,                  
			   5690 ,  28971 , 55352 , 63830 , 71528 , 73549 , 72159 , 37275 , 38797 , 27264 , 28651 , 14102 , 8061  , 17292 , 10850 , 10732 , 4611 ,  4608  ,                         
			   9035 ,  29666 , 47086 , 41100 , 58533 , 80538 , 70521 , 40192 , 27613 , 13791 , 17738 , 20259 , 12123 , 6473  , 3922  , 3825  , 3082 ,                                 
			   7924 ,  38961 , 41069 , 64760 , 64069 , 61135 , 62109 , 52702 , 36100 , 18648 , 32572 , 17751 , 18347 , 10895 , 2974  , 5828  ,                                       
			   7285 ,  25867 , 44375 , 58199 , 61245 , 48661 , 57238 , 29667 , 34557 , 8560  , 12604 , 8683  , 9660  , 4687  , 1889  ,                                             
			   3017 ,  22966 , 62909 , 54143 , 72216 , 58050 , 29522 , 25245 , 19974 , 16039 , 8083  , 9594  , 3291  , 2016  ,                                                   
			   1752 ,  25338 , 56419 , 75381 , 64677 , 58121 , 38339 , 21342 , 14446 , 13459 , 6364  , 6326  , 6185  ,                                                           
			   1181 ,  24571 , 66321 , 65515 , 62151 , 43727 , 29785 , 23981 , 12365 , 12704 , 12451 , 8272  ,                                                               
			   1706 ,  13203 , 40759 , 57844 , 48205 , 50461 , 27801 , 21222 , 14449 , 10876 , 8979  ,                                                                    
			   623  ,  14485 , 27715 , 52243 , 60190 , 45100 , 31092 , 22731 , 19950 , 18016 ,                       
			   338  ,  6254  , 24473 , 32314 , 35698 , 25849 , 30407 , 15335 , 15697 ,                                     
			   255  ,  3842  , 14086 , 26177 , 27713 , 15087 , 17085 , 12520 ,                                         
			   258  ,  7426  , 22459 , 28665 , 32847 , 28479 , 24096 ,                                               
			   1139 ,  10300 , 19750 , 32722 , 41701 , 29904 ,                                                    
			   381  ,  5671  , 34139 , 33735 , 33191 ,                                                           
			   605  ,  11242 , 24025 , 32777 ,                                                               
			   1091 ,  9970  , 31410 ,                                         
			   1221 ,  8374  ,                                               
			   2458 )                                                        
#	Paid as matrices				
Paid.mat	<- vector.2.triangle(Paid.vec,k)					
return(apc.data.list(
			response		=Paid.mat				,
			data.format		="CL"					,
			age1			=1997					,
			coh1			=1997					,
			time.adjust 	=1997					,
			label			="loss, US casualty, XL Group"	)
			)
}	#	data.loss.XL


###############################
#	AIDS reports in England and Wales
###############################
data.aids	<- function(all.age.groups=FALSE)
#	BN, 7 Feb 2016
#	Numbers of AIDS reports in England and Wales to the end of 1992 by quarter
#	
#	Attributed to
#	De Angelis and Gilks (1994)
#
#	Analysed in
#
#	Davison, A.C. and Hinkley, D.V. (1997) Bootstrap methods and their applications, Cambridge UP
{	#	data.aids
	#	data for coh 1983:3 to 1989:1
	v.cases		<- c(	2,	6,	0,	1,	1,	0,	0,	1,	0,	0,	0,	0,	0,	0,	1,
						2,	7,	1,	1,	1,	0,	0,	0,	0,	0,	0,	0,	0,	0,	0,
						4,	4,	0,	1,	0,	2,	0,	0,	0,	0,	2,	1,	0,	0,	0,
						0, 10,	0,	1,	1,	0,	0,	0,	1,	1,	1,	0,	0,	0,	0,
						6, 17,	3,	1,	1,	0,	0, 	0,	0,	0,	0,	1,	0,	0,	1,
						5, 22, 	1,	5,	2,	1,	0,	2,	1,	0,	0,	0,	0,	0,	0,
						4, 23, 	4,	5, 	2,	1,	3,	0,	1,	2,	0,	0,	0,	0,	2,
					   11, 11, 	6,	1,	1,	5,	0,	1,	1,	1,	1,	0,	0,	0,	1,
					    9, 22,	6,	2,	4,	3,	3,	4,	7,	1,	2,	0,	0,	0,	0,
						2, 28,	8,	8,	5,	2,	2,	4,	3,	0,	1,	1,	0,	0,	1,
						5, 26, 14,	6,	9,	2,	5,	5,	5,	1,	2,	0,	0,	0,	2,
						7, 49, 17, 11, 	4, 	7, 	5, 	7,	3,	1,	2,	2, 	0,	1,	4,
					   13, 37, 21,	9,	3,	5, 	7,	3,	1,	3,	1,	0,	0,	0,	6,
					   12, 53, 16, 21,	2,	7,	0,	7,	0,	0,	0,	0,	0,	1,	1,
					   21, 44, 29, 11,	6,	4,	2,	2,	1,	0,	2,	0,	2,	2, 	8,
					   17, 74, 13, 13,	3,	5,	3,	1,	2,	2,	0,	0,	0,	3,	5,
					   36, 58, 23, 14,  7,	4,	1,	2,	1,	3,	0,	0,	0,	3,	1,
					   28, 74, 23, 11,	8,	3,	3,	6,	2,	5,	4,	1,	1,	1,	3,
					   31, 80, 16,	9,	3,	2,	8,	3,	1,	4,	6,	2,	1,	2,	6,
					   26, 99, 27, 	9,	8, 11,	3,	4,	6,	3,	5,	5,	1,	1, 	3,
					   31, 95, 35, 13, 18,	4,	6,	4,	4,	3,	3,	2,	0,	3,	3,
					   36, 77, 20, 26, 11,	3,	8,	4,	8,	7,	1,	0,	0,	2,	2,
					   32, 92, 32, 10, 12, 19, 12,	4,	3,	2, 	0,	2,	2, 	0,	2,
					   15, 92, 14, 27, 22, 21, 12,	5,	3,	0,	3,	3,	0,	1,	1,
					   34,104, 29, 31, 18,	8,	6,	7, 	3,	8,	0,	2,	1,	2, NA, 
					   38,101, 34, 18, 	9, 15, 	6,	1,	2,	2,	2,	3,	2, NA, NA, 
					   31,124, 47, 24, 11, 15,	8,	6,	5,	3,	3,	4, NA, NA, NA, 
					   32,132, 36, 10,	9,	7,	6,	4,	4,	5,	0, NA, NA, NA, NA, 
					   49,107, 51, 17, 15,  8,	9,	2,	1,	1, NA, NA, NA, NA, NA, 
					   44,153, 41, 16, 11,	6,	5,	7,	2, NA, NA, NA, NA, NA, NA, 
					   41,137, 29, 33,	7, 11,	6,	4,	3, NA, NA, NA, NA, NA, NA, 
					   56,124, 39, 14, 12, 	7, 10,	1, NA, NA, NA, NA, NA, NA, NA, 
					   53,175, 35, 17, 13, 11, 	2, NA, NA, NA, NA, NA, NA, NA, NA, 
					   63,135, 24, 23, 12,	1, NA, NA, NA, NA, NA, NA, NA, NA, NA, 
					   71,161, 48, 25, 	5, NA, NA, NA, NA, NA, NA, NA, NA, NA, NA, 
					   95,178, 39,	6, NA, NA, NA, NA, NA, NA, NA, NA, NA, NA, NA, 
					   76,181, 16, NA, NA, NA, NA, NA, NA, NA, NA, NA, NA, NA, NA,
					   67, 66, NA, NA, NA, NA, NA, NA, NA, NA, NA, NA, NA, NA, NA)
col.names	<- c("0*",as.character(seq(1,13)),"14+")
cases.all	<- matrix(data=v.cases,nrow=38, ncol=15, byrow=TRUE,dimnames=list(NULL,col.names))
cases.clean	<- cases.all
for(row in 0:7)
	cases.clean[38-row,2+row]	<- NA
if(all.age.groups==FALSE)
	return(apc.data.list(
				response		=t(cases.clean)		,
				data.format		="trap"				,
				age1			=0					,
				coh1			=1983.5				,
				unit			=1/4				,
				per.zero		=0					,
				per.max			=38					,
				label			="UK AIDS - clean"	,
				))
if(all.age.groups==TRUE)
	return(apc.data.list(
				response		=cases.all			,
				data.format		="CA"				,
				age1			=0					,
				coh1			=1983.5				,
				unit			=1/4				,
				label			="UK AIDS - all: last column reporting delay >= 14, last diagonal: incomplete count",
				))	
}	#	data.aids
# 	apc.fit.table(data.Aids(),"od.poisson.response")
# 	fit <- apc.fit.model(data.Aids(),"poisson.response","AC")
#	forecast <- apc.forecast.ac(fit)
#	data.sums.coh <- apc.data.sums(data.Aids())$sums.coh
#	forecast.total <- forecast$response.forecast.coh
#	forecast.total[,1]	<- forecast.total[,1]+data.sums.coh[26:38]
#	plot(seq(1983.5,1992.75,by=1/4),data.sums.coh,xlim=c(1988,1993),ylim=c(200,600),main="Davison, Hinkley, Fig 7.6, parametric version")
#	apc.polygon(forecast.total,x.origin=1989.5,unit=1/4)


###############################
#	2-SAMPLE DATA from Riebler and Held 2010
###############################
#	BN, 17 September 2016
#	from Riebler and Held 2010

data.RH.mortality.dk	<- function()
{	#	data.RH.mortality.dk
v.dk.counts <- c(
3890 ,	3201 ,	2182 ,	1521 ,	1118 ,	1193 ,	1092 ,	895	 ,
272	 ,	309	 ,  311	 ,  241	 ,  158	 ,  119	 ,  105	 ,  74	 ,
201	 ,	246	 ,  227	 ,  193	 ,  186	 ,  128	 ,  108	 ,  95	 ,
373	 ,	390	 ,  372	 ,  336	 ,  331	 ,  282	 ,  240	 ,  185	 ,
335	 ,	391	 ,  429	 ,  387	 ,  374	 ,  375	 ,  308	 ,  267	 ,
423	 ,	424	 ,  483	 ,  462	 ,  488	 ,  434	 ,  415	 ,  361	 ,
672	 ,	580	 ,  614	 ,  719	 ,  748	 ,  663	 ,  555	 ,  585	 ,
1101 ,	1036 ,  901	 ,  892	 ,  1183 ,  1062 ,  996	 ,  942	 ,
1675 ,	1708 ,  1579 ,  1464 ,  1537 ,  1891 ,  1690 ,  1582 ,
2420 ,	2555 ,  2612 ,  2360 ,  2382 ,  2435 ,  2806 ,  2552 ,
3652 ,	3666 ,  3954 ,  3736 ,  3605 ,  3621 ,  3660 ,  4066 ,
5215 ,	5486 ,  5349 ,  5475 ,  5475 ,  5447 ,  5175 ,  5037 ,
7475 ,	7786 ,  8032 ,  7641 ,  8059 ,  8037 ,  7666 ,  7433 ,
10744, 	11341,	11028,	11042,	10928,	11419,	11504,	11021,
14707, 	15671,	15278,	15431,	15989,	15594,	16452,	15779,
18286,	18431,	19037,	20005,	20838,	21795,	21649,	21481,
17340,	18713,	19073,	21094,	24073,	25564,	27956,	26699)

v.dk.pop <- c(
909800 ,984000,	902510,	846103,	710529,	661516,	758503 ,837141 ,
901700 ,911300,	984994,	905633,	850189,	715153,	671309 ,775431 ,
956100 ,903300,	912269,	986488,	908632,	852904,	723344 ,686158 ,
1006100,954200,	904464,	914379,	990757,	914700,	862084 ,738989 ,
797500 ,996200,	957068,	909483,	920173,	997808,	928639 ,892000 ,
713900 ,792000,	992707,	954616,	908331,	922223,	1004548,954111 ,
709700 ,712400,	792252,	990075,	952219,	907818,	926494 ,1021125,
760700 ,707000,	711888,	790006,	986867,	950211,	909956 ,939409 ,
779000 ,756200,	702990,	707044,	785013,	981000,	947270 ,914657 ,
756200 ,768100,	746815,	693291,	697918,	775923,	971539 ,943327 ,
756500 ,741500,	752512,	730916,	678757,	683613,	762084 ,959231 ,
685900 ,733700,	719039,	729062,	707870,	656576,	662825 ,744259 ,
608400 ,655500,	701139,	686911,	695512,	674532,	626303 ,635704 ,
507700 ,562200,	609373,	654229,	640884,	647717,	628252 ,582828 ,
386200 ,441800,	496205,	543840,	586717,	575366,	580188 ,561017 ,
268600 ,301300,	356068,	408000,	453701,	492961,	483697 ,486525 ,
145500 ,173400,	207032,	253754,	297238,	336778,	369061 ,361484 )

col.names	<- c(paste(as.character(seq(1960,1995,by=5)),as.character(seq(1964,1999,by=5)),sep="-"))
row.names	<- c(paste(as.character(seq(0,80,by=5)),as.character(seq(4,84,by=5)),sep="-"))

return(apc.data.list(
			response 	= matrix(data=v.dk.counts, nrow =17, ncol=8, byrow=TRUE, dimnames=list(row.names,col.names)) ,
			dose 		= matrix(data=v.dk.pop	 , nrow =17, ncol=8, byrow=TRUE, dimnames=list(row.names,col.names)) ,
			data.format	="AP"			,
			age1		=0				,
			per1		=1960			,
			unit		=5				,
			label		="RH mortality Denmark"))
}	#	data.RH.mortality.dk

data.RH.mortality.no	<- function()
{	#	data.RH.mortality.no
v.no.counts <- c(
2840 ,	2557 ,	1878 ,	1354 ,	1093 ,	1127 ,	950  ,	647  ,
246	 ,  228	 ,  242	 ,  184	 ,  116	 ,  102	 ,  106	 ,  83	 ,
175	 ,  174	 ,  162	 ,  171	 ,  128	 ,  95	 ,  85	 ,  73	 ,
230	 ,  280	 ,  247	 ,  256	 ,  261	 ,  251	 ,  205	 ,  199	 ,
209	 ,  254	 ,  247	 ,  273	 ,  229	 ,  276	 ,  234	 ,  209	 ,
214	 ,  219	 ,  289	 ,  289	 ,  254	 ,  286	 ,  303	 ,  281	 ,
355	 ,  267	 ,  320	 ,  350	 ,  360	 ,  403	 ,  410	 ,  400	 ,
627	 ,  509	 ,  438	 ,  416	 ,  563	 ,  558	 ,  566	 ,  559	 ,
1027 ,  913	 ,  757	 ,  627	 ,  658	 ,  910	 ,  912	 ,  899	 ,
1500 ,  1576 ,  1398 ,  1110 ,  982	 ,  1142 ,  1369 ,  1382 ,
2172 ,  2213 ,  2262 ,  2116 ,  1665 ,  1546 ,  1575 ,  2129 ,
3338 ,  3363 ,  3278 ,  3386 ,  3005 ,  2668 ,  2256 ,  2335 ,
5162 ,  5207 ,  4909 ,  4943 ,  4893 ,  4543 ,  3869 ,  3262 ,
7574 ,  7985 ,  7666 ,  7668 ,  7272 ,  7340 ,  6686 ,  5452 ,
10690,	11575,	12336,	11289,	11184,	11361,	11279,	9669 ,
13580,	14392,	15794,	16700,	16163,	16759,	16829,	16135,
14362,	14942,	16533,	18298,	19524,	20959,	22093,	22035)

v.no.pop <- c(
748876,	783556,	789040,	692243,	623821,	639147,	715764,	736098,
746805,	748073,	785553,	795060,	698077,	631011,	642790,	726749,
746991,	748742,	749658,	789153,	798502,	702194,	639877,	650683,
712240,	741735,	749907,	751872,	793545,	804705,	721477,	650225,
545465,	719251,	736965,	750894,	757556,	804345,	817266,	738758,
488599,	543814,	713489,	740375,	756612,	769602,	810871,	838049,
529956,	482970,	542226,	716570,	743565,	764545,	776659,	824076,
603561,	521154,	481973,	544204,	717998,	746045,	763994,	783355,
648664,	589043,	518559,	481586,	543408,	717093,	747004,	766289,
613973,	646268,	584627,	515283,	478966,	540241,	695692,	745169,
581830,	605701,	637047,	577247,	509106,	472977,	524105,	688829,
536976,	571711,	592874,	623648,	565379,	498482,	464017,	515832,
490135,	518834,	552400,	573254,	603628,	546792,	488078,	451985,
408079,	462531,	489158,	522196,	543438,	573257,	528830,	466228,
310512,	367706,	414868,	443133,	476199,	498316,	524441,	489160,
219036,	252507,	299993,	343106,	374704,	407047,	426695,	456943,
129532,	148848,	174326,	212774,	251131,	280834,	306362,	330722)

col.names	<- c(paste(as.character(seq(1960,1995,by=5)),as.character(seq(1964,1999,by=5)),sep="-"))
row.names	<- c(paste(as.character(seq(0,80,by=5)),as.character(seq(4,84,by=5)),sep="-"))

return(apc.data.list(
			response 	= matrix(data=v.no.counts, nrow =17, ncol=8, byrow=TRUE, dimnames=list(row.names,col.names)) ,
			dose 		= matrix(data=v.no.pop	 , nrow =17, ncol=8, byrow=TRUE, dimnames=list(row.names,col.names)) ,
			data.format	="AP"			,
			age1		=0				,
			per1		=1960			,
			unit		=5				,
			label		="RH mortality Norway"))
}	#	data.RH.mortality.no

###############################
#	2-SAMPLE DATA from Riebler, Held, Rue and Bopp 2012
###############################
#	BN, 3 July 2025
# data for Swiss suicides
data.Swiss.suicides	<- function()
{	#	data.RHRB.swiss
unit<- 1
G	  <- 5	
H	  <- 1	
age1<- 15
per1<- 1950
AG  <- 13
PH  <- 58
col.names	<- 1950:2007
row.names	<- c(paste(as.character(seq(15,75,by=5)),as.character(seq(19,79,by=5)),sep="-"))
v.suicides.females <- c(
15,	 9,	 9,	14,	 8,	 6,	 8,	 8,	 9,	11,	 8,	12,	10,	13,	 6,	11,	11,	10,	10,	 6,	10,	11,	 8,	 7,	14,	17,	 9,	20,	19,	23,	17,	14,	 8,	11,	 9,	13,	 9,	 8,	 9,	11,	10,	 8,	 6,	 5,	 6,	14,	 7,	 8,	 9,	12,	 8,	13,	10,	10,	11,	11,	 8,	14,
23,	19,	17,	14,	23,	16,	15,	 9,	19,	12,	21,	18,	22,	17,	20,	17,	21,	19,	16,	19,	14,	20,	22,	21,	20,	30,	22,	23,	35,	41,	43,	36,	29,	28,	30,	35,	33,	32,	25,	28,	19,	20,	22,	16,	16,	14,	16,	15,	14,	 9,	12,	 7,	18,	11,	10,	12,	15,	22,
26,	24,	16,	23,	20,	16,	19,	25,	17,	24,	26,	15,	14,	22,	20,	20,	21,	27,	19,	27,	21,	23,	29,	17,	28,	31,	30,	38,	27,	27,	29,	37,	37,	29,	30,	33,	37,	30,	18,	23,	27,	17,	17,	21,	23,	22,	24,	16,	16,	18,	23,	16,	15,	11,	13,	16,	15,	17,
16,	16,	23,	12,	22,	16,	19,	20,	25,	17,	20,	26,	20,	26,	18,	21,	15,	19,	13,	22,	24,	26,	25,	32,	31,	33,	41,	39,	33,	39,	43,	37,	37,	28,	35,	34,	33,	29,	33,	38,	31,	29,	23,	37,	29,	23,	26,	27,	14,	19,	24,	19,	16,	17,	21,	24,	18,	15,
21,	25,	27,	15,	18,	15,	26,	26,	22,	23,	21,	20,	22,	17,	23,	23,	24,	16,	20,	20,	18,	26,	32,	15,	36,	25,	36,	34,	39,	45,	44,	41,	45,	42,	39,	46,	27,	28,	40,	38,	31,	22,	34,	34,	32,	28,	36,	33,	32,	20,	29,	26,	36,	22,	23,	19,	22,	15,
37,	27,	21,	18,	24,	28,	16,	18,	24,	13,	18,	26,	32,	27,	28,	24,	31,	26,	29,	26,	15,	38,	21,	30,	35,	40,	32,	29,	36,	37,	31,	38,	36,	36,	41,	46,	40,	40,	28,	42,	33,	34,	32,	38,	54,	28,	35,	24,	26,	33,	35,	36,	36,	22,	26,	38,	31,	34,
43,	37,	22,	27,	32,	50,	30,	23,	34,	25,	30,	30,	26,	25,	24,	28,	22,	16,	22,	29,	40,	33,	31,	29,	32,	28,	37,	42,	45,	49,	46,	40,	38,	37,	41,	50,	26,	33,	39,	40,	41,	27,	33,	47,	37,	30,	34,	26,	30,	36,	26,	23,	38,	32,	33,	37,	44,	30,
33,	26,	29,	33,	31,	54,	32,	42,	33,	32,	37,	38,	31,	24,	25,	28,	28,	27,	38,	22,	28,	32,	43,	40,	34,	34,	36,	51,	39,	41,	37,	34,	44,	43,	42,	34,	47,	28,	33,	35,	43,	47,	30,	34,	36,	44,	45,	34,	41,	36,	33,	39,	40,	40,	39,	29,	32,	32,
37,	38,	30,	24,	34,	38,	46,	28,	39,	23,	41,	36,	36,	30,	29,	31,	35,	31,	27,	35,	37,	31,	37,	39,	35,	34,	32,	35,	35,	53,	42,	39,	36,	45,	48,	54,	41,	32,	34,	33,	35,	28,	28,	34,	37,	32,	31,	33,	31,	29,	33,	44,	46,	46,	48,	26,	41,	44,
24,	23,	26,	28,	30,	28,	31,	25,	33,	28,	25,	23,	17,	20,	27,	27,	26,	28,	30,	23,	29,	37,	28,	28,	34,	35,	28,	35,	36,	29,	26,	31,	47,	36,	29,	34,	32,	35,	37,	35,	24,	36,	32,	26,	29,	36,	39,	23,	21,	16,	37,	32,	46,	33,	31,	32,	30,	30,
14,	12,	23,	19,	22,	14,	19,	30,	13,	18,	15,	20,	17,	19,	22,	23,	26,	26,	36,	32,	35,	28,	21,	29,	31,	48,	44,	34,	43,	38,	37,	36,	37,	29,	31,	41,	28,	39,	27,	37,	41,	37,	26,	27,	29,	39,	30,	21,	26,	30,	26,	20,	28,	24,	38,	31,	33,	31,
13,	18,	18,	13,	17,	22,	28,	 9,	14,	14,	16,	16,	22,	16,	22,	15,	28,	16,	15,	27,	24,	28,	22,	21,	43,	26,	28,	31,	39,	32,	44,	36,	42,	36,	40,	39,	34,	47,	37,	23,	27,	31,	30,	24,	29,	37,	24,	25,	28,	33,	32,	28,	27,	35,	28,	30,	44,	33,
 7,	 9,	12,	 9,	14,	12,	11,	 7,	 9,	11,	16,	13,	12,	11,	15,	 6,	18,	10,	 9,	10,	11,	13,	23,	15,	16,	18,	22,	31,	11,	24,	25,	22,	21,	27,	23,	20,	22,	35,	29,	21,	34,	26,	30,	24,	31,	17,	24,	30,	23,	24,	26,	21,	36,	33,	40,	35,	33,	36)
v.suicides.males <- c(
20,	27,	27,	28,	28,	20,	36,	30,	25,	29,	23,	21,	35,	38,	41,	23,	30,	41,	34,	25,	29,	26,	37,	32,	31,	43,	39,	48,	47,	43,	59,	42,	67,	53,	41,	42,	45,	39,	37,	40,	37,	20,	22,	23,	37,	32,	30,	33,	25,	26,	27,	34,	18,	26,	18,	18,	29,	25,
53,	54,	56,	59,	47,	47,	57,	61,	65,	58,	70,	57,	61,	50,	61,	67,	63,	70,	64,	65,	82,	87,	76,	69,	84,106,	97,	95,111,110,112,126,129,117,133,	99,105,101,	94,	97,	81,101,	83,	88,	95,	72,	75,	64,	67,	59,	52,	61,	59,	53,	46,	51,	35,	49,
49,	39,	51,	56,	66,	47,	52,	53,	44,	63,	42,	39,	45,	54,	54,	48,	66,	49,	57,	67,	71,	66,	57,	86,	70,	95,101,105,108,113,	86,	85,116,	99,103,112,106,	98,107,109,	92,109,112,103,	93,	73,	82,	67,	71,	58,	79,	44,	78,	41,	42,	43,	45,	46,
46,	45,	34,	65,	56,	50,	50,	37,	43,	39,	54,	53,	62,	41,	40,	43,	50,	64,	37,	51,	57,	40,	68,	48,	72,	65,	81,	96,	81,100,	94,	84,	89,	88,	91,	91,	73,	91,	90,113,	90,	91,	95,	77,	76,	73,	71,	80,	87,	75,	68,	54,	62,	45,	43,	46,	51,	34,
50,	51,	42,	55,	50,	55,	40,	53,	56,	59,	49,	51,	63,	49,	49,	57,	64,	66,	55,	62,	46,	63,	66,	62,	56,	84,	65,	69,	73,	74,	98,	79,	81,107,	97,110,	91,	99,	81,	88,	82,	87,	80,	69,	79,	94,	88,	82,	70,	70,	95,	78,	82,	83,	62,	66,	55,	49,
65,	62,	77,	54,	66,	62,	67,	67,	61,	53,	59,	46,	55,	67,	63,	72,	51,	63,	57,	54,	66,	72,	87,	68,	76,	86,	93,	84,	82,	76,	86,	66,	80,	85,104,	89,101,110,105,	65,	83,112,	86,	99,	86,	69,	89,	86,	86,	87,	87,	77,	86,	64,	74,	76,	72,	80,
83,	81,	72,	76,	92,	76,	86,	80,	88,	83,	77,	58,	61,	69,	52,	71,	81,	70,	70,	76,	77,	63,	57,	64,	72,	93,	91,	94,	89,	83,	83,	95,	83,105,	91,	95,	71,	85,	79,	74,	88,112,	89,	92,	99,117,	92,106,	82,	79,	89,	84,	80,	74,	80,	90,	82,	87,
95,	64,	87,	98,	93,	94,	89,	92,	92,	88,	67,	70,	98,	55,	66,	73,	79,	86,	85,	67,	80,	76,	77,	79,	85,110,	77,	99,	84,	86,	85,	73,	73,102,	79,	81,	78,	94,	84,	88,	81,	83,	90,	81,	91,	78,	88,	80,	93,100,	89,	85,	83,	82,	93,	76,	68,	81,
84,	69,	71,	84,	78,	71,	76,117,	84,	86,	75,	98,	73,	85,	74,	82,	80,	70,	87,	87,	78,	68,	85,	71,	73,	82,	80,	80,	94,	87,105,	86,	93,	91,	90,	82,	88,	77,	85,	70,	76,	86,	88,	83,	71,	73,	71,	63,	87,	69,	79,	94,	98,	80,	77,	91,	99,	80,
80,	67,	75,	79,	70,	70,	70,	61,	67,	60,	69,	68,	62,	55,	66,	76,	61,	64,	59,	75,	71,	82,	66,	77,	79,	70,	68,	67,	66,	76,	81,	74,	61,	75,	70,	66,	65,	61,	50,	67,	64,	60,	59,	58,	57,	66,	66,	61,	75,	62,	44,	66,	54,	45,	64,	48,	69,	69,
66,	58,	56,	50,	59,	55,	40,	60,	54,	42,	54,	44,	58,	43,	51,	64,	52,	37,	54,	45,	58,	58,	57,	53,	67,	49,	58,	68,	62,	74,	75,	71,	64,	61,	63,	66,	57,	68,	62,	61,	67,	57,	60,	51,	64,	68,	60,	54,	56,	55,	53,	51,	55,	44,	40,	60,	43,	69,
49,	48,	52,	47,	56,	47,	47,	35,	52,	42,	33,	39,	33,	32,	30,	49,	47,	34,	38,	32,	41,	44,	59,	61,	51,	50,	54,	57,	56,	62,	60,	47,	59,	75,	69,	72,	47,	60,	48,	51,	52,	55,	52,	54,	50,	55,	58,	51,	53,	49,	61,	52,	61,	63,	55,	53,	52,	57,
29,	28,	24,	37,	27,	31,	41,	27,	38,	32,	23,	29,	21,	32,	20,	23,	24,	26,	26,	26,	32,	22,	41,	38,	51,	37,	43,	33,	43,	37,	50,	53,	43,	45,	49,	43,	54,	58,	55,	63,	54,	52,	50,	47,	58,	50,	49,	49,	59,	60,	59,	57,	55,	53,	55,	62,	50,	61)
v.population.females <- c( 
165550,	164350,	164850,	164600,	164400,	164450,	166500,	172300,	181200,	191550,	202965.5,	216391  ,	229814  ,	239583  , 243263.5,	240928	, 236476	, 233097	, 230074.5,	226091	, 222010.5,	221382.5,	225225.5,	229285.5,	231027	, 230952	, 231258	, 233070	, 236564.5,	241688	,	246979.5,	250818.5,	251951	,	249428.5,	244612	,	239105.5,	233321	,	227326.5,	220767	,	214115.5,	207981	,	201955	,	197184.5,	194408.5,	193205.5,	194030.5,	195277	,	196538.5,	198427.5,	200243	,	201809.5,	203449.5,	204827.5,	207043.5,	210249.5,	213291	,	216304	,	219511  ,
181600,	182750,	185200,	188300,	191050,	193200,	194850,	195750,	195950,	196050,	196276.5,	200293  ,	209441.5,	220446	, 230052.5,	236118	, 241097.5,	246795.5,	252094.5,	255508.5,	251413.5,	243867	, 241499.5,	241194	, 239770.5,	235982	, 231601.5,	229777.5,	229609.5,	230638	,	232980	,	236779	,	241457.5,	246082	,	250781	,	254856	,	258042	,	260025.5,	259578.5,	256477	,	252771	,	249962	,	245726.5,	239158	,	231993.5,	225007	,	217696	,	210886	,	206369.5,	204615.5,	205550.5,	209002	,	213122.5,	216079.5,	218169	,	219285.5,	220139	,	222117.5,
184100,	184800,	185050,	185650,	186600,	188000,	189450,	191300,	193500,	195100,	195871.5,	197909.5,	201138  ,	204052	, 207011	, 209531.5,	212881.5,	219534.5,	228704.5,	238759	, 245420	, 249525	, 254422.5,	257863.5,	259651	, 257410.5,	250262.5,	243512.5,	239053	, 235754.5,	234232	,	234933	,	237312	,	239772	,	242461	,	245655	,	249726.5,	254649	,	260625	,	267420.5,	274428.5,	284068.5,	290787	,	290609.5,	287409	,	282214	,	275051	,	266223	,	256910	,	248258	,	241209	,	236753	,	234577.5,	233416.5,	233370.5,	234498	,	236321	,	239992  ,
159250,	164900,	171000,	176800,	182100,	185050,	185750,	186300,	187250,	188650,	190465.5,	192713.5,	195710.5,	198607.5,	200746.5,	202116	, 203588.5,	205506	, 207692.5,	210817	, 213810.5,	216964	, 222191	, 228864	, 235959	, 241111.5,	243792	, 246052.5,	247966.5,	248846.5,	249107	,	247907	,	245562	,	243225	,	241255	,	240884	,	242259	,	245139	,	248977	,	253886	,	259657.5,	269480	,	279652.5,	287671.5,	295807.5,	301717	,	305044	,	305497.5,	303093.5,	298738	,	293278.5,	288326	,	283631.5,	277613	,	271086	,	264976.5,	259697	,	257261.5,
179150,	174900,	170750,	166700,	163700,	164800,	170200,	175900,	181350,	186350,	188909.5,	189828.5,	191226	, 192773	, 194104.5,	195382.5,	196667.5,	198534.5,	200726	, 202669.5,	204162	, 205347.5,	206379	, 207165.5,	208687.5,	209513	, 209897.5,	212965	, 218849.5,	225752.5,	232637	,	239415.5,	245032.5,	248600.5,	250167.5,	250428	,	249299.5,	247526.5,	246297	,	245933.5,	247294	,	251289.5,	256550	,	261398	,	266976.5,	273291	,	279421	,	284913.5,	291275.5,	298509.5,	304594.5,	309498	,	313026	,	313517.5,	310935	,	306381.5,	300941.5,	295839.5,
184650,	184850,	184550,	183350,	181300,	178000,	173350,	168350,	163500,	159800,	160317.5,	166348.5,	173890  , 181289.5,	188073	, 191543	, 192196	, 192895.5,	194019	, 195260	, 196542.5,	197521.5,	198657	, 200237	, 201688.5,	201922	, 201304.5,	200814.5,	200566.5,	201270.5,	202750	,	205587.5,	210968	,	218057.5,	225439.5,	232584	,	239249	,	244747	,	248818.5,	251179.5,	252426.5,	252258	,	251188.5,	250577	,	250457.5,	251499.5,	254021.5,	257541.5,	261423.5,	266392	,	272503.5,	279685.5,	287458.5,	295813.5,	304197.5,	310927.5,	315753.5,	319138.5,
174600,	176150,	176950,	178900,	181150,	182350,	182550,	182550,	181750,	180050,	177115.5,	172794.5,	167835.5,	162825  ,	159097	, 160268	, 166630	, 173787	, 180980	, 187775.5,	191165	, 191516	, 192029.5,	192903.5,	193832.5,	194500	, 194255	, 194149.5,	194822.5,	195524.5,	196156.5,	197152.5,	198152.5,	198621.5,	199461	,	200858.5,	203604.5,	208922	,	216195	,	224031.5,	231651	,	238624.5,	244122.5,	247932	,	250049	,	251187	,	250445.5,	248630.5,	247606	,	247534.5,	248807.5,	251993.5,	256926.5,	262204	,	268259.5,	275041	,	282190	,	289795.5,
157100,	160850,	164400,	167050,	168950,	171050,	172700,	173750,	175950,	178450,	179904.5,	180448  ,	180853	, 180385  ,	178866	, 175833	, 171196.5,	166090	, 161039	, 157314.5,	158370	, 164516	, 171555.5,	178523.5,	184943.5,	188060	, 187919.5,	187508.5,	187595.5,	188036	,	188823.5,	189496.5,	190299.5,	191333.5,	192065	,	192616.5,	193452	,	194271	,	194864	,	196022	,	197649	,	200082.5,	204824	,	211889.5,	219636.5,	226929.5,	233397	,	238372.5,	241819.5,	243987	,	245268.5,	244977.5,	244313	,	244265.5,	244678	,	246488	,	249919.5,	254778.5,
133650,	137300,	140700,	144200,	147950,	151800,	155600,	159200,	161900,	163900,	166156	, 168060  , 169285.5, 171617.5,	174304	, 175862.5,	176463.5,	176850	, 176341	, 174899	, 172012	, 167419.5,	162368	, 157410.5,	153742	, 154612.5,	160221.5,	166536.5,	172724.5,	178459.5,	181508	,	181988	,	182157	,	182375.5,	182742.5,	183358	,	183668	,	184145.5,	185108	,	185829.5,	186321.5,	187014.5,	187318	,	187448	,	188327.5,	189781	,	192308.5,	197098	,	204021.5,	211580.5,	218820	,	225268	,	230791.5,	234665	,	237387.5,	239637.5,	239652	,	238891.5,
113850,	115550,	118000,	120700,	123450,	126600,	130050,	133350,	136800,	140550,	144413	, 148338  , 152168	, 155008.5,	157097.5,	159339	, 161143	, 162412	, 164843	, 167577	, 169195.5,	169873.5,	170195.5,	169710.5,	168416	, 165666	, 161011.5,	155784.5,	150782	, 147027	,	147836	,	153531.5,	160030.5,	166150	,	171505	,	174220.5,	174423.5,	174274.5,	174347.5,	174636	,	175150.5,	175263.5,	175523	,	176422.5,	177148	,	177508	,	177990	,	178252	,	178475.5,	179446	,	180960.5,	183569.5,	188577	,	195670	,	203210.5,	210513.5,	217374.5,	223108.5,
96750	,  97900,	98900	, 100200,	101850,	103250,	105000,	107450,	110100,	112900,	116057	, 119617.5, 123052.5, 126369.5,	129960	, 133776	, 137597.5,	141400.5,	144402	, 146533	, 148845.5,	150656.5,	151954.5,	154516	, 157304	, 158863	, 159586	, 159989.5,	159492.5,	158371	,	155960	,	151817.5,	147204.5,	142714.5,	139265	,	140064	,	145415	,	151286.5,	156823	,	162006	,	164810	,	165039.5,	164947	,	165112.5,	165196.5,	165573.5,	165957	,	166468.5,	167323.5,	168089.5,	168651	,	169156.5,	169352.5,	169659	,	170812.5,	172343	,	175172.5,	180492.5,
77100	,  77800,	78650	,  79550,	80500	,  81600,	82800	,  83900,	85300	,  87200,	88887	  ,  90822  ,	93284	  ,  95727  ,	98300.5	, 101203.5,	104414.5,	107729.5,	110942.5,	114226.5,	117800.5,	121526	, 125288	, 128340	, 130813.5,	133475	, 135589	, 137271.5,	139887.5,	142463.5,	144083	,	145289	,	146157	,	146016.5,	145355	,	143468.5,	139781.5,	135600.5,	131608	,	128615.5,	129508	,	134824	,	140473.5,	145680.5,	150841	,	153557.5,	153689.5,	153608.5,	154003	,	154470.5,	155145	,	156062.5,	156857	,	157551.5,	158098	,	158624	,	159444	,	160009.5,
47500	,  49900,	52050	,  53950,	55600	,  56700,	57450	,  58450,	59600	,  60900,	62160.5	,  63517  ,	64820	  ,  66043.5,	67705	  , 69132.5	, 70655.5	, 72898	  , 75403	  , 77915	  , 80528	  , 83373	  , 86315.5	, 89285.5	, 92587	  , 96224.5	, 99982.5	, 103841	, 107048	, 109639.5,	112170.5,	114213.5,	116067	,	118811	,	121617	,	123615	,	125141.5,	126359	,	126735	,	126672	,	125398.5,	122497	,	119201.5,	115982.5,	113708	,	114982	,	119937	,	125038	,	129722.5,	134382	,	137189	,	138265.5,	138964	,	139409	,	140030	,	140990	,	141830	,	142766.5)
v.population.males <- c(
165100,	163800,	162250,	160850,	160200,	159900,	162500,	170100,	180800,	192700,	205857.5,	221112  ,	235664.5,	245627.5,	248830	, 245481.5,	240375	, 237420.5,	235257.5,	232263.5,	228667	, 226309	, 227434.5,	230330	, 233345.5,	235269.5,	236595	, 239402	, 244366	, 250888	,	257638	,	262512	,	263753	,	261319	,	256569	,	250849	,	244986.5,	238888.5,	232333	,	225764	,	219361.5,	212665.5,	207562.5,	204973	,	203429	,	203875	,	205865	,	207706	,	209541	,	212167	,	214169	,	214209	,	214780.5,	217555.5,	220685.5,	223936	,	227688	,	231383.5,
167500,	169250,	172800,	176200,	179300,	181950,	184000,	185400,	186900,	189350,	192472	, 199311  , 211484	, 225084.5,	235857.5,	242685	, 248440	, 253787	, 256894.5,	256953	, 250122	, 241948.5,	239300	, 236652.5,	231725	, 226371	, 223154	, 223258.5,	224892.5,	227855.5,	231890.5,	236827	,	242764	,	248687	,	254838.5,	260837	,	264975.5,	266452.5,	265196	,	261606.5,	256990.5,	252275	,	246591	,	239724.5,	232690	,	225710.5,	218398.5,	212389.5,	208977.5,	207531.5,	208516	,	212073	,	215679.5,	218203.5,	220805	,	222844	,	224163	,	226605  ,
173850,	173500,	175000,	176900,	179000,	181700,	184400,	187100,	189750,	192250,	194429	, 198454.5, 203350	, 207536.5,	211714.5,	214330	, 217853	, 225792	, 236714.5,	248197.5,	253534	, 255601.5,	260266	, 263394	, 263303	, 258476	, 250004	, 241968	, 235680	, 230811.5,	228507	,	229959	,	234340.5,	238419.5,	242353	,	246731.5,	251305	,	256745	,	263236.5,	270889.5,	279632.5,	288597.5,	292893	,	290965.5,	285335.5,	277130	,	267364	,	257358.5,	248653.5,	241497.5,	235416	,	232868	,	232227	,	230654.5,	230089.5,	231027	,	233275.5,	237972.5,
154700,	160250,	166250,	171700,	176550,	178900,	179000,	179350,	179900,	180550,	181772	, 184967  , 190146	, 195193.5,	199744.5,	203405.5,	207001.5,	210782	, 214693	, 219471.5,	222698.5,	224940	, 230193.5,	237747	, 245951	, 252032	, 255001.5,	256455.5,	256629.5,	255816.5,	254223	,	251286	,	247731	,	244637	,	242203.5,	241954	,	244695.5,	249594.5,	255087.5,	261287	,	268597	,	278284	,	287584	,	295043.5,	302535.5,	308078	,	310122.5,	308317.5,	303201.5,	296038.5,	287866.5,	283156.5,	279918	,	273924.5,	267804.5,	262034	,	256757	,	254977.5,
171050,	167650,	164500,	161050,	158400,	159550,	164300,	168650,	172500,	175700,	176386.5,	176672  ,	179053	, 181461  ,	183741	, 186758.5,	190095	, 193938.5,	198254	, 202723	, 207340	, 210755	, 212576	, 213926	, 215766.5,	215935	, 215165.5,	218048.5,	224370.5,	232158.5,	240708	,	248858	,	254554	,	257347.5,	258013	,	257469.5,	255481.5,	252998	,	251562.5,	251372	,	253429.5,	258840.5,	265533.5,	271936	,	278490	,	284359.5,	289155.5,	293442.5,	298499	,	304774	,	310441.5,	314456.5,	316953.5,	316269	,	312017	,	305787.5,	299463	,	294872  ,
172650,	173200,	173050,	172600,	171450,	168600,	164800,	160950,	156750,	153250,	153535.5,	159005.5,	166053.5,	173171  ,	179315	, 181588	, 181289.5, 182052	, 183493.5,	185505	, 189524.5,	193121.5,	195242	, 197804.5,	200783	, 202507.5,	202616	, 202462.5,	202651.5,	203636	,	204934.5,	207907	,	214135.5,	222137	,	230510.5,	239281	,	247364.5,	253179	,	256932	,	258923.5,	259676.5,	258748.5,	256902.5,	255637	,	255307.5,	256441	,	259517.5,	263933	,	268832.5,	274501.5,	280310	,	286055	,	292524.5,	300294.5,	308715	,	316120	,	321529	,	325300  ,
158450,	161150,	163150,	165550,	168050,	170000,	170850,	171150,	171200,	170650,	168439.5,	164978.5,	160934.5,	156333  ,	152690.5,	153861.5,	159776	, 166314.5,	173120.5,	179170.5,	182179.5,	182195	, 182122.5,	182392	, 183366	, 184950.5,	185893.5,	186982	, 188883.5,	191219	,	193542	,	195693	,	197378.5,	198706.5,	200124	,	201565	,	204527	,	210581	,	218847	,	227884.5,	237166	,	245564.5,	251145.5,	254346.5,	255912	,	256242	,	254616.5,	251964.5,	250148	,	249638	,	250790	,	254637.5,	260778.5,	267365	,	274340.5,	281167	,	287535	,	294899  ,
136450,	140150,	144000,	147750,	150800,	153650,	156350,	158350,	160900,	163600,	165573.5,	166334  ,	166648	, 166914  ,	166604	, 164627	, 161325.5,	157403.5,	152938	, 149529	, 151262	, 157395	, 163446.5,	169291.5,	174212	, 175763	, 174434.5,	173609	, 173472	, 174246.5,	176213	,	178460.5,	180778	,	183193	,	185572.5,	187750	,	189738.5,	191321	,	192766.5,	194446.5,	196173	,	198926	,	204355	,	212214.5,	220783.5,	229547	,	237287.5,	242324	,	245390.5,	247076.5,	247529.5,	246593	,	245626	,	245270	,	241617	,	243492	,	251841	,	258290  ,
112400,	115800,	119100,	122150,	125450,	129050,	132400,	135950,	139350,	142050,	144658.5,	147297  ,	149224.5,	151449  ,	153994.5,	156163	, 157261.5,	157939	, 158547	, 158609.5,	157320.5,	154567	, 150816	, 146548.5,	143257	, 144152	, 148974	, 154016.5,	158928.5,	163106.5,	164722.5,	164299	,	164211	,	164433.5,	165261	,	167007.5,	168948.5,	171040.5,	173355	,	175721	,	177893.5,	179524.5,	180132	,	180919.5,	182479	,	184088	,	186852	,	192336	,	199988	,	208504	,	217308	,	225108	,	231061	,	235238.5,	237686.5,	239000	,	238622.5,	237680.5,
94050	,  94800,	96400	,  98200,	100250,	102900,	105850,	108600,	111150,	114000,	117016	, 120021.5, 123528	, 126832  ,	129489	, 132083	, 134649	, 136628	, 138930	, 141309.5,	143351	, 144461.5,	145172	, 145920	, 146095	, 144476	, 141311.5,	137701.5,	133831.5,	130743	,	131579.5,	136349	,	141396	,	145886	,	149417	,	150751.5,	150146	,	149825	,	150040	,	150944	,	152637	,	154122.5,	155568	,	157411.5,	159638.5,	161828	,	163467	,	164572	,	165893.5,	167621.5,	169249	,	172116	,	177981.5,	186138.5,	194913.5,	203919	,	212109	,	218073.5,
78450	,  79150,	79700	,  80200,	80550	,  80850,	81650	,  83000,	84550	,  86400,	88684	  ,  91189.5,	93611	  ,  95731.5,	98247.5	, 101126	, 103963	, 107183.5,	110235	, 112721	, 115104.5,	117449.5,	119342.5,	121491	, 123704	, 125357	, 126124.5,	126681	, 127171.5,	127257	,	126000.5,	123612.5,	120891	,	117767.5,	115184	,	116000.5,	120085.5,	124192	,	127989	,	131199.5,	132587	,	132180.5,	131929.5,	132130	,	132856.5,	134394.5,	136101.5,	137968	,	140046	,	142476	,	144825	,	146369	,	147526.5,	149238.5,	151299	,	153207.5,	156731.5,	163215  ,
58250	,  58700,	59200	,  60050,	61100	,  61850,	62400	,  62850,	63300	,  63850,	64363.5	,  65183  ,	66285	  ,  67476  ,	68998	  , 70907	  , 72894.5	, 74802.5	, 76644.5	, 78742.5	, 80955	  , 83318.5	, 86195.5	, 89032	  , 91404.5	, 93682	  , 95822.5	, 97505	  , 99353	  , 101309.5,	102902.5,	103849	,	104739	,	105480.5,	105917	,	105306.5,	103569	,	101383.5,	98961.5 ,	97011.5 ,	97950.5 ,	101796.5,	105435.5,	108805	,	111867.5,	113312	,	113248	,	113511.5,	114281.5,	115499	,	117337.5,	119423	,	121582.5,	123610.5,	125897	,	128353.5,	130541.5,	132244  ,
33900	,  35450,	36850	,  38000,	38800	,  39400,	39850	,  40400,	41350	,  42500,	43159	  ,  43698.5,	44175	  ,  44235.5,	44464	  , 44819.5	, 45365.5	, 46426	  , 47486.5	, 48606	  , 50057	  , 51600	  , 53155	  , 54612.5	, 56403.5	, 58417.5	, 60392	  , 62716.5	, 65012.5	, 66974	  ,	68925.5 ,	70775.5 ,	72305	  ,	74038  	,	75826.5 ,	77247.5 ,	78251	  ,	79279.5 ,	80177	  ,	80903.5 ,	80658	  ,	79435	  ,	78109.5 ,	76635	  ,	75555	  ,	76810.5 ,	80341   ,	83689.5 ,	86769.5 ,	89543.5 ,	91128.5 ,	91991.5 ,	92956	  ,	93982  	,	95553.5 ,	97597.5 ,	99674	  ,	102182  )
m.r.f <- matrix(data=v.suicides.females  ,nrow=AG,ncol=PH,byrow=TRUE,dimnames=list(row.names,col.names))
m.r.m <- matrix(data=v.suicides.males    ,nrow=AG,ncol=PH,byrow=TRUE,dimnames=list(row.names,col.names))
m.d.f <- matrix(data=v.population.females,nrow=AG,ncol=PH,byrow=TRUE,dimnames=list(row.names,col.names))
m.d.m <- matrix(data=v.population.males  ,nrow=AG,ncol=PH,byrow=TRUE,dimnames=list(row.names,col.names))
data.list.f	<- apc.data.list(response=m.r.f,dose=m.d.f,data.format="APm",age1=age1,per1=per1,coh1=NULL,unit=c(G,H),per.zero=NULL,per.max=NULL,label="females")
data.list.m	<- apc.data.list(response=m.r.m,dose=m.d.m,data.format="APm",age1=age1,per1=per1,coh1=NULL,unit=c(G,H),per.zero=NULL,per.max=NULL,label="males"  )
# 	Read covariate data, from Andrea Riebler's code.
## read the covariate information on divorces (source: Matthias Bopp)
# get the divorce rate per 1000 habitants
v.divorcePer1000 <- c(0.9,0.9,0.9,0.9,0.9,0.9,0.8,0.9,0.8,0.9,0.9,0.9,0.8,0.8,0.8,0.8,0.8,0.9,0.9,1  ,1  ,1.1,1.2,1.3,1.3,1.4,1.5,1.7,1.7,1.7,1.7,1.8,1.8,1.8,1.7,1.8,1.8,1.8,1.9,1.9,2  ,2  ,2.1,2.2,2.2,2.2,2.3,2.4,2.5,2.9,1.5,2.2,2.2,2.3,2.4,2.9,2.8,2.6)
names(v.divorcePer1000) <- col.names
# substitute the rates of divorce in the years 1999 and 2000 by the average  
# for these two year (in order to adjust for the introduction of a new   
# divorce law in the year 2000)} #	data.RHRB.swiss
v.divorcePer1000[c("1999", "2000")] <- mean(v.divorcePer1000[c("1999", "2000")])
# get the cumulated first-marriage index of males
v.cumMarriage.m <- c(95,96.2,94.6,92.7,93.6,96.4,97.3,99.2,94.3,93.3,95.4,95.2,96.4,92.4,89.7,89.3,86.1,85.3,83.7,83.5,83.2,80.8,76.9,72.3,68.1,62.9,58,59.9,58.7,61.9,64.6,64,65.1,64.8,65.5,64.4,65.9,68.8,71.7,69.2,70.2,69.2,65.5,61.9,60.6,58.1,58.5,56.5,56.5,59.5,58.2,57.3,59,58.8,57.7,58.6,58.1,58.6)
names(v.cumMarriage.m) <- col.names
# get the cumulated first-marriage index of females
v.cumMarriage.f <- c(92.2,93.1,91.9,90.2,91.4,94.7,95.7,98.8,94.1,93.8,96.3,95.5,97,92.4,89.9,89.7,86.9,86.9,86.1,86.9,87.1,84.9,81.2,76.1,71.4,65.3,60.1,61.9,60,63.2,66.2,65.4,66.7,66.6,67.4,66.7,68.3,72.1,74.9,73.1,75,74.2,70.6,67.1,65.9,63.6,64.1,61.9,62.1,65.6,64,62.4,64.2,62.9,61.7,63,62.8,63.1)
names(v.cumMarriage.f) <- col.names
## read the matrix of all covariates (source: Matthias Bopp)
v.marriagePer1000 <- c(7.9,7.9,7.8,7.7,7.7,7.9,8,8.1,7.7,7.6,7.8,7.8,8,7.7,7.6,7.7,7.5,7.6,7.5,7.6,7.6,7.2,6.9,6.5,6.1,5.6,5.1,5.3,5.1,5.4,5.7,5.6,5.8,5.9,6,6,6.2,6.6,6.9,6.8,6.9,7,6.6,6.2,6.1,5.8,5.7,5.5,5.4,5.7,5.5,5.4,5.5,5.4,5.3,5.4,5.3,5.3)
names(v.marriagePer1000) <- col.names
## NOTE: we use the adjusted divorce rate
v.divIndex <- v.divorcePer1000
v.F_index <- (v.marriagePer1000 - v.divIndex)/(v.marriagePer1000 + v.divIndex)
###########
return(list(
          data.list.f       = data.list.f       ,
          data.list.m       = data.list.m       ,
          v.divorcePer1000  = v.divorcePer1000  ,
          v.cumMarriage.f   = v.cumMarriage.f   ,
          v.cumMarriage.m   = v.cumMarriage.m   ,
          v.marriagePer1000 = v.marriagePer1000 ,
          v.F_index         = v.F_index         ))
} #	data.RHRB.swiss         





















































































































































































































































































































































































































































































































































































































































































































































































































