#' Estimate Semipartial Correlation Coefficients
#' and Generate the Corresponding Sampling Distribution
#' Using the Monte Carlo Method
#'
#' @author Ivan Jacob Agaloos Pesigan
#'
#' @details The vector of semipartial correlation coefficients
#'   (\eqn{r_{s}})
#'   is derived from each randomly generated vector of parameter estimates.
#'   Confidence intervals are generated by obtaining
#'   percentiles corresponding to \eqn{100(1 - \alpha)\%}
#'   from the generated sampling
#'   distribution of \eqn{r_{s}},
#'   where \eqn{\alpha} is the significance level.
#'
#' @return Returns an object
#'   of class `betamc` which is a list with the following elements:
#'   \describe{
#'     \item{call}{Function call.}
#'     \item{args}{Function arguments.}
#'     \item{thetahatstar}{Sampling distribution of
#'       \eqn{r_{s}}.}
#'     \item{vcov}{Sampling variance-covariance matrix of
#'       \eqn{r_{s}}.}
#'     \item{est}{Vector of estimated
#'       \eqn{r_{s}}.}
#'     \item{fun}{Function used ("SCorMC").}
#'   }
#'
#' @inheritParams BetaMC
#'
#'
#' @examples
#' # Data ---------------------------------------------------------------------
#' data("nas1982", package = "betaMC")
#'
#' # Fit Model in lm ----------------------------------------------------------
#' object <- lm(QUALITY ~ NARTIC + PCTGRT + PCTSUPP, data = nas1982)
#'
#' # MC -----------------------------------------------------------------------
#' mc <- MC(
#'   object,
#'   R = 100, # use a large value e.g., 20000L for actual research
#'   seed = 0508
#' )
#'
#' # SCorMC -------------------------------------------------------------------
#' out <- SCorMC(mc, alpha = 0.05)
#'
#' ## Methods -----------------------------------------------------------------
#' print(out)
#' summary(out)
#' coef(out)
#' vcov(out)
#' confint(out, level = 0.95)
#'
#' @family Beta Monte Carlo Functions
#' @keywords betaMC scor
#' @export
SCorMC <- function(object,
                   alpha = c(0.05, 0.01, 0.001)) {
  stopifnot(
    inherits(
      x = object,
      what = "mc"
    )
  )
  if (object$lm_process$p < 2) {
    stop("Two or more regressors is required.")
  }
  if (object$fun == "MCMI") {
    est <- colMeans(
      do.call(
        what = "rbind",
        args = lapply(
          X = object$args$mi_output$lm_process,
          FUN = function(x) {
            x$scor
          }
        )
      )
    )
  } else {
    est <- object$lm_process$scor
  }
  std <- BetaMC(object)
  std <- std$thetahatstar
  thetahatstar <- mapply(
    theta = object$thetahatstar,
    std = std,
    SIMPLIFY = FALSE,
    FUN = function(theta,
                   std) {
      .SPCor(
        betastar = std,
        sigmacapx = theta$sigmacapx
      )
    }
  )
  out <- list(
    call = match.call(),
    args = list(
      object = object,
      alpha = alpha
    ),
    thetahatstar = thetahatstar,
    est = est,
    fun = "SCorMC"
  )
  class(out) <- c(
    "betamc",
    class(out)
  )
  out
}
