#' Estimate Differences of Standardized Slopes
#' and Generate the Corresponding Sampling Distribution
#' Using the Monte Carlo Method
#'
#' @author Ivan Jacob Agaloos Pesigan
#'
#' @details The vector of differences of standardized regression slopes
#'   is derived from each randomly generated vector of parameter estimates.
#'   Confidence intervals are generated by obtaining
#'   percentiles corresponding to \eqn{100(1 - \alpha)\%}
#'   from the generated sampling
#'   distribution of differences of standardized regression slopes,
#'   where \eqn{\alpha} is the significance level.
#'
#' @return Returns an object
#'   of class `betamc` which is a list with the following elements:
#'   \describe{
#'     \item{call}{Function call.}
#'     \item{args}{Function arguments.}
#'     \item{thetahatstar}{Sampling distribution of
#'       differences of standardized regression slopes.}
#'     \item{vcov}{Sampling variance-covariance matrix of
#'       differences of standardized regression slopes.}
#'     \item{est}{Vector of estimated
#'       differences of standardized regression slopes.}
#'     \item{fun}{Function used ("DiffBetaMC").}
#'   }
#'
#' @inheritParams BetaMC
#'
#' @examples
#' # Data ---------------------------------------------------------------------
#' data("nas1982", package = "betaMC")
#'
#' # Fit Model in lm ----------------------------------------------------------
#' object <- lm(QUALITY ~ NARTIC + PCTGRT + PCTSUPP, data = nas1982)
#'
#' # MC -----------------------------------------------------------------------
#' mc <- MC(
#'   object,
#'   R = 100, # use a large value e.g., 20000L for actual research
#'   seed = 0508
#' )
#'
#' # DiffBetaMC ---------------------------------------------------------------
#' out <- DiffBetaMC(mc, alpha = 0.05)
#'
#' ## Methods -----------------------------------------------------------------
#' print(out)
#' summary(out)
#' coef(out)
#' vcov(out)
#' confint(out, level = 0.95)
#'
#' @family Beta Monte Carlo Functions
#' @keywords betaMC diff
#' @export
DiffBetaMC <- function(object,
                       alpha = c(0.05, 0.01, 0.001)) {
  stopifnot(
    inherits(
      x = object,
      what = "mc"
    )
  )
  if (object$lm_process$p < 2) {
    stop("Two or more regressors is required.")
  }
  if (object$fun == "MCMI") {
    est <- colMeans(
      do.call(
        what = "rbind",
        args = lapply(
          X = object$args$mi_output$lm_process,
          FUN = function(x) {
            x$dif_betastar
          }
        )
      )
    )
  } else {
    est <- object$lm_process$dif_betastar
  }
  std <- BetaMC(object)
  std <- std$thetahatstar
  std <- do.call(
    what = "rbind",
    args = std
  )
  p_dif <- dim(object$lm_process$dif_idx)[2]
  thetahatstar <- matrix(
    data = 0.0,
    ncol = p_dif,
    nrow = dim(std)[1]
  )
  for (i in seq_len(p_dif)) {
    thetahatstar[, i] <- std[
      ,
      object$lm_process$dif_idx[1, i]
    ] - std[
      ,
      object$lm_process$dif_idx[2, i]
    ]
  }
  colnames(thetahatstar) <- names(object$lm_process$dif_betastar)
  thetahatstar <- as.data.frame(
    t(
      thetahatstar
    )
  )
  out <- list(
    call = match.call(),
    args = list(
      object = object,
      alpha = alpha
    ),
    thetahatstar = thetahatstar,
    est = est,
    fun = "DiffBetaMC"
  )
  class(out) <- c(
    "betamc",
    class(out)
  )
  out
}
