

##
# ------------------------------------------------------------------------
#
#  "compute_power(null_dist, alt_dist, alpha)" --
#
#
# ------------------------------------------------------------------------
##
#' @aliases compute_power
#' @title Compute the Power of a Statistical Test
#' @description This function computes the power of a statistical test given 
#' the distributions under the null and alternative hypotheses for a specified 
#' significance level.
#' @param null_dist A numeric vector representing the distribution under the 
#' null hypothesis.
#' @param alt_dist  A numeric vector representing the distribution under an 
#' alternative hypothesis.
#' @param alpha A numeric value in \eqn{(0,1)}; type I error rate of the test.
#' @details The function calculates the proportion of values of the alternative 
#' distribution that falls into the critical region determined by the distribution 
#' under the null for an error rate \code{alpha}. 
#' @return A numeric value estimating the power of the test.
#' @references Bertail, P. and Dudek, A. (2025). \emph{Bootstrap for 
#' Dependent Data, with an R package} (by Bernard Desgraupes and Karolina Marek) - submitted.
#' 
#' Beran, R. (1986). Simulated Power Functions. \emph{The Annals of Statistics}, \bold{14}, 
#' 151-173.
#' @seealso \code{\link{pkc}}, \code{\link{bootglm}}, \code{\link{bootsemi}}.
#' @keywords Bootstrap Test Power.
#' @examples 
#' # Generate two normally distributed samples as null and alternative distributions
#' set.seed(123)
#' null_dist <- rnorm(1000, mean = 0, sd = 1) # Null distribution
#' alt_dist <- rnorm(1000, mean = 0.5, sd = 1) # Alternative distribution
#' alpha <- 0.05 # Significance level
#' # Compute the power of the test
#' test_power <- compute_power(null_dist, alt_dist, alpha)
#' print(test_power)
#' @export
##
compute_power <- function(null_dist, alt_dist, alpha) {
  power1 <- sum(alt_dist < quantile(null_dist, alpha/2)) /length(null_dist)
  
  power2 <- sum(alt_dist > quantile(null_dist, 1-alpha/2))/length(null_dist)
  
  power=power1+power2
  return(power)
}