#' Pairwise Parametric Bootstrap Mean Test (PPBMT)
#'
#' Performs a parametric bootstrap test to compare all pairwise group means
#' under heteroscedasticity, assuming normality of the data.
#'
#' @param means A numeric vector containing the sample means for each group.
#' @param vars A numeric vector containing the sample variances for each group.
#' @param ns An integer vector indicating the sample sizes of each group.
#' @param B Number of bootstrap re-samples. Default is 1000.
#' @param alpha Significance level for the hypothesis test. Default is 0.05.
#'
#' @return A list of class \code{"PPBMT"} containing:
#' \item{test_statistic}{Observed value of the test statistic.}
#' \item{critical_value}{Bootstrap-based critical value.}
#' \item{decision}{Conclusion of the hypothesis test.}
#'
#' @examples
#' # Example with 3 groups
#' set.seed(123)
#' g1 <- rnorm(20, mean = 5, sd = 1.5)
#' g2 <- rnorm(25, mean = 6.5, sd = 2)
#' g3 <- rnorm(22, mean = 7.2, sd = 2.5)
#'
#' means <- c(mean(g1), mean(g2), mean(g3))
#' vars <- c(var(g1), var(g2), var(g3))
#' ns <- c(length(g1), length(g2), length(g3))
#'
#' result <- PPBMT(means, vars, ns, B = 1000, alpha = 0.05)
#' print(result)
#'
#' @importFrom stats quantile rnorm var
#' @importFrom utils combn
#' @export
PPBMT <- function(means, vars, ns, B = 1000, alpha = 0.05) {

  k <- length(means)
  pair_indices <- combn(1:k, 2)

  total_distance <- function(means, vars, ns) {
    sum_vals <- 0
    for (idx in 1:ncol(pair_indices)) {
      i <- pair_indices[1, idx]
      j <- pair_indices[2, idx]

      m1 <- means[i]
      m2 <- means[j]

      s1 <- vars[i]
      s2 <- vars[j]

      n1 <- ns[i]
      n2 <- ns[j]

      se <- sqrt(s1/n1 + s2/n2)
      sum_vals <- sum_vals + abs((m1 - m2) / se)
    }
    return(sum_vals)
  }

  obs <- total_distance(means, vars, ns)
  boot_vals <- numeric(B)

  mu_pooled <- sum(means * ns) / sum(ns)

  for (b in 1:B) {
    resamples <- lapply(1:k, function(i) rnorm(ns[i], mean = mu_pooled, sd = sqrt(vars[i])))
    res_vals <- sapply(resamples, var)
    res_means <- sapply(resamples, mean)
    boot_vals[b] <- total_distance(res_means, res_vals, ns)
  }

  crit_val <- quantile(boot_vals, probs = 1 - alpha)
  decision <- if (obs > crit_val) {
    "Reject H0: At least one pair of means differs significantly."
  } else {
    "Do not reject H0: No significant difference among group means."
  }

  # Return as a list
  result <- list(
    test_statistic = obs,
    critical_value = crit_val,
    decision = decision
  )
  class(result) <- "PPBMT"
  return(result)
}

