% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rmst_delta.R
\name{rmst_delta}
\alias{rmst_delta}
\title{Calculate the variance of the marginal restricted mean survival time (RMST) when adjusting covariates using the delta method}
\usage{
rmst_delta(fit, time, trt, covariates, tau, surv0, surv1, cumhaz0, cumhaz1)
}
\arguments{
\item{fit}{A \link[survival]{coxph} object with strata(trt) in the model. See example.}

\item{time}{A vector containing the event time of the sample.}

\item{trt}{A vector indicating the treatment assignment. 1 for treatment group. 0 for placebo group.}

\item{covariates}{A data frame containing the covariates.}

\item{tau}{Numeric. A value for the restricted time or the pre-specified cutoff time point.}

\item{surv0}{A vector containing the cumulative survival function for the placebo group or trt0.}

\item{surv1}{A vector containing the cumulative survival function for the treatment group or trt1.}

\item{cumhaz0}{A data frame containing the cumulative hazard function for the placebo group or trt0.}

\item{cumhaz1}{A data frame containing the cumulative hazard function for the placebo group or trt1.}
}
\value{
A value of the SE.
}
\description{
Standard errors (SE) were estimated using the delta methods from Zucker (1998),
Chen and Tsiatis (2001), and Wei et al.(2023).
}
\details{
Restricted mean survival time is a measure of average survival time up to a specified time point. We adopted the methods from Karrison et al.(2018) for
estimating the marginal RMST when adjusting covariates. For the SE, both nonparametric bootstrap and delta method are good for estimation.
We used the delta estimation from Zucker (1998) but we also included an additional variance component which treats the covariates as random as described in Chen and Tsiatis (2001).
}
\examples{
library(survival)
data("oak")

tau <- 26
time <- oak$OS
status <- oak$os.status
trt <- oak$trt
covariates <- oak[, c("btmb", "pdl1")]

dt <- as.data.frame(cbind(time, status, trt, covariates))
fit <- coxph(Surv(time, status) ~ btmb + pdl1 + strata(trt), data = dt)
delta <- rmst_point_estimate(fit, dt = dt, tau)
rmst_delta(fit, time, trt, covariates, tau,
  surv0 = delta$surv0, surv1 = delta$surv1,
  cumhaz0 = delta$cumhaz0, cumhaz1 = delta$cumhaz1
)
}
\references{
\itemize{
\item Karrison T, Kocherginsky M. Restricted mean survival time: Does covariate adjustment improve precision in randomized clinical trials? Clinical Trials. 2018;15(2):178-188. doi:10.1177/1740774518759281
\item Zucker, D. M. (1998). Restricted Mean Life with Covariates: Modification and Extension of a Useful Survival Analysis Method. Journal of the American Statistical Association, 93(442), 702–709. https://doi.org/10.1080/01621459.1998.10473722
\item Wei, J., Xu, J., Bornkamp, B., Lin, R., Tian, H., Xi, D., … Roychoudhury, S. (2024). Conditional and Unconditional Treatment Effects in Randomized Clinical Trials: Estimands, Estimation, and Interpretation. Statistics in Biopharmaceutical Research, 16(3), 371–381. https://doi.org/10.1080/19466315.2023.2292774
\item Chen, P. and Tsiatis, A. (2001), “Causal Inference on the Difference of the Restricted Mean Lifetime Between Two Groups,” Biometrics; 57: 1030–1038. DOI: 10.1111/j.0006-341x.2001.01030.x.
}
}
