% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/calibrateBinary.R
\name{calibrateBinary}
\alias{calibrateBinary}
\title{Calibration for Binary Outputs}
\usage{
calibrateBinary(Xp, yp, Xs1, Xs2, ys, K = 5, lambda = seq(0.001, 0.1,
  0.005), kernel = c("matern", "exponential")[1], nu = 1.5, power = 1.95,
  rho = seq(0.05, 0.5, 0.05), sigma = seq(100, 20, -1), lower, upper,
  verbose = TRUE)
}
\arguments{
\item{Xp}{a design matrix with dimension \code{np} by \code{d}.}

\item{yp}{a response vector with length \code{np}. The values in the vector are 0 or 1.}

\item{Xs1}{a design matrix with dimension \code{ns} by \code{d}. These columns should one-by-one correspond to the columns of \code{Xp}.}

\item{Xs2}{a design matrix with dimension \code{ns} by \code{q}.}

\item{ys}{a response vector with length \code{ns}. The values in the vector are 0 or 1.}

\item{K}{a positive integer specifying the number of folds for fitting kernel logistic regression and generalized Gaussian process. The default is 5.}

\item{lambda}{a vector specifying lambda values at which CV curve will be computed for fitting kernel logistic regression. See \code{\link{cv.KLR}}.}

\item{kernel}{input for fitting kernel logistic regression. See \code{\link{KLR}}.}

\item{nu}{input for fitting kernel logistic regression. See \code{\link{KLR}}.}

\item{power}{input for fitting kernel logistic regression. See \code{\link{KLR}}.}

\item{rho}{\code{rho} value at which CV curve will be computed for fitting kernel logistic regression. See \code{\link{KLR}}.}

\item{sigma}{a vector specifying values of the tuning parameter \eqn{\sigma} at which CV curve will be computed for fitting generalized Gaussian process. See Details.}

\item{lower}{a vector of size \code{p+q} specifying lower bounds of the input space for \code{rbind(Xp,Xs1)} and \code{Xs2}.}

\item{upper}{a vector of size \code{p+q} specifying upper bounds of the input space for \code{rbind(Xp,Xs1)} and \code{Xs2}.}

\item{verbose}{logical. If \code{TRUE}, additional diagnostics are printed. The default is \code{TRUE}.}
}
\value{
a matrix with number of columns \code{q+1}. The first \code{q} columns are the local (the first row is the global) minimal solutions which are the potential estimates of calibration parameters, and the \code{(q+1)}-th column is the corresponding L2 distance.
}
\description{
The function performs the L2 calibration method for binary outputs.
}
\details{
The function performs the L2 calibration method for computer experiments with binary outputs. The input and ouput of physical data are assigned to \code{Xp} and \code{yp}, and the input and output of computer data are assigned to \code{cbind(Xs1,Xs2)} and \code{ys}. Note here we separate the input of computer data by \code{Xs1} and \code{Xs2}, where \code{Xs1} is the shared input with \code{Xp} and \code{Xs2} is the calibration input. The idea of L2 calibration is to find the calibration parameter that minimizes the discrepancy measured by the L2 distance between the underlying probability functions in the physical and computer data. That is, \deqn{\hat{\theta}=\arg\min_{\theta}\|\hat{\eta}(\cdot)-\hat{p}(\cdot,\theta)\|_{L_2(\Omega)},} where \eqn{\hat{\eta}(x)} is the fitted probability function for physical data, and \eqn{\hat{p}(x,\theta)} is the fitted probability function for computer data. In this L2 calibration framework, \eqn{\hat{\eta}(x)} is fitted by the kernel logistic regression using the input \code{Xp} and the output \code{yp}. The tuning parameter \eqn{\lambda} for the kernel logistic regression can be chosen by k-fold cross-validation, where k is assigned by \code{K}. The choices of the tuning parameter are given by the vector \code{lambda}. The kernel function for the kernel logistic regression can be given by \code{kernel}, where Matern kernel or power exponential kernel can be chosen. The arguments \code{power}, \code{nu}, \code{rho} are the tuning parameters in the kernel functions. See \code{\link{KLR}}. For computer data, the probability function \eqn{\hat{p}(x,\theta)} is fitted by the Bayesian Gaussian process in Williams and Barber (1998) using the input \code{cbind(Xs1,Xs2)} and the output \code{ys}, where the Gaussian correlation function, \deqn{R_{\sigma}(\mathbf{x}_i,\mathbf{x}_j)=\exp\{-\sum^{d}_{l=1}\sigma(x_{il}-x_{jl})^2 \},} is used here. The vector \code{sigma} is the choices of the tuning parameter \eqn{\sigma}, and it will be chosen by k-fold cross-validation. More details can be seen in Sung et al. (unpublished). The arguments \code{lower} and \code{upper} are lower and upper bounds of the input space, which will be used in scaling the inputs and optimization for \eqn{\theta}. If they are not given, the default is the range of each column of \code{rbind(Xp,Xs1)}, and \code{Xs2}.
}
\examples{
library(calibrateBinary)

set.seed(1)
#####   data from physical experiment   #####
np <- 10
xp <- seq(0,1,length.out = np)
eta_fun <- function(x) exp(exp(-0.5*x)*cos(3.5*pi*x)-1) # true probability function
eta_x <- eta_fun(xp)
yp <- rep(0,np)
for(i in 1:np) yp[i] <- rbinom(1,1, eta_x[i])

#####   data from computer experiment   #####
ns <- 20
xs <- matrix(runif(ns*2), ncol=2)  # the first column corresponds to the column of xp
p_xtheta <- function(x,theta) {
     # true probability function
     exp(exp(-0.5*x)*cos(3.5*pi*x)-1) - abs(theta-0.3) *exp(-0.5*x)*cos(3.5*pi*x)
}
ys <- rep(0,ns)
for(i in 1:ns) ys[i] <- rbinom(1,1, p_xtheta(xs[i,1],xs[i,2]))

#####    check the true parameter    #####
curve(eta_fun, lwd=2, lty=2, from=0, to=1)
curve(p_xtheta(x,0.3), add=TRUE, col=4)   # true value = 0.3: L2 dist = 0
curve(p_xtheta(x,0.9), add=TRUE, col=3)   # other value

##### calibration: true parameter is 0.3 #####
\donttest{
calibrate.result <- calibrateBinary(xp, yp, xs[,1], xs[,2], ys)
print(calibrate.result)
}
}
\author{
Chih-Li Sung <iamdfchile@gmail.com>
}
\seealso{
\code{\link{KLR}} for performing a kernel logistic regression with given \code{lambda} and \code{rho}. \code{\link{cv.KLR}} for performing cross-validation to estimate the tuning parameters.
}

