% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cgmguru-functions-docs.R
\name{detect_hypoglycemic_events}
\alias{detect_hypoglycemic_events}
\title{Detect Hypoglycemic Events}
\usage{
detect_hypoglycemic_events(df,
                                 reading_minutes = NULL,
                                 dur_length = 120,
                                 end_length = 15,
                                 start_gl = 70)
}
\arguments{
\item{df}{A dataframe containing continuous glucose monitoring (CGM) data.
Must include columns:
\itemize{
  \item \code{id}: Subject identifier (string or factor)
  \item \code{time}: Time of measurement (POSIXct)
  \item \code{gl}: Glucose value (integer or numeric, mg/dL)
}}

\item{reading_minutes}{Time interval between readings in minutes (optional)}

\item{dur_length}{Minimum duration in minutes for event classification (default: 120)}

\item{end_length}{End length criteria in minutes (default: 15)}

\item{start_gl}{Starting glucose threshold in mg/dL (default: 70)}
}
\value{
A list containing:
\itemize{
  \item \code{events_total}: Tibble with summary statistics per subject (id, total_events, avg_ep_per_day)
  \item \code{events_detailed}: Tibble with detailed event information (id, start_time, start_glucose, end_time, end_glucose, start_indices, end_indices, duration_below_54_minutes)
}
}
\description{
Identifies and segments hypoglycemic events in CGM data based on international consensus 
CGM metrics (Battelino et al., 2023). Supports three event types:
\itemize{
  \item \strong{Level 1}: \eqn{\geq} 15 consecutive min of \eqn{<} 70 mg/dL, ends with \eqn{\geq} 15 consecutive min \eqn{\geq} 70 mg/dL
  \item \strong{Level 2}: \eqn{\geq} 15 consecutive min of \eqn{<} 54 mg/dL, ends with \eqn{\geq} 15 consecutive min \eqn{\geq} 54 mg/dL
  \item \strong{Extended}: \eqn{>} 120 consecutive min of \eqn{<} 70 mg/dL, ends with \eqn{\geq} 15 consecutive min \eqn{\geq} 70 mg/dL
}
Events are detected when glucose falls below the start threshold for the minimum duration and ends
when glucose rises above the end threshold for the specified end length.
}
\section{Units and sampling}{

- \code{reading_minutes} can be a scalar (all rows) or a vector per-row.
- If \code{reading_minutes} is \code{NULL}, duration is computed from \code{time} deltas.
}

\examples{
# Load sample data
library(iglu)
data(example_data_5_subject)
data(example_data_hall)

# Level 1: \eqn{<} 70 \eqn{\geq} 15 min,
# ends \eqn{\geq} 70 \eqn{\geq} 15 min
hypo_lv1 <- detect_hypoglycemic_events(
  example_data_5_subject, 
  start_gl = 70, 
  dur_length = 15, 
  end_length = 15
)
print(hypo_lv1$events_total)

# Level 2: \eqn{<} 54 \eqn{\geq} 15 min,
# ends \eqn{\geq} 54 \eqn{\geq} 15 min
hypo_lv2 <- detect_hypoglycemic_events(
  example_data_5_subject, 
  start_gl = 54, 
  dur_length = 15, 
  end_length = 15
)

# Extended: \eqn{<} 70 \eqn{\geq} 120 min, 
# ends \eqn{\geq} 70 \eqn{\geq} 15 min
hypo_extended <- detect_hypoglycemic_events(example_data_5_subject)
print(hypo_extended$events_total)

# Compare event rates across levels
cat("Level 1 events:", sum(hypo_lv1$events_total$total_events), "\n")
cat("Level 2 events:", sum(hypo_lv2$events_total$total_events), "\n")
cat("Extended events:", sum(hypo_extended$events_total$total_events), "\n")

# Analysis on larger dataset with Level 1 criteria
large_hypo <- detect_hypoglycemic_events(example_data_hall, 
                                         start_gl = 70, 
                                         dur_length = 15, 
                                         end_length = 15)
print(large_hypo$events_total)

# Analysis on larger dataset with Level 2 criteria
large_hypo_lv2 <- detect_hypoglycemic_events(example_data_hall,
                                             start_gl = 54,
                                             dur_length = 15,
                                             end_length = 15)
print(large_hypo_lv2$events_total)

# Analysis on larger dataset with Extended criteria
large_hypo_extended <- detect_hypoglycemic_events(example_data_hall)
print(large_hypo_extended$events_total)
}
\references{
Battelino, T., et al. (2023). Continuous glucose monitoring and metrics for clinical trials: an international consensus statement. The Lancet Diabetes & Endocrinology, 11(1), 42-57.
}
\seealso{
\link{detect_all_events}
}
