% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cgmguru-functions-docs.R
\name{grid}
\alias{grid}
\title{GRID Algorithm for Glycemic Event Detection}
\usage{
grid(df, gap = 15, threshold = 130)
}
\arguments{
\item{df}{A dataframe containing continuous glucose monitoring (CGM) data.
Must include columns:
\itemize{
  \item \code{id}: Subject identifier (string or factor)
  \item \code{time}: Time of measurement (POSIXct)
  \item \code{gl}: Glucose value (integer or numeric, mg/dL)
}}

\item{gap}{Gap threshold in minutes for event detection (default: 15).
This parameter defines the minimum time interval between consecutive GRID events. For example, if gap is set to 60, only one GRID event can be detected within any one-hour window; subsequent events within the gap interval are not counted as new events.}

\item{threshold}{GRID slope threshold in mg/dL/hour for event classification (default: 130)}
}
\value{
A list containing:
\itemize{
  \item \code{grid_vector}: A tibble with the results of the GRID analysis. Contains a \code{grid} column (0/1 values; 1 denotes a detected GRID event) and all relevant input columns.
  \item \code{episode_counts}: A tibble summarizing the number of GRID events per subject (\code{id}) as \code{episode_counts}.
  \item \code{episode_start}: A tibble listing the start of each GRID episode, with columns:
    \itemize{
      \item \code{id}: Subject ID.
      \item \code{time}: The timestamp (POSIXct) at which the GRID event was detected.
      \item \code{gl}: The glucose value (mg/dL; integer or numeric) at the GRID event.
      \item \code{indices}: R-based (1-indexed) row number(s) in the original dataframe where the GRID event occurs. The occurrence time equals \code{df$time[indices]} and glucose equals \code{df$gl[indices]}.
    }
}
}
\description{
Implements the GRID (Glucose Rate Increase Detector) algorithm for detecting rapid glucose rate increases in continuous glucose monitoring (CGM) data.
This algorithm identifies rapid glucose changes using specific rate-based criteria, and is commonly applied for meal detection.
Meals are detected when the CGM value is \eqn{\geq} 7.2 mmol/L (\eqn{\geq} 130 mg/dL) and the rate-of-change is \eqn{\geq} 5.3 mmol/L/h [\eqn{\geq} 95 mg/dL/h] for the last two consecutive readings, or \eqn{\geq} 5.0 mmol/L/h [\eqn{\geq} 90 mg/dL/h] for two of the last three readings.
}
\section{Algorithm}{

- Flags points where \code{gl >= 130 mg/dL} and rate-of-change meets the GRID criteria (see references).
- Enforces a minimum \code{gap} in minutes between detected events to avoid duplicates.
}

\section{Units and sampling}{

- \code{gl} is mg/dL; \code{time} is POSIXct; \code{gap} is minutes.
- The effective sampling interval is derived from \code{time} deltas.
}

\examples{
# Load sample data
library(iglu)
data(example_data_5_subject)
data(example_data_hall)

# Basic GRID analysis on smaller dataset
grid_result <- grid(example_data_5_subject, gap = 15, threshold = 130)
print(grid_result$episode_counts)
print(grid_result$episode_start)
print(grid_result$grid_vector)

# More sensitive GRID analysis
sensitive_result <- grid(example_data_5_subject, gap = 10, threshold = 120)

# GRID analysis on larger dataset
large_grid <- grid(example_data_hall, gap = 15, threshold = 130)
print(paste("Detected", sum(large_grid$episode_counts$episode_counts), "episodes"))
print(large_grid$episode_start)
print(large_grid$grid_vector)

}
\references{
Harvey, R. A., et al. (2014). Design of the glucose rate increase detector: a meal detection module for the health monitoring system. Journal of Diabetes Science and Technology, 8(2), 307-320.

Adolfsson, Peter, et al. "Increased time in range and fewer missed bolus injections after introduction of a smart connected insulin pen." Diabetes technology & therapeutics 22.10 (2020): 709-718.
}
\seealso{
\link{mod_grid}, \link{maxima_grid}, \link{find_local_maxima}, \link{detect_between_maxima}

Other GRID pipeline: 
\code{\link{detect_between_maxima}()},
\code{\link{find_local_maxima}()},
\code{\link{find_max_after_hours}()},
\code{\link{find_max_before_hours}()},
\code{\link{find_min_after_hours}()},
\code{\link{find_min_before_hours}()},
\code{\link{find_new_maxima}()},
\code{\link{maxima_grid}()},
\code{\link{mod_grid}()},
\code{\link{start_finder}()},
\code{\link{transform_df}()}
}
\concept{GRID pipeline}
