# ============================================================================
# Test scalar/atomic checks
# ============================================================================

test_that("check_int_cli returns TRUE on valid input", {
  result <- check_int_cli(5L)
  expect_true(result)
})

test_that("assert_int_cli passes silently on valid input", {
  expect_no_error(assert_int_cli(5L))
})

test_that("assert_int_cli raises cli error on invalid input", {
  expect_error(
    assert_int_cli("not an int"),
    class = "rlang_error"
  )
})

test_that("check_atomic_cli returns TRUE on atomic vector", {
  result <- check_atomic_cli(c(1, 2, 3))
  expect_true(result)
})

test_that("assert_atomic_cli raises cli error on non-atomic input", {
  expect_error(
    assert_atomic_cli(list(a = 1)),
    class = "rlang_error"
  )
})

test_that("check_scalar_cli returns TRUE on scalar", {
  result <- check_scalar_cli(42)
  expect_true(result)
})

test_that("assert_scalar_cli raises cli error on vector", {
  expect_error(
    assert_scalar_cli(c(1, 2)),
    class = "rlang_error"
  )
})

test_that("check_string_cli returns TRUE on character scalar", {
  result <- check_string_cli("hello")
  expect_true(result)
})

test_that("assert_string_cli raises cli error on non-string", {
  expect_error(
    assert_string_cli(123),
    class = "rlang_error"
  )
})

test_that("check_flag_cli returns TRUE on logical scalar", {
  result <- check_flag_cli(TRUE)
  expect_true(result)
})

test_that("assert_flag_cli raises cli error on non-flag", {
  expect_error(
    assert_flag_cli(c(TRUE, FALSE)),
    class = "rlang_error"
  )
})

test_that("check_null_cli returns TRUE on NULL", {
  result <- check_null_cli(NULL)
  expect_true(result)
})

test_that("assert_null_cli raises cli error on non-NULL", {
  expect_error(
    assert_null_cli(42),
    class = "rlang_error"
  )
})

# ============================================================================
# Test numeric variants
# ============================================================================

test_that("check_double_cli returns TRUE on double", {
  result <- check_double_cli(3.14)
  expect_true(result)
})

test_that("check_integerish_cli returns TRUE on integer-like numeric", {
  result <- check_integerish_cli(c(1.0, 2.0, 3.0))
  expect_true(result)
})

test_that("assert_integerish_cli raises cli error on non-integer-like", {
  expect_error(
    assert_integerish_cli(c(1.5, 2.3)),
    class = "rlang_error"
  )
})

test_that("check_count_cli returns TRUE on positive integer", {
  result <- check_count_cli(5L)
  expect_true(result)
})

test_that("assert_count_cli raises cli error on negative count", {
  expect_error(
    assert_count_cli(-1L),
    class = "rlang_error"
  )
})

test_that("check_number_cli returns TRUE on single number", {
  result <- check_number_cli(3.14)
  expect_true(result)
})

test_that("assert_number_cli raises cli error on vector", {
  expect_error(
    assert_number_cli(c(1, 2)),
    class = "rlang_error"
  )
})

# ============================================================================
# Test logical and complex
# ============================================================================

test_that("check_logical_cli returns TRUE on logical vector", {
  result <- check_logical_cli(c(TRUE, FALSE))
  expect_true(result)
})

test_that("assert_logical_cli raises cli error on non-logical", {
  expect_error(
    assert_logical_cli(1:3),
    class = "rlang_error"
  )
})

test_that("check_complex_cli returns TRUE on complex vector", {
  result <- check_complex_cli(complex(real = 1, imaginary = 2))
  expect_true(result)
})

test_that("assert_complex_cli raises cli error on non-complex", {
  expect_error(
    assert_complex_cli("not complex"),
    class = "rlang_error"
  )
})

# ============================================================================
# Test character and naming
# ============================================================================

test_that("check_character_cli returns TRUE on character vector", {
  result <- check_character_cli(c("a", "b", "c"))
  expect_true(result)
})

test_that("assert_character_cli raises cli error on non-character", {
  expect_error(
    assert_character_cli(1:3),
    class = "rlang_error"
  )
})

test_that("check_names_cli returns TRUE on named object", {
  result <- check_names_cli(c("a" = "test", "b" = "test"))
  expect_true(result)
})

test_that("assert_names_cli raises cli error on unnamed object", {
  expect_error(
    assert_names_cli(c(1, 2)),
    class = "rlang_error"
  )
})

# ============================================================================
# Test containers
# ============================================================================

test_that("check_vector_cli returns TRUE on vector", {
  result <- check_vector_cli(1:5)
  expect_true(result)
})

test_that("assert_vector_cli raises cli error on df", {
  expect_error(
    assert_vector_cli(as.data.frame(1), strict=T),
    class = "rlang_error"
  )
})

test_that("check_list_cli returns TRUE on list", {
  result <- check_list_cli(list(a = 1, b = 2))
  expect_true(result)
})

test_that("assert_list_cli raises cli error on vector", {
  expect_error(
    assert_list_cli(1:5),
    class = "rlang_error"
  )
})

test_that("check_data_frame_cli returns TRUE on data.frame", {
  result <- check_data_frame_cli(mtcars)
  expect_true(result)
})

test_that("assert_data_frame_cli raises cli error on non-data.frame", {
  expect_error(
    assert_data_frame_cli(list(a = 1)),
    class = "rlang_error"
  )
})

test_that("check_matrix_cli returns TRUE on matrix", {
  result <- check_matrix_cli(matrix(1:6, nrow = 2))
  expect_true(result)
})

test_that("assert_matrix_cli raises cli error on vector", {
  expect_error(
    assert_matrix_cli(1:6),
    class = "rlang_error"
  )
})

test_that("check_array_cli returns TRUE on array", {
  result <- check_array_cli(array(1:8, dim = c(2, 2, 2)))
  expect_true(result)
})

test_that("assert_array_cli raises cli error on vector", {
  expect_error(
    assert_array_cli(1:8),
    class = "rlang_error"
  )
})

test_that("check_factor_cli returns TRUE on factor", {
  result <- check_factor_cli(factor(c("a", "b", "c")))
  expect_true(result)
})

test_that("assert_factor_cli raises cli error on non-factor", {
  expect_error(
    assert_factor_cli(c("a", "b")),
    class = "rlang_error"
  )
})

test_that("check_function_cli returns TRUE on function", {
  result <- check_function_cli(function(x) x + 1)
  expect_true(result)
})

test_that("assert_function_cli raises cli error on non-function", {
  expect_error(
    assert_function_cli(42),
    class = "rlang_error"
  )
})

test_that("check_environment_cli returns TRUE on environment", {
  result <- check_environment_cli(new.env())
  expect_true(result)
})

test_that("assert_environment_cli raises cli error on list", {
  expect_error(
    assert_environment_cli(list(a = 1)),
    class = "rlang_error"
  )
})

# ============================================================================
# Test special classes
# ============================================================================

test_that("check_class_cli returns TRUE on matching class", {
  result <- check_class_cli(mtcars, "data.frame")
  expect_true(result)
})

test_that("assert_class_cli raises cli error on non-matching class", {
  expect_error(
    assert_class_cli(1:5, "data.frame"),
    class = "rlang_error"
  )
})

test_that("check_date_cli returns TRUE on Date object", {
  result <- check_date_cli(Sys.Date())
  expect_true(result)
})

test_that("assert_date_cli raises cli error on non-Date", {
  expect_error(
    assert_date_cli("2024-01-01"),
    class = "rlang_error"
  )
})

test_that("check_posixct_cli returns TRUE on POSIXct", {
  result <- check_posixct_cli(Sys.time())
  expect_true(result)
})

test_that("assert_posixct_cli raises cli error on non-POSIXct", {
  expect_error(
    assert_posixct_cli("2024-01-01 12:00:00"),
    class = "rlang_error"
  )
})

test_that("check_formula_cli returns TRUE on formula", {
  result <- check_formula_cli(y ~ x)
  expect_true(result)
})

test_that("assert_formula_cli raises cli error on non-formula", {
  expect_error(
    assert_formula_cli("y ~ x"),
    class = "rlang_error"
  )
})

test_that("check_raw_cli returns TRUE on raw vector", {
  result <- check_raw_cli(raw(5))
  expect_true(result)
})

test_that("assert_raw_cli raises cli error on non-raw", {
  expect_error(
    assert_raw_cli(1:5),
    class = "rlang_error"
  )
})

# ============================================================================
# Test set and relation helpers
# ============================================================================

test_that("check_subset_cli returns TRUE on valid subset", {
  result <- check_subset_cli(c("a", "b"), c("a", "b", "c"))
  expect_true(result)
})

test_that("assert_subset_cli raises cli error on invalid subset", {
  expect_error(
    assert_subset_cli(c("a", "d"), c("a", "b", "c")),
    class = "rlang_error"
  )
})

test_that("check_choice_cli returns TRUE on valid choice", {
  result <- check_choice_cli("a", c("a", "b", "c"))
  expect_true(result)
})

test_that("assert_choice_cli raises cli error on invalid choice", {
  expect_error(
    assert_choice_cli("d", c("a", "b", "c")),
    class = "rlang_error"
  )
})

test_that("check_set_equal_cli returns TRUE on equal sets", {
  result <- check_set_equal_cli(c("a", "b"), c("b", "a"))
  expect_true(result)
})

test_that("assert_set_equal_cli raises cli error on unequal sets", {
  expect_error(
    assert_set_equal_cli(c("a", "b"), c("a", "b", "c")),
    class = "rlang_error"
  )
})

test_that("check_disjunct_cli returns TRUE on disjoint sets", {
  result <- check_disjunct_cli(c("a", "b"), c("c", "d"))
  expect_true(result)
})

test_that("assert_disjunct_cli raises cli error on overlapping sets", {
  expect_error(
    assert_disjunct_cli(c("a", "b"), c("b", "c")),
    class = "rlang_error"
  )
})

# ============================================================================
# Test file and path helpers
# ============================================================================

test_that("assert_file_exists_cli raises cli error on missing file", {
  expect_error(
    assert_file_exists_cli("/nonexistent/file.txt"),
    class = "rlang_error"
  )
})

test_that("assert_file_exists_cli passes on existing file", {
  # Create temporary file
  tmp <- tempfile()
  file.create(tmp)
  on.exit(file.remove(tmp))

  expect_no_error(assert_file_exists_cli(tmp))
})

test_that("assert_directory_cli raises cli error on missing directory", {
  expect_error(
    assert_directory_cli("/nonexistent/directory"),
    class = "rlang_error"
  )
})

test_that("assert_directory_cli passes on existing directory", {
  tmp_dir <- tempdir()
  expect_no_error(assert_directory_cli(tmp_dir))
})

test_that("assert_directory_exists_cli raises cli error on missing directory", {
  expect_error(
    assert_directory_exists_cli("/nonexistent/directory"),
    class = "rlang_error"
  )
})

test_that("assert_directory_exists_cli passes on existing directory", {
  tmp_dir <- tempdir()
  expect_no_error(assert_directory_exists_cli(tmp_dir))
})

# ============================================================================
# Test boolean assertions
# ============================================================================

test_that("assert_true_cli raises cli error on FALSE", {
  expect_error(
    assert_true_cli(FALSE),
    class = "rlang_error"
  )
})

test_that("assert_true_cli passes on TRUE", {
  expect_no_error(assert_true_cli(TRUE))
})

test_that("assert_false_cli raises cli error on TRUE", {
  expect_error(
    assert_false_cli(TRUE),
    class = "rlang_error"
  )
})

test_that("assert_false_cli passes on FALSE", {
  expect_no_error(assert_false_cli(FALSE))
})

# ============================================================================
# Test make_assertion helper
# ============================================================================

test_that("make_assertion aborts immediately without collection", {
  expect_error(
    make_assertion(5L, "some error", "x", NULL),
    class = "rlang_error"
  )
})

test_that("make_assertion pushes to collection with add parameter", {
  collection <- checkmate::makeAssertCollection()

  make_assertion(5L, "first error", "x", collection)
  make_assertion(10L, "second error", "y", collection)

  expect_equal(length(collection$getMessages()), 2)
  expect_match(collection$getMessages()[1], "first error")
  expect_match(collection$getMessages()[2], "second error")
})

# ============================================================================
# Test assert_cli combining function
# ============================================================================

test_that("assert_cli 'or' mode passes if any check succeeds", {
  expect_no_error(
    assert_cli(
      checkmate::check_int(5L),
      checkmate::check_character("hello"),
      combine = "or"
    )
  )
})

test_that("assert_cli 'or' mode fails if all checks fail", {
  expect_error(
    assert_cli(
      checkmate::check_int("not int"),
      checkmate::check_character(123),
      combine = "or",
      .var.name = c("arg1", "arg2")
    ),
    class = "rlang_error"
  )
})

test_that("assert_cli 'and' mode passes if all checks succeed", {
  expect_no_error(
    assert_cli(
      checkmate::check_int(5L),
      checkmate::check_number(5.0),
      combine = "and"
    )
  )
})

test_that("assert_cli 'and' mode fails if any check fails", {
  expect_error(
    assert_cli(
      checkmate::check_int(5L),
      checkmate::check_character(123),
      combine = "and",
      .var.name = c("arg1", "arg2")
    ),
    class = "rlang_error"
  )
})

# ============================================================================
# Test helper functions
# ============================================================================

test_that("sanitize_cli returns TRUE unchanged", {
  result <- sanitize_cli(TRUE)
  expect_true(result)
})

test_that("sanitize_cli escapes braces in error strings", {
  result <- sanitize_cli("Error in {var}")
  expect_equal(result, "Error in {{var}}")
})

test_that("fmt_bullet_cli returns TRUE unchanged", {
  result <- fmt_bullet_cli(TRUE)
  expect_true(result)
})

test_that("fmt_bullet_cli formats error string with bullet names", {
  result <- fmt_bullet_cli("Error message")
  expect_named(result, "i")
  expect_equal(as.character(result), "Error message")
})

# ============================================================================
# Test multi-class checks
# ============================================================================

test_that("check_multi_class_cli returns TRUE on matching class", {
  x <- data.frame(a = 1)
  result <- check_multi_class_cli(x, c("data.frame", "list"))
  expect_true(result)
})

test_that("assert_multi_class_cli raises cli error on non-matching class", {
  expect_error(
    assert_multi_class_cli(1:5, c("data.frame", "list")),
    class = "rlang_error"
  )
})

# ============================================================================
# Test permutation check
# ============================================================================

test_that("check_permutation_cli returns TRUE on valid permutation", {
  result <- check_permutation_cli(c(2, 1, 3), c(1, 2, 3))
  expect_true(result)
})

test_that("assert_permutation_cli raises cli error on invalid permutation", {
  expect_error(
    assert_permutation_cli(c(1, 2, 4), c(1, 2, 3)),
    class = "rlang_error"
  )
})
