% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cggm_cv.R
\name{cggm_cv}
\alias{cggm_cv}
\title{Cross Validation for the Clusterpath Estimator of the Gaussian Graphical Model}
\usage{
cggm_cv(
  X,
  tune_grid,
  kfold = 5,
  folds = NULL,
  connected = TRUE,
  fit = TRUE,
  refit = TRUE,
  lasso_unit_weights = FALSE,
  estimate_Sigma = FALSE,
  verbose = 0,
  n_jobs = 1,
  ...
)
}
\arguments{
\item{X}{The \code{n} times \code{p} matrix holding the data, with \code{n}
observations and \code{p} variables.}

\item{tune_grid}{A data frame with values of the tuning parameters. Each row
is a combination of parameters that is evaluated. The columns have the names
of the tuning parameters and should include \code{k} and \code{phi}. The
sparsity parameter \code{lambda_lasso} and the aggregation parameter
\code{lambda} are optional. If there is no column named \code{lambda_lasso},
the sparsity parameter is set to 0. If there is no column named
\code{lambda}, an appropriate range for the aggregation parameter is
selected for each combination of \code{k}, \code{phi}, and
\code{lambda_lasso}.}

\item{kfold}{The number of folds. Defaults to 5.}

\item{folds}{Optional argument to manually set the folds for the cross
validation procedure. If this is not \code{NULL}, it overrides the
\code{kfold} argument. Defaults to \code{NULL}.}

\item{connected}{Logical, indicating whether connectedness of the weight
matrix should be ensured. Defaults to \code{TRUE}. See
\code{\link{clusterpath_weights}()}.}

\item{fit}{Logical, indicating whether the cross-validation procedure
should consider the result from \code{\link{cggm}()}, before refitting is
applied. Defaults to \code{TRUE}. At least one of \code{fit} and \code{refit}
should be \code{TRUE}.}

\item{refit}{Logical, indicating whether the cross-validation procedure
should also consider the refitted result from \code{\link{cggm}()}. See also
\code{\link{cggm_refit}()}. Defaults to \code{TRUE}. At least one of
\code{fit} and \code{refit} should be \code{TRUE}.}

\item{lasso_unit_weights}{Logical, indicating whether the weights in the
sparsity penalty should be all one or decreasing in the magnitude of the
corresponding element of the inverse of the sample covariance matrix.
Defaults to \code{FALSE}.}

\item{estimate_Sigma}{Logical, indicating whether CGGM should be used to
estimate the covariance matrix based on the sample precision matrix. Defaults
to \code{FALSE}.}

\item{verbose}{Determines the amount of information printed during the
cross validation. Defaults to \code{0}.}

\item{n_jobs}{Number of parallel jobs used for cross validation. If 0 or
smaller, uses the maximum available number of physical cores. Defaults to
\code{1} (sequential).}

\item{...}{Additional arguments to be passed down to \code{\link{cggm}()}
and \code{\link{cggm_refit}()}.}
}
\value{
An object of class \code{"CGGM_CV"} with the following components:
\item{fit}{A list with cross-validation results for CGGM without the
refitting step. It consists of four components:
\itemize{
\item{\code{final}} (an object of class \code{"CGGM"} corresponding to the
final model fit using the optimal values of the tuning parameters; see
\code{\link{cggm}()})
\item{\code{scores}} (a data frame containing the values of the tuning
parameters and the corresponding cross-validation scores)
\item{\code{opt_index}} (the index of the optimal aggregation parameter
\code{lambda_cpath} in the final model fit)
\item{\code{opt_tune}} (a data frame containing the values of the tuning
parameters)
}}
\item{refit}{A list with cross-validation results for CGGM including the
refitting step. It contains the same four components as above, except that
\code{final} is an object of class \code{"CGGM_refit"} (see
\code{\link{cggm_refit}()}).}
\item{raw_cv_results}{A list of raw cross-validation results before
restructuring.}
\item{best}{A character string indicating whether the optimal model fit
without the refitting step (\code{"fit"}) or including the refitting step
("refit") has a better cross-validation score.}
}
\description{
Perform cross validation to tune the weight matrix parameters \code{phi}
and \code{k} (for k-nearest-neighbors) as well as the aggregation parameter
\code{lambda_cpath} and the sparsity parameter \code{lambda_lasso} of the
clusterpath estimator of the Gaussian Graphical Model (CGGM) in order to
obtain a sparse estimate with variable clustering of the precision matrix or
the covariance matrix. The scoring metric is the negative log-likelihood
(lower is better).
}
\note{
The function interface and output structure are still experimental and
may change in the next version.
}
\examples{
\donttest{
# Generate data
set.seed(3)
Theta <- matrix(
  c(2, 1, 0, 0,
    1, 2, 0, 0,
    0, 0, 4, 1,
    0, 0, 1, 4),
  nrow = 4
)
X <- mvtnorm::rmvnorm(n = 100, sigma = solve(Theta))

# Use cross-validation to select the tuning parameters
fit_cv <- cggm_cv(
  X = X,
  tune_grid = expand.grid(
    phi = 1,
    k = 2,
    lambda_lasso = c(0, 0.02),
    lambda = seq(0, 0.2, by = 0.01)
  ),
  folds = cv_folds(nrow(X), 5)
)

# The best solution has 2 clusters
get_Theta(fit_cv)
get_clusters(fit_cv)
}

}
\references{
D.J.W. Touw, A. Alfons, P.J.F. Groenen and I. Wilms (2025)
\emph{Clusterpath Gaussian Graphical Modeling}. arXiv:2407.00644.
doi:10.48550/arXiv.2407.00644.
}
\seealso{
\code{\link{clusterpath_weights}()}, \code{\link{lasso_weights}()},
\code{\link{cggm}()}, \code{\link{cggm_refit}()}
}
\author{
Daniel J.W. Touw, modifications by Andreas Alfons
}
