% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/crplot.R
\name{crplot}
\alias{crplot}
\title{Plotting Two-Dimensional Confidence Regions}
\usage{
crplot(dataset, alpha, distn,
                cen       = rep(1, length(dataset)),
                heuristic = 1,
                maxdeg    = 5,
                ellipse_n = 4,
                pts       = TRUE,
                mlelab    = TRUE,
                sf        = NULL,
                mar       = c(4, 4.5, 2, 1.5),
                xyswap    = FALSE,
                xlab      = "",
                ylab      = "",
                main      = "",
                xlas      = 0,
                ylas      = 0,
                origin    = FALSE,
                xlim      = NULL,
                ylim      = NULL,
                tol       = .Machine$double.eps ^ 1,
                info      = FALSE,
                maxcount  = 30,
                repair    = TRUE,
                jumpshift = 0.5,
                jumpuphill = min(alpha, 0.01),
                jumpinfo  = FALSE,
                showjump  = FALSE,
                showplot  = TRUE,
                animate   = FALSE,
                delay     = 0.5,
                exact     = FALSE,
                silent    = FALSE )
}
\arguments{
\item{dataset}{a vector of n data values.}

\item{alpha}{significance level; resulting plot illustrates a 100(1 - \code{alpha})\% confidence region.}

\item{distn}{distribution to fit the dataset to; accepted values: \code{'cauchy'}, \code{'gamma'}, \code{'invgauss'},
\code{'logis'}, \code{'llogis'}, \code{'lnorm'}, \code{'norm'}, \code{'unif'}, \code{'weibull'}.}

\item{cen}{a vector of binary values specifying if the corresponding data values are right-censored (0), or
observed (1, default); its length must match length(dataset).}

\item{heuristic}{numeric value selecting method for plotting: 0 for elliptic-oriented point distribution, and
1 for smoothing boundary search heuristic.}

\item{maxdeg}{maximum angle tolerance between consecutive plot segments in degrees.}

\item{ellipse_n}{number of roughly equidistant confidence region points to plot using the
elliptic-oriented point distribution (must be a multiple of four because its algorithm
exploits symmetry in the quadrants of an ellipse).}

\item{pts}{displays confidence region boundary points identified if \code{TRUE}.}

\item{mlelab}{logical argument to include the maximum
likelihood estimate coordinate point (default is \code{TRUE}).}

\item{sf}{significant figures in axes labels specified using sf = c(x, y), where x and y represent the optional
digits argument in the R function \code{\link{round}} as it pertains to the horizontal and vertical labels.}

\item{mar}{specifies margin values for \code{par(mar = c( ))} (see \code{mar} in \code{\link{par}}).}

\item{xyswap}{logical argument to switch the axes that the distribution parameter are shown.}

\item{xlab}{string specifying the x axis label.}

\item{ylab}{string specifying the y axis label.}

\item{main}{string specifying the plot title.}

\item{xlas}{numeric value of 0, 1, 2, or 3 specifying the style of axis labels (see \code{las} in \code{\link{par}}).}

\item{ylas}{numeric value of 0, 1, 2, or 3 specifying the style of axis labels (see \code{las} in \code{\link{par}}).}

\item{origin}{logical argument to include the plot origin (default is \code{FALSE}).}

\item{xlim}{two-element vector containing horizontal axis minimum and maximum values.}

\item{ylim}{two-element vector containing vertical axis minimum and maximum values.}

\item{tol}{the \code{\link{uniroot}} parameter specifying its required accuracy.}

\item{info}{logical argument to return plot information: MLE is returned as a list; (x, y) plot point coordinates
and corresponding phi angles (with respect to MLE) are returned as a list.}

\item{maxcount}{integer value specifying the number of smoothing search iterations before terminating with \code{maxdeg} not met.}

\item{repair}{logical argument to repair regions inaccessible using a radial angle from its MLE due to multiple
roots at select \eqn{\phi} angles.}

\item{jumpshift}{see vignette "conf Advanced Options" for details; location (as a fractional value between 0 and 1) along the
vertical or horizontal "gap" (near an uncharted region) to locate a jump-center toward; can be either a scalar value (uniformly
applied to all jump-centers) or vector of length four (with unique values for its respective quadrants, relative to the MLE).}

\item{jumpuphill}{see vignette "conf Advanced Options" for details; significance level increase to \code{alpha} for the jump-center
(corresponds to an "uphill" location on its likelihood function); can be either a scalar value (uniformly applied to all jump-centers)
or vector of length four (with unique values for its respective quadrants, relative to the MLE).}

\item{jumpinfo}{logical argument to return plot info (see \code{info} argument) and jump-center info; returned within `repair`
attribute are \code{jumpuphill} value, \code{jumpshift} value, "|" or "-" gap type, jump-center(s) coordinates, and coordinates
of points left & right of the inaccessible region.}

\item{showjump}{logical argument specifying if jump-center repair reference points appear on the confidence region plot.}

\item{showplot}{logical argument specifying if a plot is output; altering from its default of \code{TRUE} is
only logical assuming \code{crplot} is run for its data only (see the \code{info} argument).}

\item{animate}{logical argument specifying if an animated plot build will display; the annimation sequence is given in successive plots.}

\item{delay}{numeric value of delay (in seconds) between successive plots when \code{animate = TRUE}.}

\item{exact}{logical argument specifying if alpha value is adjusted to compensate for negative coverage bias to achieve
(1 - alpha) coverage probability using previously recorded Monte Carlo simulation results; available for limited values of
alpha (roughly <= 0.2--0.3), n (typically n = 4, 5, ..., 50) and distributions (distn suffixes: weibull, llogis, norm).}

\item{silent}{logical argument specifying if console output should be suppressed.}
}
\value{
If the optional argument \code{info = TRUE} is included then a list is returned with:
\itemize{
\item parm1*: a vector containing the associated confidence region boundary values for parameter 1
\item parm2*: a vector containing the associated confidence region boundary values for parameter 2
\item phi: a vector containing the angles used
\item parm1hat*: the MLE for parameter 1
\item parm2hat*: the MLE for parameter 2
}
*Note: "param1" and "param2" are placeholders that will be replaced with the appropriate parameter names
based on the probability distribution.
}
\description{
Plotting a two-dimensional confidence region for probability distribution parameters (supported distribution
suffixes: cauchy, gamma, invgauss, lnorm, llogis, logis, norm, unif, weibull) corresponding to a user given
complete or right-censored dataset and level of significance.  See the CRAN website
https://CRAN.R-project.org/package=conf for a link to two \code{crplot} vignettes.
}
\details{
This function plots a confidence region for a variety of two-parameter distributions.  It requires:
\itemize{
\item a vector of dataset values,
\item the level of significance (alpha), and
\item a population distribution to fit the data to.
}
Plots display according to probability density function parameterization given later in this section.
Two heuristics (and their associated combination) are available to plot confidence regions.  Along
with their descriptions, they are:
\enumerate{
\item \emph{Smoothing Boundary Search Heuristic (default)}.  This heuristic plots more points in areas of
greater curvature to ensure a smooth appearance throughout the confidence region boundary.  Its
\code{maxdeg} parameter specifies the maximum tolerable angle between three successive points.
Lower values of \code{maxdeg} result in smoother plots, and its default value of 5 degrees
provides adequate smoothing in most circumstances.  Values of \code{maxdeg} \eqn{\le} 3 are not
recommended due to their complicating implications to trigonometric numerical approximations near 0
and 1; their use may result in plot errors.
\item \emph{Elliptic-Oriented Point Distribution}.  This heuristic allows the user to specify
a number of points to plot along the confidence region boundary at roughly uniform intervals.
Its name is derived from the technique it uses to choose these points---an extension of the Steiner
generation of a non-degenerate conic section, also known as the parallelogram method---which identifies
points along an ellipse that are approximately equidistant.  To exploit the computational benefits of
ellipse symmetry over its four quadrants, \code{ellipse_n} value must be divisible by four.}
By default, \code{crplot} implements the smoothing boundary search heuristic.  Alternatively,
the user can plot using the elliptic-oriented point distribution algorithm, or a combination
of them both.  Combining the two techniques initializes the plot using the elliptic-oriented point
distribution algorithm, and then subsequently populates additional points in areas of high curvature
(those outside of the maximum angle tolerance parameterization) in accordance with the smoothing
boundary search heuristic.  This combination results when the smoothing boundary search heuristic
is specified in conjunction with an \code{ellipse_n} value greater than four.

Both of the aforementioned heuristics use a radial profile log likelihood function to identify
points along the confidence region boundary.  It cuts the log likelihood function in a directional
azimuth from its MLE, and locates the associated confidence region boundary point using the
asymptotic results associated with the ratio test statistic \eqn{-2 [\log L(\theta) - \log L(\hat{\theta})]}
which converges in distribution to the chi-square distribution with two degrees of freedom (for
a two parameter distribution).

The default axes convention in use by \code{crplot} are

\tabular{lcc}{
\tab Horizontal \tab Vertical\cr
Distribution  \tab  Axis  \tab Axis\cr
Cauchy \tab \eqn{a} \tab \eqn{s}\cr
gamma \tab \eqn{\theta} \tab \eqn{\kappa}\cr
inverse Gaussian \tab \eqn{\mu} \tab \eqn{\lambda}\cr
log logistic \tab \eqn{\lambda} \tab \eqn{\kappa}\cr
log normal \tab \eqn{\mu} \tab \eqn{\sigma}\cr
logistic \tab \eqn{\mu} \tab \eqn{\sigma}\cr
normal \tab \eqn{\mu} \tab \eqn{\sigma}\cr
uniform \tab \eqn{a} \tab \eqn{b}\cr
Weibull \tab \eqn{\kappa} \tab \eqn{\lambda}
}

where each respective distribution is defined below.

\itemize{
\item The Cauchy distribution
for the real-numbered location parameter \eqn{a}, scale parameter \eqn{s}, and \eqn{x} is a real number,
has the probability density function
\deqn{1 / (s \pi (1 + ((x - a) / s) ^ 2)).}

\item The gamma distribution
for shape parameter \eqn{\kappa > 0}, scale parameter \eqn{\theta > 0}, and \eqn{x > 0},
has the probability density function
\deqn{1 / (Gamma(\kappa) \theta ^ \kappa) x ^ {(\kappa - 1)} exp(-x / \theta).}

\item The inverse Gaussian distribution
for mean \eqn{\mu > 0}, shape parameter \eqn{\lambda > 0}, and \eqn{x > 0},
has the probability density function
\deqn{\sqrt (\lambda / (2 \pi x ^ 3)) exp( - \lambda (x - \mu) ^ 2 / (2 \mu ^ 2 x)).}

\item The log logistic distribution
for scale parameter \eqn{\lambda > 0}, shape parameter \eqn{\kappa > 0}, and \eqn{x \ge 0},
has a probability density function
\deqn{(\kappa \lambda) (x \lambda) ^ {(\kappa - 1)} / (1 + (\lambda x) ^ \kappa) ^ 2.}

\item The log normal distribution
for the real-numbered mean \eqn{\mu} of the logarithm, standard deviation \eqn{\sigma > 0}
of the logarithm, and \eqn{x > 0},
has the probability density function
\deqn{1 / (x \sigma \sqrt(2 \pi)) exp(-(\log x - \mu) ^ 2 / (2 \sigma ^ 2)).}

\item The logistic distribution
for the real-numbered location parameter \eqn{\mu}, scale parameter \eqn{\sigma}, and \eqn{x} is a real number,
has the probability density function
\deqn{(1 / \sigma) exp((x - \mu) / \sigma) (1 + exp((x - \mu) / \sigma)) ^ {-2}}

\item The normal distribution
for the real-numbered mean \eqn{\mu}, standard deviation \eqn{\sigma > 0}, and \eqn{x} is a real number,
has the probability density function
\deqn{1 / \sqrt (2 \pi \sigma ^ 2) exp(-(x - \mu) ^ 2 / (2 \sigma ^ 2)).}

\item The uniform distribution for real-valued parameters \eqn{a} and \eqn{b} where \eqn{a < b}
and \eqn{a \le x \le b},
has the probability density function
\deqn{1 / (b - a).}

\item The Weibull distribution
for scale parameter \eqn{\lambda > 0}, shape parameter \eqn{\kappa > 0}, and \eqn{x > 0},
has the probability density function
\deqn{\kappa (\lambda ^ \kappa) x ^ {(\kappa - 1)} exp(-(\lambda x) ^ \kappa).}
}
}
\examples{
## plot the 95\% confidence region for Weibull shape and scale parameters
## corresponding to the given ballbearing dataset
ballbearing <- c(17.88, 28.92, 33.00, 41.52, 42.12, 45.60, 48.48, 51.84,
                 51.96, 54.12, 55.56, 67.80, 68.64, 68.64, 68.88, 84.12,
                 93.12, 98.64, 105.12, 105.84, 127.92, 128.04, 173.40)
crplot(dataset = ballbearing, distn = "weibull", alpha = 0.05)

## repeat this plot using the elliptic-oriented point distribution heuristic
crplot(dataset = ballbearing, distn = "weibull", alpha = 0.05,
       heuristic = 0, ellipse_n = 80)

## combine the two heuristics, compensating any elliptic-oriented point verticies whose apparent
## angles > 6 degrees with additional points, and expand the plot area to include the origin
crplot(dataset = ballbearing, distn = "weibull", alpha = 0.05,
       maxdeg = 6, ellipse_n = 80, origin = TRUE)

## next use the inverse Gaussian distribution and show no plot points
crplot(dataset = ballbearing, distn = "invgauss", alpha = 0.05,
       pts = FALSE)
}
\references{
A. Jaeger (2016), "Computation of Two- and Three-Dimensional Confidence Regions with the Likelihood Ratio",
The American Statistician, 49, 48--53.

C. Weld, A. Loh, L. Leemis (2020), "Plotting Two-Dimensional Confidence Regions",
The American Statistician, Volume 72, Number 2, 156--168.
}
\seealso{
\code{\link{coversim}}, \code{\link{uniroot}}
}
\author{
Christopher Weld (\email{ceweld241@gmail.com})

Lawrence Leemis (\email{leemis@math.wm.edu})
}
\concept{confidence region plot graphics visualization coverage parameter estimation}
\keyword{distribution}
\keyword{models}
\keyword{univar}
