% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/generator.R
\name{generator}
\alias{generator}
\alias{gen}
\title{Create a generator function}
\usage{
generator(fn)

gen(expr)
}
\arguments{
\item{fn}{A function template for generators. The function can
\code{\link[=yield]{yield()}} values. Within a generator, \code{for} loops have
\link{iterator} support.}

\item{expr}{A yielding expression.}
}
\description{
\code{generator()} creates an generator factory. A generator is an
\link[=iterator]{iterator function} that can pause its execution with
\code{\link[=yield]{yield()}} and resume from where it left off. Because they manage
state for you, generators are the easiest way to create
iterators. See \code{vignette("generator")}.

The following rules apply:
\itemize{
\item Yielded values do not terminate the generator. If you call the
generator again, the execution resumes right after the yielding
point. All local variables are preserved.
\item Returned values terminate the generator. If called again after a
\code{return()}, the generator keeps returning the \code{\link[=exhausted]{exhausted()}}
sentinel.
}

Generators are compatible with all features based on the iterator
protocol such as \code{\link[=loop]{loop()}} and \code{\link[=collect]{collect()}}.
}
\examples{
# A generator statement creates a generator factory. The
# following generator yields three times and then returns `"d"`.
# Only the yielded values are visible to the callers.
generate_abc <- generator(function() {
  yield("a")
  yield("b")
  yield("c")
  "d"
})

# Equivalently:
generate_abc <- generator(function() {
  for (x in c("a", "b", "c")) {
    yield(x)
  }
})

# The factory creates generator instances. They are iterators
# that you can call successively to obtain new values:
abc <- generate_abc()
abc()
abc()

# Once a generator has returned it keeps returning `exhausted()`.
# This signals to its caller that new values can no longer be
# produced. The generator is exhausted:
abc()
abc()

# You can only exhaust a generator once but you can always create
# new ones from a factory:
abc <- generate_abc()
abc()


# As generators implement the coro iteration protocol, you can use
# coro tools like `loop()`. It makes it possible to loop over
# iterators with `for` expressions:
loop(for (x in abc) print(x))

# To gather values of an iterator in a list, use `collect()`. Pass
# the `n` argument to collect that number of elements from a
# generator:
abc <- generate_abc()
collect(abc, 1)

# Or drain all remaining elements:
collect(abc)


# coro provides a short syntax `gen()` for creating one-off
# generator _instances_. It is handy to adapt existing iterators:
numbers <- 1:10
odds <- gen(for (x in numbers) if (x \%\% 2 != 0) yield(x))
squares <- gen(for (x in odds) yield(x^2))
greetings <- gen(for (x in squares) yield(paste("Hey", x)))

collect(greetings)


# Arguments passed to generator instances are returned from the
# `yield()` statement on reentry:
new_tally <- generator(function() {
  count <- 0
  while (TRUE) {
    i <- yield(count)
    count <- count + i
  }
})
tally <- new_tally()
tally(1)
tally(2)
tally(10)
}
\seealso{
\code{\link[=yield]{yield()}}, \code{\link[=coro_debug]{coro_debug()}} for step-debugging.
}
