#' Generates the dataframe requires for the tufte lines plot component in raw plots.
#'
#' This function returns a dataframe with summary statistics, such as mean, standard
#' deviation and quantiles (depending on the effect size) based on the input dataset.
#'
#' @param raw_data The tidy dataset passed to [load()] that was cleaned and altered for plotting.
#' @param enquo_x Quosure of x as initially passed to [load()].
#' @param enquo_y Quosure of y as initially passed to [load()].
#' @param proportional Boolean value as initially passed to [load()].
#' @param gap Integer value specifying the amount of gap for each tufte line.
#' @param effsize_type String value specifying type of effect size.
#'
#' @returns a dataframe for plotting of the tufte lines plot component.
#' @noRd
create_df_for_tufte <- function(raw_data, enquo_x, enquo_y, proportional, gap, effsize_type) {
  # Input validation
  if (!is.logical(proportional)) {
    stop("'proportional' must be a logical value.")
  }

  if (!is.numeric(gap)) {
    stop("'gap' must be a numeric value.")
  }

  # Compute summary statistics
  tufte_lines_df <- raw_data %>%
    dplyr::group_by(!!enquo_x) %>%
    dplyr::summarize(
      mean = mean(!!enquo_y),
      median = stats::median(!!enquo_y),
      sd = stats::sd(!!enquo_y),
      lower_quartile = stats::quantile(!!enquo_y, names = FALSE)[2],
      upper_quartile = stats::quantile(!!enquo_y, names = FALSE)[4]
    )

  # Adjust SD if proportional is TRUE
  if (proportional) {
    tufte_lines_df$sd <- tufte_lines_df$sd / 7
  }

  # Compute lower and upper SD
  tufte_lines_df <- tufte_lines_df %>%
    dplyr::mutate(
      lower_sd = mean - sd,
      upper_sd = mean + sd
    )

  # Compute additional columns based on effsize_type
  if (!is.null(effsize_type) && grepl("edian", effsize_type, ignore.case = TRUE)) {
    tufte_lines_df <- tufte_lines_df %>%
      dplyr::mutate(no_diff = (sd == 0)) %>%
      dplyr::mutate(
        y_top_start = dplyr::case_when(no_diff ~ NA, !no_diff ~ median + gap),
        y_top_end = dplyr::case_when(no_diff ~ NA, !no_diff ~ upper_quartile),
        y_bot_start = dplyr::case_when(no_diff ~ NA, !no_diff ~ median - gap),
        y_bot_end = dplyr::case_when(no_diff ~ NA, !no_diff ~ lower_quartile)
      )
  } else {
    tufte_lines_df <- tufte_lines_df %>%
      dplyr::mutate(no_diff = (sd == 0)) %>%
      dplyr::mutate(
        y_top_start = dplyr::case_when(no_diff ~ NA, !no_diff ~ mean + gap),
        y_top_end = dplyr::case_when(no_diff ~ NA, !no_diff ~ upper_sd),
        y_bot_start = dplyr::case_when(no_diff ~ NA, !no_diff ~ mean - gap),
        y_bot_end = dplyr::case_when(no_diff ~ NA, !no_diff ~ lower_sd)
      )
  }

  return(tufte_lines_df)
}

#' Generates df for tufte lines plot component for raw plot WITH flow = FALSE.
#'
#' This function rearranges and duplicates rows in a given dataframe (containing
#' summary statistics) in the order of the unlisted given list of vectors.
#'
#' @param idx List of vectors of control-test groupings that determines the arrangement
#' of the final dataframe output.
#' @param tufte_lines_df dataframe for plotting of the tufte lines plot component as
#' generated by [create_df_for_tufte()].
#' @param enquo_x Quosure of x as initially passed to [load()].
#'
#' @returns dataframe
#' @noRd
create_dfs_for_nonflow_tufte_lines <- function(idx,
                                               tufte_lines_df,
                                               enquo_x) {
  # Input validation
  if (is.null(idx)) {
    cli::cli_abort(c("Column {.field idx} is currently NULL.",
      "x" = "Please enter a valid entry for {.field idx}"
    ))
  }
  if (!is.data.frame(tufte_lines_df)) {
    cli::cli_abort(c("Column {.field tufte_lines_df} is not a data frame",
      "x" = "Please enter a valid entry for {.field tufte_lines_df}"
    ))
  }

  new_tufte_lines_df <- tibble::tibble()
  total_length <- length(unlist(idx))
  temp_idx <- unlist(idx)
  for (i in (1:total_length)) {
    group_name <- temp_idx[i]
    row <- tufte_lines_df %>%
      dplyr::filter(!!enquo_x == group_name)
    new_tufte_lines_df <- dplyr::bind_rows(new_tufte_lines_df, row)
  }
  return(new_tufte_lines_df)
}

#' Generates df for sankey plot component for raw plot WITH flow = TRUE
#'
#' This function generates data frames necessary for generating
#' sankey diagrams for visualizing flows and proportions of categorical data.
#'
#'
#' @param float_contrast Boolean value determining if a Gardner-Altman plot or Cumming estimation plot will be produced.
#' @param raw_data The tidy dataset passed to [load()] that was cleaned and altered for plotting.
#' @param proportional_data List of calculations related to the plotting of proportion plots.
#' @param enquo_x Quosure of x as initially passed to [load()].
#' @param enquo_y Quosure of y as initially passed to [load()].
#' @param enquo_id_col Quosure of id_col as initially passed to [load()].
#' @param idx List of vectors of control-test groupings that determines the arrangement
#' of the final dataframe output.
#' @param scale_factor_sig Numeric value determining the significance of the scale factor.
#' @param bar_width Numeric value determining the width of the bar in the sankey diagram.
#' @param gap Integer value specifying the amount of gap for each tufte line.
#' @param sankey Boolean value determining if the flows between the bar charts will be plotted.
#' @param flow Boolean value determining whether the bars will be plotted in pairs.
#' @param N Total number of observations or data points in the dataset.
#'
#' @returns dataframe
#' @noRd
create_dfs_for_sankey <- function(
    float_contrast = FALSE,
    raw_data,
    proportional_data,
    enquo_x,
    enquo_y,
    enquo_id_col,
    idx,
    scale_factor_sig = 0.8,
    bar_width = 0.15,
    gap,
    sankey = TRUE,
    flow = TRUE,
    N) {
  type <- ifelse(length(unlist(idx)) <= 2,
    "single sankey",
    "multiple sankeys"
  )

  flow_success_to_failure <- tibble::tibble()
  flow_success_to_success <- tibble::tibble()
  flow_failure_to_success <- tibble::tibble()
  flow_failure_to_failure <- tibble::tibble()

  bar_width <- ifelse(float_contrast, 0.15, 0.03)

  scale_factor_sig <- switch(type,
    "single sankey" = if (float_contrast) 0.72 else 0.95,
    "multiple sankeys" = 0.92
  )

  x_padding <- ifelse(float_contrast, 0.008, 0.006)

  ind <- 1
  x_start <- 1

  sankey_bars <- if (flow) {
    proportional_data
  } else {
    create_sankey_bars(proportional_data, enquo_x, enquo_y, idx)
  }

  means_c_t <- sankey_bars$proportion_success
  if (sankey) {
    sankey_flows <- create_sankey_flows(
      raw_data,
      enquo_x,
      enquo_y,
      enquo_id_col,
      idx,
      N,
      means_c_t,
      gap,
      bar_width,
      x_padding,
      scale_factor_sig
    )

    flow_success_to_failure <- sankey_flows$flow_success_to_failure
    flow_success_to_success <- sankey_flows$flow_success_to_success
    flow_failure_to_success <- sankey_flows$flow_failure_to_success
    flow_failure_to_failure <- sankey_flows$flow_failure_to_failure
  } else {
    flow_success_to_failure <- data.frame(x = NaN, y = NaN, tag = NaN)
    flow_failure_to_success <- data.frame(x = NaN, y = NaN, tag = NaN)
    flow_success_to_success <- data.frame(x = NaN, y = NaN, tag = NaN)
    flow_failure_to_failure <- data.frame(x = NaN, y = NaN, tag = NaN)
  }

  redraw_x_axis <- seq_along(unlist(idx))
  dfs_for_sankeys <- list(
    flow_success_to_failure = flow_success_to_failure,
    flow_failure_to_success = flow_failure_to_success,
    flow_success_to_success = flow_success_to_success,
    flow_failure_to_failure = flow_failure_to_failure,
    sankey_bars = sankey_bars,
    redraw_x_axis = redraw_x_axis
  )

  return(dfs_for_sankeys)
}

#' Generates df for xaxis redraw for float_contrast = FALSE plot
#'
#' This function generates data frames that
#' define the positions of lines and ticks on an x-axis for when float_contrast = FALSE
#' to produce a Cumming estimation plot.
#'
#' @param idx List of vectors of control-test groupings that determines the arrangement
#' of the final dataframe output.
#'
#' @returns dataframe
#' @noRd
create_dfs_for_xaxis_redraw <- function(idx) {
  x_axis_pointer <- 0
  xaxis_line_x_vector <- c()
  xaxis_line_xend_vector <- c()
  xaxis_ticks_x_vector <- c()

  for (j in 1:length(idx)) {
    # calculate xaxis line x coords
    x_coord <- x_axis_pointer + 1
    xaxis_line_x_vector <- append(xaxis_line_x_vector, x_coord)
    xend_coord <- x_axis_pointer + length(idx[[j]])
    xaxis_line_xend_vector <- append(xaxis_line_xend_vector, xend_coord)

    # calculate xaxis ticks x coords
    for (k in 1:length(idx[[j]])) {
      x_coord <- x_axis_pointer + k
      xaxis_ticks_x_vector <- append(xaxis_ticks_x_vector, x_coord)
    }
    x_axis_pointer <- x_axis_pointer + length(idx[[j]])
  }

  dfs_for_xaxis_redraw <- list(
    df_for_line = data.frame(
      x = xaxis_line_x_vector,
      xend = xaxis_line_xend_vector
    ),
    df_for_ticks = data.frame(x = xaxis_ticks_x_vector)
  )

  return(dfs_for_xaxis_redraw)
}

#' Generates df for proportion bar component
#'
#' This function generates data frames to
#' represent bars with proportional data in a graphical display.
#' @param proportion_success Percentage value of dataset with success
#' @param bar_width Numeric value determining the width of the bar in the sankey diagram.
#' @param gap Integer value specifying the amount of gap for each tufte line.
#'
#' @returns dataframe
#' @noRd
create_dfs_for_proportion_bar <- function(proportion_success, bar_width = 0.3, gap = 0) {
  df_for_proportion_bar <- data.frame(
    x_failure = NA,
    y_failure = NA,
    x_success = NA,
    y_success = NA,
    tag = NA
  )

  for (x in 1:length(proportion_success)) {
    y <- proportion_success[x]
    if ((y > 1) || (y < 0)) {
      cli::cli_abort(c("Proportion plots must be supplied with data of values between 0 and 1."))
    }

    x_failure_success <- c(x - bar_width / 2, x + bar_width / 2, x + bar_width / 2, x - bar_width / 2)
    y_success <- c(y - gap / 2, y - gap / 2, 0, 0)
    y_failure <- c(1, 1, y + gap / 2, y + gap / 2)

    # For treatment groups with all 1s/all 0s
    if (y == 0) {
      y_failure <- c(1, 1, 0, 0)
      y_success <- c(0, 0, 0, 0)
    }

    if (y == 1) {
      y_success <- c(1, 1, 0, 0)
      y_failure <- c(0, 0, 0, 0)
    }

    temp_df_proportion_bar <- data.frame(
      x_failure = x_failure_success,
      y_failure = y_failure,
      x_success = x_failure_success,
      y_success = y_success,
      tag = rep(toString(x), 4)
    )

    df_for_proportion_bar <- rbind(df_for_proportion_bar, temp_df_proportion_bar)
  }
  df_for_proportion_bar <- df_for_proportion_bar %>% stats::na.omit()

  return(df_for_proportion_bar)
}

#' Generates df for baseline violin plot WITH flow = TRUE
#'
#' This function generates data frames to represent
#' data points for plotting violin plots
#'
#' @param boots Boot result obtained from boot.ci
#' @param x_idx_position Numeric value determining the position of the plots on the x-axis.
#' @param float_contrast Boolean value determining if a Gardner-Altman plot or Cumming estimation plot will be produced.
#' @param flow Boolean value determining whether the bars will be plotted in pairs.
#'
#' @return dataframe
#' @noRd
create_dfs_for_baseline_ec_violin <- function(boots, x_idx_position, float_contrast, flow = TRUE) {
  df_for_violin <- data.frame(
    x = NA,
    y = NA,
    tag = NA
  )
  x_axis_scalar <- ifelse(flow, 0, 0.5)
  curr_boot_idx <- 1

  for (i in x_idx_position) {
    ci_coords <- stats::density(boots[[curr_boot_idx]])

    x_coords_ci <- ci_coords$x
    y_coords_ci <- ci_coords$y

    # Standardise y
    y_coords_ci <- (y_coords_ci - min(y_coords_ci)) / (max(y_coords_ci) - min(y_coords_ci))
    y_coords_ci <- y_coords_ci / 6

    if (!(float_contrast)) {
      y_coords_ci <- y_coords_ci / 1.5
    }

    y_coords_ci <- y_coords_ci + i - x_axis_scalar

    temp_df_violin <- data.frame(
      x = x_coords_ci,
      y = y_coords_ci,
      tag = rep(toString(i), 512)
    )

    df_for_violin <- rbind(df_for_violin, temp_df_violin)

    curr_boot_idx <- curr_boot_idx + 1
  }
  df_for_violin <- df_for_violin %>%
    dplyr::arrange(tag, x, y)

  return(df_for_violin)
}
