#' Check if a Data Commons API key is available
#'
#' Checks whether the `DATACOMMONS_API_KEY` environment variable has been set.
#' Useful for examples, tests, or conditional execution of functions requiring
#' authentication.
#'
#' @return A logical value: `TRUE` if an API key is set, `FALSE` otherwise.
#'
#' @export
dc_has_api_key <- function() {
  Sys.getenv("DATACOMMONS_API_KEY") != ""
}

#' Set the Data Commons API key
#'
#' Stores the provided API key in the `DATACOMMONS_API_KEY` environment
#' variable, which is used for authentication in API calls.
#'
#' @param api_key A string containing a valid Data Commons API key.
#'
#' @return No return value, called for side effects.
#'
#' @export
dc_set_api_key <- function(api_key) {
  Sys.setenv("DATACOMMONS_API_KEY" = api_key)
}


#' Set the Data Commons base URL
#'
#' Stores the provided base URL in the `DATACOMMONS_BASE_URL` environment
#' variable. Useful for pointing to alternative or testing endpoints.
#'
#' @param base_url A string containing the base URL for the Data Commons API.
#'
#' @return No return value, called for side effects.
#'
#' @export
dc_set_base_url <- function(base_url) {
  Sys.setenv("DATACOMMONS_BASE_URL" = base_url)
}

#' @keywords internal
#' @noRd
next_req <- function(resp, req) {
  body <- resp_body_json(resp)

  next_token <- body$nextToken
  if (is.null(next_token)) {
    return(NULL)
  }

  req |>
    req_url_query(nextToken = next_token)
}

#' @keywords internal
#' @noRd
format_response <- function(data, return_type) {
  if (return_type == "json") {
    data |>
      resps_data(\(resp) resp_body_string(resp))
  } else if (return_type == "list") {
    data |>
      resps_data(\(resp) resp_body_json(resp))
  } else if (return_type == "data.frame") {
    raw <- data |>
      resps_data(\(resp) resp_body_json(resp))

    rows <- list()

    by_variable <- raw$byVariable
    facets_info <- raw$facets

    for (variable_name in names(by_variable)) {
      by_entity <- by_variable[[variable_name]]$byEntity
      for (entity_name in names(by_entity)) {
        ordered_facets <- by_entity[[entity_name]]$orderedFacets
        for (facet in ordered_facets) {
          facet_id <- facet$facetId
          observations <- facet$observations

          if (!is.null(facets_info[[facet_id]]$importName)) {
            facet_name <- facets_info[[facet_id]]$importName[[1]]
          } else {
            facet_name <- NA
          }

          for (obs in observations) {
            row <- list(
              entity_dcid = entity_name,
              variable_dcid = variable_name,
              date = obs$date,
              value = obs$value,
              facet_id = facet_id,
              facet_name = facet_name
            )
            rows[[length(rows) + 1]] <- row
          }
        }
      }
    }

    df <- as.data.frame(do.call(
      rbind,
      lapply(rows, as.data.frame, stringsAsFactors = FALSE)
    ))

    entity_names_raw <- dc_get_property_values(
      unique(df$entity_dcid),
      properties = "name",
      return_type = "json"
    )

    entity_names_parsed <- fromJSON(entity_names_raw)

    df2 <- do.call(
      rbind,
      lapply(names(entity_names_parsed$data), function(id) {
        entity <- entity_names_parsed$data[[id]]
        entity_name <- entity$arcs$name$nodes$value[1]
        data.frame(
          entity_dcid = id,
          entity_name = entity_name,
          stringsAsFactors = FALSE
        )
      })
    )

    df <- merge(df, df2, by = "entity_dcid", all.x = TRUE)

    variable_names_raw <- dc_get_property_values(
      unique(df$variable_dcid),
      properties = "name",
      return_type = "json"
    )

    variable_names_parsed <- fromJSON(variable_names_raw)

    df3 <- do.call(
      rbind,
      lapply(names(variable_names_parsed$data), function(id) {
        entity <- variable_names_parsed$data[[id]]
        variable_name <- entity$arcs$name$nodes$value[1]
        data.frame(
          variable_dcid = id,
          variable_name = variable_name,
          stringsAsFactors = FALSE
        )
      })
    )

    df <- merge(df, df3, by = "variable_dcid", all.x = TRUE)

    df <- df[, c(
      "entity_dcid",
      "entity_name",
      "variable_dcid",
      "variable_name",
      setdiff(
        names(df),
        c("entity_dcid", "entity_name", "variable_dcid", "variable_name")
      )
    )]

    df
  }
}

#' @keywords internal
#' @noRd
construct_request <- function(
  request_type = "get",
  base_url,
  path,
  key,
  nodes = NULL,
  property = NULL,
  date = NULL,
  select = NULL,
  variable_dcids = NULL,
  entity_dcids = NULL,
  entity_expression = NULL,
  filter_domains = NULL,
  filter_facet_ids = NULL,
  query = NULL
) {
  query_params <- list(
    key = key,
    nodes = nodes,
    property = property,
    date = date,
    select = select,
    "variable.dcids" = variable_dcids,
    "entity.dcids" = entity_dcids,
    "entity.expression" = entity_expression,
    "filter.domains" = filter_domains,
    "filter.facet_ids" = filter_facet_ids,
    query = query
  )

  if (request_type == "get") {
    query_params <- Filter(Negate(is.null), query_params)

    request(base_url) |>
      req_url_path_append(path) |>
      req_url_query(!!!query_params, .multi = "explode") |>
      req_user_agent(
        paste(
          "datacommons R package",
          "(https://github.com/tidy-intelligence/r-datacommons)"
        )
      )
  } else if (request_type == "post") {
    request(base_url) |>
      req_url_path_append(path) |>
      req_method("POST") |>
      req_headers(
        "X-API-Key" = key
      ) |>
      req_body_json(list(query = query)) |>
      req_user_agent(
        paste(
          "datacommons R package",
          "(https://github.com/tidy-intelligence/r-datacommons)"
        )
      )
  } else {
    cli::cli_abort(
      c("!" = "{.param request_type} must be 'get' or 'post'")
    )
  }
}

#' @keywords internal
#' @noRd
perform_request <- function(req) {
  req_perform_iterative(req, next_req = next_req)
}

#' @keywords internal
#' @noRd
handle_failures <- function(resps) {
  failed_resps <- resps_failures(resps)
  if (length(failed_resps) > 0) {
    for (resp in failed_resps) {
      url <- resp$request$url %||% "<unknown>" # nolint
      status <- resp$status_code %||% "unknown" # nolint
      msg <- tryCatch(
        {
          content <- resp_body_string(resp)
          jsonlite::fromJSON(content)$message %||% "<no message>"
        },
        error = function(e) "<could not parse error message>"
      )
      cli::cli_warn(c(
        "!" = "Request to {.url {url}} failed with status {.code {status}}.",
        ">" = msg
      ))
    }
  }
}

#' @keywords internal
#' @noRd
handle_successes <- function(resps) {
  resps |>
    resps_successes()
}

#' @keywords internal
#' @noRd
`%||%` <- function(a, b) if (!is.null(a)) a else b
