#' Get DRI from Case
#'
#' `get_dri_case` calculates the pre- and post-deliberation DRI from a specific
#' deliberation case
#'
#' @param case a character string specifying the name of the case in human_data
#' @param adjusted a logical indicating whether you want the original or adjusted
#' DRI formula
#' @param method a character string specifying the method for statistical testing,
#' must be one of "wilcox" (default) or "t.test"
#' @param alternative a character string specifying the alternative hypothesis,
#' must be one of "greater" (default), "two.sided" or "less".
#' You can specify just the initial letter
#' @param data a survey responses dataframe, must include pre-
#' (`stage_id == 1`) and post-deliberation (`stage_id == 2`) data
#'
#' @returns a tibble with with the following components: `case`, `pre`,
#' `post`, `delta`, `p_value`, and `significance`
#'
#' @export
#'
#' @importFrom stats wilcox.test t.test
#' @import tibble dplyr
#' @importFrom rlang .data
#'
#' @seealso [deliberr::human_data] for raw survey response data formatting
#' @family DRI survey methods
#'
#' @examples
#'
#' get_dri_case("Activate")
#'
#' # same as
#' get_dri_case("Activate", data = human_data)
#'
get_dri_case <- function(case, adjusted = TRUE, method = "wilcox",
                         alternative = "greater", data = NULL) {

  if (is.null(data)) data <- deliberr::human_data

  # get pre/post data
  data_pre <- data %>% filter(case == !!case, .data$stage_id == 1)
  data_post <- data %>% filter(case == !!case, .data$stage_id == 2)

  # get ic
  ic_pre <- get_dri_ic(data_pre)
  ic_post <- get_dri_ic(data_post)

  # calculate individual dri
  dri_ind_pre <- get_dri_ind(ic_pre, adjusted)
  dri_ind_post <- get_dri_ind(ic_post, adjusted)

  # calculate dri
  dri_pre <- get_dri(ic_pre, adjusted)
  dri_post <- get_dri(ic_post, adjusted)

  # calculate delta
  dri_delta <- dri_post - dri_pre

  # test pre/post difference in means
  if (method == "wilcox") {
    res <- wilcox.test(dri_ind_post$dri,
                       dri_ind_pre$dri,
                       paired = TRUE,
                       alternative = alternative)
    p_value <- res$p.value
  } else if (method == "t.test") {
    res <- t.test(dri_ind_post$dri,
                  dri_ind_pre$dri,
                  paired = TRUE,
                  alternative = alternative)
    p_value <- res$p.value
  } else {
    stop("Method", method, "not supported.")
  }

  # compile result
  dri_case <- tibble(case,
                     pre = dri_pre,
                     post = dri_post,
                     delta = dri_delta,
                     p_value) %>%
    mutate(
      significance = case_when(
        p_value < 0.001 ~ "***",
        p_value < 0.01  ~ "**",
        p_value < 0.05  ~ "*",
        TRUE            ~ "n.s."
      )
    )

  return(dri_case)

}
