setClass("hfilter",
  contains = "decomp",
  slots = c(
    regression_output = "lm"
  )
)

create_hfilter <- function(decomp, ts_name, frequency, regression_output) {
  
  methods::new("hfilter",
    decomp = decomp,
    ts_name = ts_name,
    frequency = frequency,
    regression_output = regression_output
  )
  
}

#'Fitted Components of the Hamilton Filter
#'
#'Obtain either fitted values or residuals from a fitted Hamilton filter.
#'
#'@param object an object of class \code{"hfilter"}, i.e. generated by a 
#'call to \code{\link{hamilton_filter}}.
#'@param ... currently without further use; implemented for compatibility.
#'
#'@details
#'Obtain the fitted and the residual values from the result of a fitted 
#'Hamilton filter. The name of the method indicates, what is returned.
#'
#'@return
#'A time series object of class \code{"ts"} is returned.
#'
#'@author
#'\itemize{
#'\item Dominik Schulz (Research Assistant) (Department of Economics, Paderborn
#'University), \cr
#'Author and Package Creator
#'}
#'
#'@export
#'
#'@examples
#'est <- hamilton_filter(log(EXPENDITURES))
#'residuals(est)
#'fitted(est)
#'
setMethod("fitted", "hfilter", function(object, ...) {
  
  na.omit(object@decomp[, "Fitted"])
  
})

#'@rdname fitted-hfilter-method
#'@export
setMethod("residuals", "hfilter", function(object, ...) {
  
  na.omit(object@decomp[, "Rest"])
  
})

# Plot method

#'Plot Method for the Results of a Hamilton Filter
#'
#'Visualize the results of an applied Hamilton filter.
#'
#'@param x an object of class \code{"hfilter"}, as returned by the function 
#'\code{\link{hamilton_filter}}.
#'@param which either a string or a number can be entered to select a 
#'plot type from the function call; options are (1) a facet plot of the 
#'components, (2) the observed time series, (3) the observations together 
#'with the fitted values, and (4) the residuals; for \code{which = NULL}, 
#'the plot type can be selected interactively in the console.
#'@param ... further arguments to pass to \code{\link[stats]{plot.ts}} or 
#'\code{\link[graphics]{matplot}} (depending on whether only one time series
#'or multiple time series are shown in the plot).
#'
#'@return
#'This function returns \code{NULL}.
#'
#'@author
#'\itemize{
#'\item Dominik Schulz (Research Assistant) (Department of Economics, Paderborn
#'University), \cr
#'Author and Package Creator
#'}
#'
#'@export
#'
#'@examples
#'est <- hamilton_filter(log(EXPENDITURES))
#'plot(est, which = 3, col = c(1, 6))
#'plot(est, which = 4)
#'
setMethod("plot", "hfilter", function(x, which = NULL, ...) {
  
  which <- check_which_hf(which)
  
  if (which == "0") {
    return(invisible(NULL))
  }
  
  plot_fun <- switch(
    which,
    "facets" = plot_facets,
    "observations" = plot_obs,
    "fitted" = plot_fit,
    "residuals" = plot_resid,
    "1" = plot_facets,
    "2" = plot_obs,
    "3" = plot_fit,
    "4" = plot_resid
  )
  
  plot_fun(x, ...)
  
})

#'\code{ggplot2} Plot Method for the Results of a Hamilton Filter
#'
#'Visualize the results of an applied Hamilton filter in the style of 
#'\code{ggplot2}.
#'
#'@param object an object of class \code{"hfilter"}, as returned by the function 
#'\code{\link{hamilton_filter}}.
#'@param which either a string or a number can be entered to select a 
#'plot type from the function call; options are (1) a facet plot of the 
#'components, (2) the observed time series, (3) the observations together 
#'with the fitted values, and (4) the residuals; for \code{which = NULL}, 
#'the plot type can be selected interactively in the console.
#'@param ... currently without use, implemented for compatibility.
#'
#'@return
#'This method returns a \code{ggplot2} plot object, i.e. an object of classes 
#'\code{"gg"} and \code{"ggplot"}.
#'
#'@author
#'\itemize{
#'\item Dominik Schulz (Research Assistant) (Department of Economics, Paderborn
#'University), \cr
#'Author and Package Creator
#'}
#'
#'@export
#'
#'@examples
#'est <- hamilton_filter(log(EXPENDITURES))
#'autoplot(est, which = 3, col = c(1, 6))
#'autoplot(est, which = 4)
#'
setMethod("autoplot", "hfilter", function(object, which = NULL, ...) {
  
  which <- check_which_hf(which)
  
  if (which == "0") {
    return(invisible(NULL))
  }
  
  plot_fun <- switch(
    which,
    "facets" = plot_facets_gg,
    "observations" = plot_obs_gg,
    "fitted" = plot_fit_gg,
    "residuals" = plot_resid_gg,
    "1" = plot_facets_gg,
    "2" = plot_obs_gg,
    "3" = plot_fit_gg,
    "4" = plot_resid_gg
  )
  
  plot_fun(object, ...)
  
})