#' @title Shannon diversity decomposition
#'
#' @description Computes Shannon diversity and its decomposition
#'
#' @param x Vector of dimension S (spicies) with the numbers of observed individuals in each spicy. NA values are allowed. 0 values are converted to NA.
#' @param groups Vector of dimension S of factors indicating the groups.
#'
#' @return
#' \itemize{
#'   \item \code{shannon}: Shannon's total Entropy.
#'   \item \code{within}: Within groups entropy.
#'   \item \code{between}: Between groups entropy.
#'   \item \code{groups}: A data frame with information about each group: relative frequency, internal entropy and number of spicies.
#' }
#'
#' @seealso \link{shannon}
#'
#' @references
#' "Quantifying Diversity through Entropy Decomposition: Insights into Hominin Occupation and Carcass Processing at Qesem cave"
#'
#' @examples
#' data(Qesem_s)
#' dec_shannon(Qesem_s$HU, Qesem_s$Group)
#'
#' @export
dec_shannon <- function(x, groups){
  stopifnot(
    length(x)==length(as.character(groups)),
    is.factor(groups)
  )
  x <- ifelse(x==0, NA, x)
  xx <- na.omit(data.frame(x, groups))
  stopifnot(
    is.numeric(xx$x),
    all(xx$x > 0),
    isTRUE(all.equal(xx$x, round(xx$x)))
  )
  p <- xx$x/sum(xx$x)
  l <- levels(xx$groups)
  m <- length(l)
  P <- numeric(m)
  Hg <- numeric(m)
  S <- numeric(m)
  for (i in 1:m){
    P[i] <- sum(p[xx$groups==l[i]])
    Hg[i] <- -sum(p[xx$groups==l[i]]/P[i]*log(p[xx$groups==l[i]]/P[i]))
    S[i] <- sum(xx$groups==l[i])
  }
  g <- data.frame(group=l, Prop=P, Entropy=Hg, S=S)
  rownames(g) <- NULL
  return(list(shannon = -sum(p*log(p)),
              within = sum(P*Hg),
              between = -sum(P*log(P)),
              groups = g))
}
