% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dynemu_exact.R
\name{dynemu_exact}
\alias{dynemu_exact}
\title{Predictive posterior computation via exact closed-form expression for one-steap-ahead Gaussian process (GP) emulators}
\usage{
dynemu_exact(mean, var, dynGP)
}
\arguments{
\item{mean}{numeric vector or matrix specifying the mean of uncertain inputs. The number of columns must match the input dimension of `dynGP`.}

\item{var}{numeric vector or matrix specifying the variance of uncertain inputs. The number of columns must match the input dimension of `dynGP`.}

\item{dynGP}{list of trained GP emulators fitted by \code{dynemu_GP}, each corresponding to a state variable.}
}
\value{
A list containing:
\itemize{
  \item \code{predy}: A single-row matrix of predictive mean values, with each column for corresponding state variable.
  \item \code{predsig2}: A single-row matrix of predictive variance values, with each column for corresponding state variable.
}
}
\description{
The function computes the predictive posterior distribution (mean and variance) for GP emulators
using closed-form expression, given uncertain inputs.
}
\details{
Given a trained set of GP emulators `dynGP` fitted using \code{dynemu_GP}, this function:
1. Computes the closed-form predictive posterior mean and variance for each state variable.
2. Incorporates input uncertainty by integrating over the input distribution via exact computation.

The computation follows a closed-form approach, leveraging the posterior distributions of Linked GP.
}
\examples{
library(lhs)
### Lotka-Volterra equations ###
LVmod0D <- function(Time, State, Pars) {
  with(as.list(c(State, Pars)), {
    IngestC <- rI * P * C
    GrowthP <- rG * P * (1 - P/K)
    MortC <- rM * C

    dP <- GrowthP - IngestC
    dC <- IngestC * AE - MortC
    return(list(c(dP, dC)))
  })
}

### Define parameters ###
pars <- c(rI = 1.5, rG = 1.5, rM = 2, AE = 1, K = 10)

### Define time sequence ###
times <- seq(0, 30, by = 0.01)

### Initial conditions ###
set.seed(1)
d <- 2
n <- 12*d
Design <- maximinLHS(n, d) * 5 # Generate LHS samples in range [0,5]
colnames(Design) <- c("P", "C")

### Fit GP emulators ###
fit.dyn <- dynemu_GP(LVmod0D, times, pars, Design)

### Define uncertain inputs ###
xmean <- c(P = 1, C = 2)
xvar <- c(P = 1e-5, C = 1e-5)

### Compute the next point ###
dynemu_exact(xmean, xvar, fit.dyn)

}
