% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/aux_getevent.R
\name{aux_getevent}
\alias{aux_getevent}
\title{Load seismic data of a user-defined event}
\usage{
aux_getevent(
  start,
  duration,
  station,
  component = "BHZ",
  format,
  dir,
  simplify = TRUE,
  eseis = TRUE,
  try = FALSE,
  verbose = FALSE
)
}
\arguments{
\item{start}{\code{POSIXct} value, start time of the data to import. If
lazy users only submit a text string instead of a POSIXct obejct, the 
function will try to convert that text string.}

\item{duration}{\code{Numeric} value, duration of the data to import,
in seconds.}

\item{station}{\code{Character} value, seismic station ID, which must
correspond to the ID in the file name of the data directory structure 
(cf. \code{aux_organisecubefiles}).}

\item{component}{\code{Character} value, seismic component, which must
correspond to the component name in the file name of the data directory  
structure (cf. \code{aux_organisecubefiles}). Default is 
\code{"BHZ"} (vertical component of a sac file).}

\item{format}{\code{Character} value, seismic data format. One out of 
\code{"sac"} and \code{"mseed"}. If omitted, the function will try to 
identify the right format automatically.}

\item{dir}{\code{Character} value, path to the seismic data directory.
See details for further info on data structure.}

\item{simplify}{\code{Logical} value, option to simplify output
when possible. This basically means that if only data from one station 
is loaded, the list object will have one level less. Default is 
\code{TRUE}.}

\item{eseis}{\code{Logical} value, option to read data to an \code{eseis}
object (recommended, see documentation of 
\code{aux_initiateeseis}), default is \code{TRUE}}

\item{try}{\code{Logical} value, option to run the function in try-mode, 
i.e., to let it return \code{NA} in case an error occurs during data
import. Default is \code{FALSE}.}

\item{verbose}{\code{Logical} value, option to show messages during 
function execution. Default is \code{FALSE}.}
}
\value{
A \code{list} object containing either a set of \code{eseis}
objects or a data set with the time vector (\code{$time}) 
and a list of seismic stations (\code{$station_ID}) with their seismic
signals as data frame (\code{$signal}). If \code{simplify = TRUE} (the 
default option) and only one seismic station is provided, the output  
object containseither just one eseis object or the vectors for 
\code{$time} and \code{$signal}.
}
\description{
The function loads seismic data from a data directory structure (see 
\code{aux_organisecubefiles()}) based on the event start time, duration,
component and station ID.
}
\details{
The data to be read needs to be adequately structured. The data directory
must contain SAC files organised by year (e.g.2022) then by Julian Day
in full three digits (e.g. 001) and then by a dedicated SAC file name, 
containing the station ID, two-digit year, three-digit Julian Day, start 
time hour, minute and second, three channel ID and the file extension SAC. 
All these items need to be separated by stops (e.g. 
sac/2022/001/LAU01.22.001.08.00.00. BHZ.SAC). This data structure will be 
most conveniently created by the functions \code{aux_organisecubefiles()} 
or \code{aux_organisecentaurfiles()}, or by manually written R code. 

The function assumes complete data sets, i.e., not a single hourly 
data set must be missing. The time 
vector is loaded only once, from the first station and its first 
component. Thus, it is assumed that all loaded seismic signals are
of the same sampling frequency and length.
}
\examples{

## set seismic data directory
dir_data <- paste0(system.file("extdata", package="eseis"), "/")

## load the z component data from a station
data <- aux_getevent(start = as.POSIXct(x = "2017-04-09 01:20:00", 
                                        tz = "UTC"), 
                      duration = 120,
                      station = "RUEG1",
                      component = "BHZ",
                      dir = dir_data)                       
## plot signal
plot_signal(data = data)

## load data from two stations
data <- aux_getevent(start = as.POSIXct(x = "2017-04-09 01:20:00", 
                                        tz = "UTC"), 
                     duration = 120,
                     station = c("RUEG1", "RUEG2"),
                     component = "BHZ",
                     dir = dir_data)

## plot both signals
par(mfcol = c(2, 1))
lapply(X = data, FUN = plot_signal)
                     
}
\author{
Michael Dietze
}
\keyword{eseis}
