% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sdmw.R
\name{sdmw}
\alias{sdmw}
\title{A Markov Chain Monte Carlo (MCMC) sampler for the panel spatial Durbin model (SDM) with unknown spatial weight matrix}
\usage{
sdmw(
  Y,
  tt,
  X = matrix(0, nrow(Y), 0),
  Z = matrix(1, nrow(Y), 1),
  niter = 100,
  nretain = 50,
  W_prior = W_priors(n = nrow(Y)/tt),
  rho_prior = rho_priors(),
  beta_prior = beta_priors(k = ncol(X) * 2 + ncol(Z)),
  sigma_prior = sigma_priors()
)
}
\arguments{
\item{Y}{numeric \eqn{N \times 1} matrix containing the dependent variables, where \eqn{N = nT} is the number of
spatial (\eqn{n}) times the number of time observations (\eqn{T}, with \code{tt=T}). Note that the observations
have organized such that \eqn{Y = [Y_1',...,Y_T']'}.}

\item{tt}{single number greater or equal to 1. Denotes the number of time observations. \eqn{tt = T}.}

\item{X}{numeric \eqn{N \times k_1} design matrix of independent variables. These will be
automatically spatially lagged. If no spatially lagged variable is included in the model
a matrix with \eqn{N} rows and zero columns should be supplied (the default value). Note:
either \eqn{X} or \eqn{Z} has to be a matrix with at least one column.}

\item{Z}{numeric \eqn{N \times k_3} design matrix of independent variables which are not
spatially lagged. The default value is a \eqn{N \times 1} vector of ones (i.e. an intercept for the model).
Note: either \eqn{X} or \eqn{Z} has to be a matrix with at least one column.}

\item{niter}{single number greater or equal to 1, indicating the total number of draws.
Will be automatically coerced to integer. The default value is 100.}

\item{nretain}{single number greater or equal to 0, indicating the number of draws
kept after the burn-in. Will be automatically coerced to integer. The default value is 50.}

\item{W_prior}{list containing prior settings for estimating the spatial weight matrix \eqn{W}.
Generated by the smart constructor \code{\link{W_priors}}.}

\item{rho_prior}{list of prior settings for estimating \eqn{\rho},
generated by the smart constructor \code{\link{rho_priors}}}

\item{beta_prior}{list containing priors for the slope coefficients \eqn{\beta},
generated by the smart constructor \code{\link{beta_priors}}. The ordering of the priors is:
(1) priors of \eqn{X}, (2) priors of spatially lagged \eqn{X}, (3) priors of \eqn{Z}.}

\item{sigma_prior}{list containing priors for the error variance \eqn{\sigma^2},
generated by the smart constructor \code{\link{sigma_priors}}}
}
\value{
List with posterior samples for the slope parameters, \eqn{\rho}, \eqn{\sigma^2}, \eqn{W},
and average direct, indirect, and total effects.
}
\description{
The sampler uses independent Normal-inverse-Gamma priors for the slope and variance parameters, as well as a four-parameter
beta prior for the spatial autoregressive parameter \eqn{\rho}. It is a wrapper around \code{\link{W_sampler}}.
}
\details{
The considered panel spatial Durbin model (SDM) with unknown (\eqn{n} by \eqn{n}) spatial weight
matrix \eqn{W} takes the form:

\deqn{
 Y_t = \rho W Y_t + X_t \beta_1 + W X_t \beta_2 + Z \beta_3 + \varepsilon_t,
 }

with \eqn{\varepsilon_t \sim N(0,I_n \sigma^2)} and \eqn{W = f(\Omega)}. The \eqn{n} by \eqn{n}
matrix \eqn{\Omega} is an unknown binary adjacency matrix with zeros on the main diagonal and
\eqn{f(\cdot)} is the (optional) row-standardization function. \eqn{\rho} is a scalar spatial autoregressive parameter.

\eqn{Y_t} (\eqn{n \times 1}) collects the \eqn{n} cross-sectional (spatial) observations for time
\eqn{t=1,...,T}. \eqn{X_t} (\eqn{n \times k_1}) and \eqn{Z_t} (\eqn{n \times k_2}) are
matrices of explanatory variables, where the former will also be spatially lagged. \eqn{\beta_1}
(\eqn{k_1 \times 1}), \eqn{\beta_2} (\eqn{k_1 \times 1}) and \eqn{\beta_3} (\eqn{k_2 \times 1})
are unknown slope parameter vectors.

After vertically staking the \eqn{T} cross-sections  \eqn{Y=[Y_1',...,Y_T']'} (\eqn{N \times 1}),
\eqn{X=[X_1',...,X_T']'} (\eqn{N \times k_1}) and \eqn{Z=[Z_1', ..., Z_T']'} (\eqn{N \times k_2}),
with \eqn{N=nT}. The final model can be expressed as:

\deqn{
 Y = \rho \tilde{W}Y + X \beta_1 + \tilde{W} X \beta_2 + Z \beta_3 + \varepsilon,
}

where \eqn{\tilde{W}=I_T \otimes W} and \eqn{\varepsilon \sim N(0,I_N \sigma^2)}. Note that the input
data matrices have to be ordered first by the cross-sectional spatial units and then stacked by time.

Estimation usually even works well in cases of \eqn{n >> T}. However, note that for applications with \eqn{n > 200} the
estimation process becomes computationally demanding and slow. Consider in this case reducing \code{niter} and
\code{nretain} and carefully check whether the posterior chains have converged.
}
\examples{
n = 20; tt = 10
dgp_dat = sim_dgp(n =n, tt = tt, rho = .75, beta1 = c(.5,1),beta2 = c(-1,.5),
            beta3 = c(1.5), sigma2 = .05,n_neighbor = 3,intercept = TRUE)
res = sdmw(Y = dgp_dat$Y,tt = tt,X = dgp_dat$X,Z = dgp_dat$Z,niter = 20,nretain = 10)
}
