.qrs.prop.fast <- function(y, x, prop, w = NULL, Q1, Q2, P = 10, family,
                          gridtheta, m) {
  #' qrs.prop.fast
  #'
  #' Algorithm 3: algorithm with preprocessing and quantile grid reduction for
  #' Quantile Regression with Selection (QRS); propensity score estimated
  #' previously.
  #'
  #' @param y = Dependent variable (N x 1)
  #' @param x = Regressors matrix (N x K)
  #' @param prop = Propensity score (N x 1)
  #' @param w = Sample weights (N x 1)
  #' @param Q1 = Number of quantiles in reduced grid
  #' @param Q2 = Number of quantiles in large grid
  #' @param P =  Number of evaluated values of parameter with large quantile
  #' grid
  #' @param family = Parametric copula family
  #' @param gridtheta = Grid of values for copula parameter (T x 1)
  #' @param m =  Parameter to select interval of observations in top and bottom
  #' groups
  #'
  #' @return beta = Estimated beta coefficients (K x Q2)
  #' @return theta = Estimated copula parameter
  #' @return objf_min = Value of objective function at the optimum
  #' @return b1 = Estimated beta coefficients for the grid of values of the
  #' copula parameter with the reduced quantile grid (K x Q1 x T)
  #' @return objf1 = Value of objective function for the grid of values of the
  #' copula parameter with the reduced quantile grid
  #' @return gridtheta2 = Grid of values for copula parameter selected during
  #' the first part of the algorithm (P x 1)
  #' @return b2 = Estimated beta coefficients for the grid of values of the
  #' copula parameter with large quantile grid (K x Q2 x P)
  #' @return objf2 = Value of objective function for the grid of values of the
  #' copula parameter with large quantile grid (P x 1)
  
  #library(copula)
  
  N <- NROW(x)
  K <- NCOL(x)
  
  # Sample weights
  if (is.null(w)) {
    w <- rep(1, N)
  }
  
  # Weighted regressors
  xw <- sweep(x, 1, w, `*`)
  
  # Conservative estimate of standard error
  small <- 1e-6
  zeta <- sqrt(rowSums((xw %*% solve(t(xw) %*% xw))^2))
  zeta <- pmax(zeta, small)
  
  # Quantile grids
  gridq1 <- seq(1 / (Q1 + 1), Q1 / (Q1 + 1), length.out = Q1)
  gridq2 <- seq(1 / (Q2 + 1), Q2 / (Q2 + 1), length.out = Q2)
  
  # Instrument
  phi <- prop * w
  
  # Prevent the conditional copula from being too close to 0 or 1
  eps <- 1e-5
  
  # Begin with central values of the theta parameter and quantile grids
  initq <- floor((Q1 + 1) / 2)
  initt <- floor((length(gridtheta) + 1) / 2)
  
  # Pre-generate matrices to store values of beta coefficients and criterion function
  objf1 <- numeric(length(gridtheta))
  b1 <- array(0, dim = c(K, Q1, length(gridtheta)))
  
  # Estimation with reduced quantile grid
  for (i1 in initt:length(gridtheta)) {
    t <- gridtheta[i1]
    
    # Create the copula based on the specified family
    copula <- switch(family,
                     "Gaussian" = copula::normalCopula(param = t, dim = 2),
                     "Clayton" = copula::claytonCopula(param = t, dim = 2),
                     "Frank" = copula::frankCopula(param = t, dim = 2),
                     "Gumbel" = copula::gumbelCopula(param = t, dim = 2),
                     stop("Unsupported copula family")
    )
    
    # Copula conditional on participation
    C <- copula::pCopula(cbind(rep(gridq1, times = N), rep(prop, each = Q1)), copula)
    C <- matrix(C, nrow = N, byrow = TRUE)
    G <- matrix(C / prop, nrow = N, ncol = Q1)
    G <- pmin(pmax(G, eps), 1 - eps)
    
    # Slope parameters given copula
    if (i1 == initt) {
      b1[, , i1] <- .rqr.fast(y, x, w, G, zeta, m, initq)
    } else {
      b1[, , i1] <- .rqrb0.fast(y, x, w, G, zeta, m, b1[, , i1 - 1])
    }
    
    # Objective function for copula parameter
    objf1[i1] <- ((t(phi) %*% rowSums((y<=x%*%b1[, , i1]) - G)) / N)^2
  }
  
  for (i1 in (initt - 1):1) {
    t <- gridtheta[i1]
    
    # Create the copula based on the specified family
    copula <- switch(family,
                     "Gaussian" = copula::normalCopula(param = t, dim = 2),
                     "Clayton" = copula::claytonCopula(param = t, dim = 2),
                     "Frank" = copula::frankCopula(param = t, dim = 2),
                     "Gumbel" = copula::gumbelCopula(param = t, dim = 2),
                     stop("Unsupported copula family")
    )
    
    # Copula conditional on participation
    C <- copula::pCopula(cbind(rep(gridq1, times = N), rep(prop, each = Q1)), copula)
    C <- matrix(C, nrow = N, byrow = TRUE)
    G <- matrix(C / prop, nrow = N, ncol = Q1)
    G <- pmin(pmax(G, eps), 1 - eps)
    
    # Slope parameters given copula
    b1[, , i1] <- .rqrb0.fast(y, x, w, G, zeta, m, b1[, , i1 + 1])
    
    # Objective function for copula parameter
    objf1[i1] <- ((t(phi) %*% rowSums((y<=x%*%b1[, , i1]) - G)) / N)^2
  }
  
  # Sort parameter values by objective function; select P candidate values
  index <- order(objf1)
  gridtheta2 <- gridtheta[index[1:P]]
  
  # Estimation with large quantile grid
  if (Q1 < Q2) {
    objf2 <- numeric(P)
    b2 <- array(0, dim = c(K, Q2, P))
    
    for (i1 in 1:P) {
      t <- gridtheta2[i1]
      
      # Create the copula based on the specified family
      copula <- switch(family,
                       "Gaussian" = copula::normalCopula(param = t, dim = 2),
                       "Clayton" = copula::claytonCopula(param = t, dim = 2),
                       "Frank" = copula::frankCopula(param = t, dim = 2),
                       "Gumbel" = copula::gumbelCopula(param = t, dim = 2),
                       stop("Unsupported copula family")
      )
      
      # Copula conditional on participation
      C <- copula::pCopula(cbind(rep(gridq2, times = N), rep(prop, each = Q2)), copula)
      C <- matrix(C, nrow = N, byrow = TRUE)
      G <- matrix(C / prop, nrow = N, ncol = Q2)
      G <- pmin(pmax(G, eps), 1 - eps)
      
      # Assign values already estimated
      for (i2 in 1:Q1) {
        closest <- which.min(abs(gridq1[i2] - gridq2))
        if (i2 == 1) {
          initq2 <- closest
        }
        b2[, closest, i1] <- b1[, i2, index[i1]]
      }
      
      # Estimate remaining values
      for (i2 in initq2:Q2) {
        if (all(b2[, i2, i1] == 0)) {
          b2[, i2, i1] <- .rqrtau.fast(y, x, w, G[, i2, drop = FALSE], zeta, m, b2[, i2 - 1, i1, drop = FALSE])
        }
      }
      
      for (i2 in (initq2 - 1):1) {
        if (all(b2[, i2, i1] == 0)) {
          b2[, i2, i1] <- .rqrtau.fast(y, x, w, G[, i2, drop = FALSE], zeta, m, b2[, i2 + 1, i1, drop = FALSE])
        }
      }
      
      # Objective function for copula parameter
      objf2[i1] <- ((t(phi) %*% rowSums((y<=x%*%b2[, , i1]) - G)) / N)^2
    }
  } else {
    objf2 <- objf1[index[1:P]]
    b2 <- b1[, , index[1:P]]
  }
  
  # Find minimum of objective function
  if (P > 1) {
    objf_min <- min(objf2)
    argminf <- which.min(objf2)
  } else {
    objf_min <- objf2
    argminf <- 1
  }
  
  # Optimum copula and beta parameters
  theta <- gridtheta2[argminf]
  beta <- b2[, , argminf]
  
  list(beta = beta, theta = theta, objf_min = objf_min,  b1 = b1, objf1 = objf1,
       gridtheta2 = gridtheta2, b2 = b2, objf2 = objf2)
}
