% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/extract_psd2_features.R
\name{aggregate_psd2_keyword_features}
\alias{aggregate_psd2_keyword_features}
\title{Aggregate PSD2 Keyword Features at the Application Level with Time Window Filtering}
\usage{
aggregate_psd2_keyword_features(
  data,
  id_col,
  description_col,
  amount_col = NULL,
  time_col = NULL,
  observation_window_start_col = NULL,
  scrape_date_col = NULL,
  ops = NULL,
  period = "all",
  separate_direction = if (!is.null(amount_col)) TRUE else FALSE,
  group_cols = NULL,
  min_freq = 1,
  use_matrix = TRUE,
  convert_to_df = TRUE,
  period_agg = sum,
  period_missing_inputs = 0
)
}
\arguments{
\item{data}{A data frame containing transaction records.}

\item{id_col}{A character string specifying the column name that identifies each application (e.g., \code{"application_id"}).}

\item{description_col}{A character string specifying the column name that contains the transaction descriptions.
Note that this column may contain \code{NA} values.}

\item{amount_col}{Optional. A character string specifying the column name that contains transaction amounts.
If provided, the function aggregates a value for each keyword (default \code{ops = list(amount = sum)}).
If omitted (\code{NULL}), the function aggregates counts of keyword occurrence (default \code{ops = list(count = sum)}).}

\item{time_col}{Optional. A character string specifying the column name that contains the transaction date
(or timestamp). When \code{period} is a numeric vector, this is required to filter the data by observation window.}

\item{observation_window_start_col}{Optional. A character string indicating the column name with the observation window start date.
If \code{period} is not \code{"all"} and is not numeric, this column is used in \code{aggregate_applications}.}

\item{scrape_date_col}{Optional. A character string indicating the column name with the scrape date.
If \code{period} is not \code{"all"} and is not numeric, this column is used in \code{aggregate_applications}.}

\item{ops}{A named list of functions used to compute summary features on the aggregated values.
If \code{amount_col} is provided and \code{ops} is \code{NULL}, the default is \code{list(amount = sum)}.
If \code{amount_col} is \code{NULL} and \code{ops} is \code{NULL}, the default is \code{list(count = sum)}.}

\item{period}{Either a character string or a numeric vector controlling time aggregation.
The default is \code{"all"}, meaning no time segmentation. If a numeric vector is provided (e.g., \code{c(30, 3)}),
it defines a cycle length in days (first element) and a number of consecutive cycles (second element). In that case,
only transactions with a transaction date between \code{scrape_date - (period[1] * period[2])} and \code{scrape_date}
are considered.}

\item{separate_direction}{Logical. If \code{TRUE} (the default when \code{amount_col} is provided),
a new column \code{"direction"} is added to automatically separate incoming and outgoing transactions based on the sign
of the amount.}

\item{group_cols}{Optional. A character vector of additional grouping columns to use during aggregation.
If \code{separate_direction} is \code{TRUE}, the \code{"direction"} grouping is added automatically.}

\item{min_freq}{Numeric. The minimum frequency a token must have to be included in the keyword extraction.
Default is 1.}

\item{use_matrix}{Logical. Passed to \code{extract_keyword_features}; if \code{TRUE} (the default) a sparse matrix is used.}

\item{convert_to_df}{Logical. Passed to \code{extract_keyword_features}; if \code{TRUE} (the default) the sparse matrix
is converted to a data.frame, facilitating binding with other data.}

\item{period_agg}{A function used to aggregate values within each period (see \code{aggregate_applications}).
Default is \code{sum}.}

\item{period_missing_inputs}{A numeric value to replace missing aggregated values. Default is \code{0}.}
}
\value{
A data frame with one row per application and aggregated keyword features.
}
\description{
This function extracts keyword features from a transaction descriptions column using the
\code{extract_keyword_features} function and then aggregates these features at the application level
using the \code{aggregate_applications} function. In addition, when the aggregation period is provided as a numeric
vector (e.g., \code{c(30, 3)}), the function filters out transactions that fall outside the observation window
defined as the period between \code{scrape_date - (period[1] * period[2])} and \code{scrape_date}. This prevents spending
time processing keywords from transactions that would later be aggregated as zeros.
}
\details{
The function supports two modes:

\itemize{
  \item If \code{amount_col} is not provided (i.e., \code{NULL}), the function aggregates keyword counts
        (i.e., the number of transactions in which a keyword appears) for each application.
  \item If \code{amount_col} is provided, then for each transaction the keyword indicator is multiplied
        by the transaction amount. In this mode, the default aggregation operation is to sum these values
        (using \code{ops = list(amount = sum)}), yielding the total amount associated with transactions that mention each keyword.
}

Additionally, if \code{amount_col} is provided and \code{separate_direction} is \code{TRUE} (the default),
a new column named \code{"direction"} is created to separate incoming (\code{"in"}) and outgoing (\code{"out"})
transactions based on the sign of the amount. Any additional grouping columns can be provided via \code{group_cols}.


The function performs the following steps:

\enumerate{
  \item Basic input checks are performed to ensure the required columns exist.
  \item The full list of application IDs is stored from the original data.
  \item If \code{amount_col} is provided and \code{separate_direction} is \code{TRUE}, a \code{"direction"} column is added
        to label transactions as incoming (\code{"in"}) or outgoing (\code{"out"}) based on the sign of the amount.
  \item When \code{period} is provided as a numeric vector, the function computes the observation window as
        \code{scrape_date - (period[1] * period[2])} to \code{scrape_date} and filters the dataset to include only transactions
        within this window. Transactions for applications with no records in the window will later be assigned zeros.
  \item Keyword features are extracted from the \code{description_col} using \code{extract_keyword_features}.
        If an \code{amount_col} is provided, the binary indicators are weighted by the transaction amount.
  \item The extracted keyword features are combined with the (possibly filtered) original data.
  \item For each keyword, the function calls \code{aggregate_applications} to aggregate the feature by application.
        The aggregation is performed over time periods defined by \code{period} (if applicable) and, if requested,
        further split by direction.
  \item Aggregated results for each keyword are merged by application identifier.
  \item Finally, the aggregated results are merged with the full list of application IDs so that applications
        with no transactions in the observation window appear with zeros.
}
}
\examples{
# Example: Aggregate keyword features for PSD2 transactions.

data(featForge_transactions)

# In this example, the 'description' field is parsed for keywords.
# Since the 'amount' column is provided, each keyword indicator is
# weighted by the transaction amount, and transactions are
# automatically split into incoming and outgoing via the 'direction' column.
# Additionally, the period is specified as c(30, 1), meaning only
# transactions occurring within the last 30 days.
# (scrape_date - 30 to scrape_date) are considered.
result <- aggregate_psd2_keyword_features(
  data = featForge_transactions,
  id_col = "application_id",
  description_col = "description",
  amount_col = "amount",
  time_col = "transaction_date",
  scrape_date_col = "scrape_date",
  observation_window_start_col = "obs_start",
  period = c(30, 1),
  ops = list(amount = sum),
  min_freq = 1,
  use_matrix = TRUE,
  convert_to_df = TRUE
)

# The resulting data frame 'result' contains one
# row per application with aggregated keyword features.
# For example, if keywords "casino" and "utilities" were detected,
# aggregated columns might be named:
# "casino_amount_direction_in",
# "casino_amount_direction_out",
# "utilities_amount_direction_in", etc.
result

}
