
#' Human player
#' @description Get moves as input from the command line, allowing users to play
#' against a bot.
#'
#' @param game A \code{6x7} matrix object representing the current game board.
#'
#' @details While possible, human vs. human games can be confusing because the
#'   game switches \code{X}s and \code{O}s between turns (so that every player
#'   sees their own pieces as \code{X}).
#'
#' @returns Prints the current game board and prompts the user to input a move,
#' which must be an integer between 1 and 7 and a valid move in the current
#' game.
#'
#' @examplesIf interactive()
#' play4inaRow(humanPlayer, randomBot)
#'
#' @export
humanPlayer <- function(game){
  # detect if first or second player
  nX <- sum(game == 'X')
  nO <- sum(game == 'O')
  if(nX != nO){ # human is player 2
    game <- invertPieces(game)
  }

  # print game
  cat('   ')
  cat(paste(1:7, collapse='  '))
  cat('\n')
  cat(' _______________________\n')
  for(i in 1:nrow(game)){
    cat(' | ')
    cat(paste(game[i,], collapse='  '))
    cat(' |\n')
  }
  cat(' -----------------------\n')
  # get move
  move <- readline(prompt = 'Move: ')
  move <- as.numeric(move)
  return(move)
}


#' Bot players for Four in a Row
#' @name bots
#' @description Computer players that select their next move based on various
#'   amounts of internal logic.
#'
#' @param game A \code{6x7} matrix object representing the current game board.
#'
#' @returns Returns an integer between 1 and 7. Each bot only selects from the
#'   set of valid moves, so they won't select a column that is already full.
NULL


#' @describeIn bots Chooses moves randomly.
#'
#' @examples
#' play4inaRow(randomBot, easyBot)
#'
#' @export
randomBot <- function(game){
  # identify legal moves
  poss <- which(game[1,] == '.')
  # if only one legal move
  if(length(poss) == 1){
    return(poss)
  }
  # otherwise, random
  return(sample(poss, 1))
}


#' Easy bot player
#' @describeIn bots Tries to make 4 in a row, but does not consider its
#'   opponents moves.
#'
#' @export
easyBot <- function(game){
  # identify legal moves
  poss <- which(game[1,] == '.')
  # if only one legal move
  if(length(poss) == 1){
    return(poss)
  }
  # otherwise, pick spot that helps the most
  # for each possible choice, look at where the piece would end up
  # assign points based on all possible sets of 4 that include that spot
  # for each set:
  # if no O's
  #  - 2^[# of X's]
  score <- function(set, game){
    syms <- game[set]
    if(sum(syms=='O') == 0){
      return(2^(sum(syms=='X')+1))
    }
    return(0)
  }

  # shuffle possibilities so that it doesn't always favor one side
  poss <- sample(poss)

  scores <- sapply(poss, function(p){
    col <- p
    row <- max(which(game[,p] == '.'))

    # check all possible winning sets containing the spot
    sets <- getSetsof4()
    ind <- 6*(col-1) + row
    sx <- sets[which(sets==ind, arr.ind = TRUE)[,1], ]
    scr <- sum(apply(sx, 1, score, game))
    return(scr)
  })
  return(poss[which.max(scores)])
}


#' Medium bot player
#' @describeIn bots Selects a move based on simple internal logic. It tries to
#'   make 4 in a row and tries to block the opponent from winning, but does not
#'   consider possible downstream moves.
#'
#' @examples
#' play4inaRow(mediumBot, hardBot)
#'
#' @export
mediumBot <- function(game){
  # identify legal moves
  poss <- which(game[1,] == '.')
  # if only one legal move
  if(length(poss) == 1){
    return(poss)
  }

  score <- function(set, game){
    # for each set:
    # if no O's
    #  - 2^[# of X's]
    # if 1 O
    #  - 2 points for first X
    # if 2 O's
    #  - 10 points for blocking a potential unblocked 3-in-a-row
    # if 3 O's
    #  - 99 points (more important than everything except winning)
    # if 3 X's
    #  - 100 points
    syms <- game[set]
    if(sum(syms=='O') == 0){
      if(sum(syms=='X')==3){
        return(100)
      }
      return(2^(sum(syms=='X')+1))
    }
    if(sum(syms=='O') == 1 & sum(syms=='X') == 0){
      return(2)
    }
    if(sum(syms=='O') == 2 & sum(syms=='X') == 0){
      return(10)
    }
    if(sum(syms=='O') == 3){
      return(99)
    }
    return(0)
  }

  # shuffle possibilities so that it doesn't always favor one side
  poss <- sample(poss)

  # pick spot that helps the most
  # for each possible choice, look at where the piece would end up
  # assign points based on all possible sets of 4 that include that spot
  sets <- getSetsof4()
  scores <- sapply(poss, function(p){
    col <- p
    row <- max(which(game[,p] == '.'))

    # check all possible winning sets containing the spot
    ind <- 6*(col-1) + row
    sx <- sets[which(sets==ind, arr.ind = TRUE)[,1], ]
    scr <- sum(apply(sx, 1, score, game))
    return(scr)
  })
  return(poss[which.max(scores)])
}


#' Hard bot player
#' @describeIn bots Selects a move by looking three moves ahead (with downstream
#'   moves selected by internal logic similar to \code{mediumBot}).
#'
#' @export
hardBot <- function(game){
  # identify legal moves
  poss <- which(game[1,] == '.')
  # if only one legal move
  if(length(poss) == 1){
    return(poss)
  }


  modifiedScore <- function(set, game){
    # otherwise, pick spot that helps the most
    # for each possible choice, look at where the piece would end up
    # assign points based on all possible sets of 4 that include that spot
    # for each set:
    # if no O's
    #  - 2^[# of X's]
    # if 1 O
    #  - 2 points for first X
    # if 2 O's
    #  - 10 points for blocking a potential unblocked 3-in-a-row
    # if 3 O's
    #  - 21 points (don't want to incentivize letting the opponent get 3 in a row just so you can block it)
    # if 3 X's
    #  - 100 points
    syms <- game[set]
    if(sum(syms=='O') == 0){
      if(sum(syms=='X')==3){
        return(100)
      }
      return(2^(sum(syms=='X')+1))
    }
    if(sum(syms=='O') == 1 & sum(syms=='X') == 0){
      return(2)
    }
    if(sum(syms=='O') == 2 & sum(syms=='X') == 0){
      return(10)
    }
    if(sum(syms=='O') == 3){
      return(21)
    }
    return(0)
  }


  # shuffle possibilities so that it doesn't always favor one side
  poss <- sample(poss)

  sets <- getSetsof4()
  scores <- sapply(poss, function(p){
    col <- p
    row <- max(which(game[,p] == '.'))

    # check all possible winning sets containing the spot
    ind <- 6*(col-1) + row
    sx <- sets[which(sets==ind, arr.ind = TRUE)[,1], ]
    scr <- sum(apply(sx, 1, modifiedScore, game))

    # create possible game
    game.poss <- game
    game.poss[row,col] <- 'X'

    # check for winner
    # only check possible sets containing the most recently played piece
    ind <- 6*(col-1)+row
    sx <- sets[which(sets==ind, arr.ind = TRUE)[,1], ]
    winner <- apply(sx, 1, function(set){
      all(game.poss[set] == 'X')
    })
    if(any(winner) | !any(game.poss=='.')){
      return(scr)
    }

    # simulate 3 more turns, add up scores (opponent's scores are negative)
    for(i in 1:3){
      sign <- (-1)^i

      # switch X's and O's
      game.poss <- invertPieces(game.poss)

      # do turn
      move <- mediumBot(game.poss)
      col <- move
      row <- max(which(game.poss[,col] == '.'))

      # check all possible winning sets containing the spot
      ind <- 6*(col-1) + row
      sx <- sets[which(sets==ind, arr.ind = TRUE)[,1], ]
      scr <- scr + sign * sum(apply(sx, 1, modifiedScore, game.poss))

      # add X
      row <- max(which(game.poss[,move] == '.'))
      game.poss[row, move] <- 'X'

      # check for winner
      # only check possible sets containing the most recently played piece
      ind <- 6*(col-1)+row
      sx <- sets[which(sets==ind, arr.ind = TRUE)[,1], ]
      winner <- apply(sx, 1, function(set){
        all(game.poss[set] == 'X')
      })
      if(any(winner) | !any(game.poss=='.')){
        return(scr)
      }
    }

    return(scr)
  })

  return(poss[which.max(scores)])
}

