% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{auc_parallel}
\alias{auc_parallel}
\title{Parallel AUC and partial AUC calculation with optimized memory usage}
\usage{
auc_parallel(
  test_prediction,
  prediction,
  threshold = 5,
  sample_percentage = 50,
  iterations = 500L,
  compute_full_auc = TRUE,
  n_bins = 500L
)
}
\arguments{
\item{test_prediction}{Numeric vector of test prediction values}

\item{prediction}{Numeric vector of model predictions (background suitability data)}

\item{threshold}{Percentage threshold for partial AUC calculation (default = 5.0)}

\item{sample_percentage}{Percentage of test data to sample in each iteration (default = 50.0)}

\item{iterations}{Number of bootstrap iterations (default = 500)}

\item{compute_full_auc}{Boolean indicating whether to compute complete AUC (default = TRUE)}

\item{n_bins}{Number of bins for discretization (default = 500)}
}
\value{
A numeric matrix with `iterations` rows and 4 columns containing:
\itemize{
  \item auc_complete: Complete AUC (NA when compute_full_auc = FALSE)
  \item auc_pmodel: Partial AUC for the model (sensitivity > 1 - threshold/100)
  \item auc_prand: Partial AUC for random model (reference)
  \item ratio: Ratio of model AUC to random AUC (model/reference)
}
}
\description{
Computes bootstrap estimates of partial and complete AUC using parallel processing and optimized binning.
}
\details{
This function implements a highly optimized AUC calculation pipeline:
1. Cleans input data (removes non-finite values)
2. Combines background and test predictions
3. Performs range-based binning (discretization)
4. Computes cumulative distribution of background predictions
5. Runs bootstrap iterations in parallel:
   - Samples test predictions
   - Computes sensitivity-specificity curves
   - Calculates partial and complete AUC

Key optimizations:
- OpenMP parallelization for binning and bootstrap
- Vectorized operations using Armadillo
}
\section{Partial AUC}{

The partial AUC focuses on the high-sensitivity region defined by:
Sensitivity > 1 - (threshold/100)
}

\examples{
# Basic usage with random data
set.seed(123)
bg_pred <- runif(1000)   # bg predictions
test_pred <- runif(500)     # Test predictions

# Compute only partial AUC metrics (500 iterations)
results <- auc_parallel(test_pred, bg_pred,
                            threshold = 5.0,
                            iterations = 100)  # Reduced for example

# View first 5 iterations
head(results, 5)

# Summarize results (assume complete AUC was not computed)
summary <- summarize_auc_results(results, has_complete_auc = FALSE)

# Interpretation:
# - auc_pmodel: Model's partial AUC (higher is better)
# - auc_prand: Random model's partial AUC
# - ratio: Model AUC / Random AUC (>1 indicates better than random)

# Compute both partial and complete AUC
full_results <- auc_parallel(test_pred, bg_pred,
                                 compute_full_auc = TRUE,
                                 iterations = 100)


}
\seealso{
\code{\link{summarize_auc_results}} for results processing,
         \code{\link{trap_roc}} for integration method
}
