% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gclust.R
\name{gclust}
\alias{gclust}
\alias{gclust.default}
\alias{gclust.dist}
\alias{gclust.mst}
\alias{genie}
\alias{genie.default}
\alias{genie.dist}
\alias{genie.mst}
\title{Hierarchical Clustering Algorithm Genie}
\usage{
gclust(d, ...)

\method{gclust}{default}(
  d,
  gini_threshold = 0.3,
  distance = c("euclidean", "l2", "manhattan", "cityblock", "l1", "cosine"),
  verbose = FALSE,
  ...
)

\method{gclust}{dist}(d, gini_threshold = 0.3, verbose = FALSE, ...)

\method{gclust}{mst}(d, gini_threshold = 0.3, verbose = FALSE, ...)

genie(d, ...)

\method{genie}{default}(
  d,
  k,
  gini_threshold = 0.3,
  distance = c("euclidean", "l2", "manhattan", "cityblock", "l1", "cosine"),
  M = 1L,
  postprocess = c("boundary", "none", "all"),
  detect_noise = M > 1L,
  verbose = FALSE,
  ...
)

\method{genie}{dist}(
  d,
  k,
  gini_threshold = 0.3,
  M = 1L,
  postprocess = c("boundary", "none", "all"),
  detect_noise = M > 1L,
  verbose = FALSE,
  ...
)

\method{genie}{mst}(
  d,
  k,
  gini_threshold = 0.3,
  postprocess = c("boundary", "none", "all"),
  detect_noise = FALSE,
  verbose = FALSE,
  ...
)
}
\arguments{
\item{d}{a numeric matrix (or an object coercible to one,
e.g., a data frame with numeric-like columns) or an
object of class \code{dist} (see \code{\link[stats]{dist}}),
or an object of class \code{mst} (\code{\link{mst}})}

\item{...}{further arguments passed to \code{\link{mst}()}}

\item{gini_threshold}{threshold for the Genie correction, i.e.,
the Gini index of the cluster size distribution;
threshold of 1.0 leads to the single linkage algorithm;
low thresholds highly penalise the formation of small clusters}

\item{distance}{metric used to compute the linkage, one of:
\code{"euclidean"} (synonym: \code{"l2"}),
\code{"manhattan"} (a.k.a. \code{"l1"} and \code{"cityblock"}),
\code{"cosine"}}

\item{verbose}{logical; whether to print diagnostic messages
and progress information}

\item{k}{the desired number of clusters to detect, \eqn{k=1} with
\eqn{M>1} acts as a noise point detector}

\item{M}{smoothing factor; \eqn{M \leq 2} gives the selected \code{distance};
otherwise, the mutual reachability distance is used}

\item{postprocess}{one of \code{"boundary"} (default), \code{"none"}
or \code{"all"};  in effect only if \eqn{M > 1}.
By default, only "boundary" points are merged
with their nearest "core" points (A point is a boundary point if it is
a noise point and it is amongst its adjacent vertex's
(\eqn{M-1})-th nearest neighbours). To force a classical
k-partition of a data set (with no notion of noise),
choose \code{"all"}}

\item{detect_noise}{whether the minimum spanning tree's leaves
should be marked as noise points, defaults to \code{TRUE} if \eqn{M>1}
for compatibility with HDBSCAN*}
}
\value{
\code{gclust()} computes the entire clustering hierarchy; it
returns a list of class \code{hclust}; see \code{\link[stats]{hclust}}.
Use \code{\link[stats]{cutree}} to obtain an arbitrary \code{k}-partition.

\code{genie()} returns a \code{k}-partition - a vector whose i-th element
denotes the i-th input point's cluster label between 1 and \code{k}
If \code{detect_noise} is \code{TRUE}, missing values (\code{NA}) denote
noise points.
}
\description{
A reimplementation of \emph{Genie} - a robust and outlier resistant
clustering algorithm by Gagolewski, Bartoszuk, and Cena (2016).
The Genie algorithm is based on the minimum spanning tree (MST) of the
pairwise distance graph of a given point set.
Just like the single linkage, it consumes the edges
of the MST in an increasing order of weights. However, it prevents
the formation of clusters of highly imbalanced sizes; once the Gini index
(see \code{\link{gini_index}()}) of the cluster size distribution
raises above \code{gini_threshold}, the merging of a point group
of the smallest size is enforced.

The clustering can also be computed with respect to the
mutual reachability distances (based, e.g., on the Euclidean metric),
which is used in the definition of the HDBSCAN* algorithm
(see Campello et al., 2013). If \eqn{M>1}, then the mutual reachability
distance \eqn{m(i,j)} with a smoothing factor \eqn{M} is used instead of the
chosen "raw" distance \eqn{d(i,j)}.  It holds
\eqn{m(i,j)=\max(d(i,j), c(i), c(j))}, where the core distance \eqn{c(i)} is
the distance to the \eqn{i}-th point's (\eqn{M-1})-th
nearest neighbour.  This makes "noise" and "boundary" points being
more "pulled away" from each other.

The Genie correction together with the smoothing factor \eqn{M>1}
(note that \eqn{M=2} corresponds to the original distance) gives
a version of the HDBSCAN* algorithm that is able to detect
a predefined number of clusters. Hence it does not dependent on the DBSCAN's
\code{eps} parameter or the HDBSCAN's \code{min_cluster_size} one.
}
\details{
As in the case of all the distance-based methods,
the standardisation of the input features is definitely worth giving a try.
Oftentimes, more sophisticated feature engineering (e.g., dimensionality
reduction) will lead to more meaningful results.

If \code{d} is a numeric matrix or an object of class \code{dist},
\code{\link{mst}()} will be called to compute an MST, which generally
takes at most \eqn{O(n^2)} time. However, by default, a faster algorithm
based on K-d trees is selected automatically for low-dimensional Euclidean
spaces; see \code{\link[quitefastmst]{mst_euclid}} from
the \pkg{quitefastmst} package.

Once a minimum spanning tree is determined, the Genie algorithm runs in
\eqn{O(n \sqrt{n})} time.  If you want to test different
\code{gini_threshold}s or \code{k}s,  it is best to compute
the MST first explicitly.

According to the algorithm's original definition,
the resulting partition tree (dendrogram) might violate
the ultrametricity property (merges might occur at levels that
are not increasing w.r.t. a between-cluster distance).
\code{gclust()} automatically corrects departures from
ultrametricity by applying \code{height = rev(cummin(rev(height)))}.
}
\examples{
library("datasets")
data("iris")
X <- jitter(as.matrix(iris[2:3]))
h <- gclust(X)
y_pred <- cutree(h, 3)
y_test <- as.integer(iris[,5])
plot(X, col=y_pred, pch=y_test, asp=1, las=1)
adjusted_rand_score(y_test, y_pred)
normalized_clustering_accuracy(y_test, y_pred)

y_pred2 <- genie(X, 3, M=5)  # clustering wrt 5-mutual reachability distance
plot(X[,1], X[,2], col=y_pred2, pch=y_test, asp=1, las=1)
noise <- is.na(y_pred2)  # noise/boundary points
points(X[noise, ], col="gray", pch=10)
normalized_clustering_accuracy(y_test[!noise], y_pred2[!noise])

}
\references{
Gagolewski, M., Bartoszuk, M., Cena, A.,
Genie: A new, fast, and outlier-resistant hierarchical clustering algorithm,
\emph{Information Sciences} 363, 2016, 8-23,
\doi{10.1016/j.ins.2016.05.003}

Campello, R.J.G.B., Moulavi, D., Sander, J.,
Density-based clustering based on hierarchical density estimates,
\emph{Lecture Notes in Computer Science} 7819, 2013, 160-172,
\doi{10.1007/978-3-642-37456-2_14}

Gagolewski, M., Cena, A., Bartoszuk, M., Brzozowski, L.,
Clustering with minimum spanning trees: How good can it be?,
\emph{Journal of Classification} 42, 2025, 90-112,
\doi{10.1007/s00357-024-09483-1}
}
\seealso{
The official online manual of \pkg{genieclust} at \url{https://genieclust.gagolewski.com/}

Gagolewski, M., \pkg{genieclust}: Fast and robust hierarchical clustering, \emph{SoftwareX} 15:100722, 2021, \doi{10.1016/j.softx.2021.100722}

\code{\link{mst}()} for the minimum spanning tree routines

\code{\link{normalized_clustering_accuracy}()} (amongst others) for external
cluster validity measures
}
\author{
\href{https://www.gagolewski.com/}{Marek Gagolewski} and other contributors
}
