% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tar_terra_tiles.R
\name{tar_terra_tiles}
\alias{tar_terra_tiles}
\title{Split a raster into tiles that can be iterated over with dynamic branching}
\usage{
tar_terra_tiles(
  name,
  raster,
  tile_fun,
  filetype = geotargets_option_get("gdal.raster.driver"),
  gdal = geotargets_option_get("gdal.raster.creation.options"),
  ...,
  packages = targets::tar_option_get("packages"),
  library = targets::tar_option_get("library"),
  repository = targets::tar_option_get("repository"),
  error = targets::tar_option_get("error"),
  memory = targets::tar_option_get("memory"),
  garbage_collection = targets::tar_option_get("garbage_collection"),
  deployment = targets::tar_option_get("deployment"),
  priority = targets::tar_option_get("priority"),
  resources = targets::tar_option_get("resources"),
  storage = targets::tar_option_get("storage"),
  retrieval = targets::tar_option_get("retrieval"),
  cue = targets::tar_option_get("cue"),
  description = targets::tar_option_get("description")
)
}
\arguments{
\item{name}{Symbol, name of the target. A target name must be a valid name
for a symbol in R, and it must not start with a dot. See
\code{\link[targets:tar_target]{targets::tar_target()}} for more information.}

\item{raster}{a \code{SpatRaster} object to be split into tiles.}

\item{tile_fun}{a helper function that returns a list of numeric vectors such
as \code{\link[=tile_grid]{tile_grid()}}, \code{\link[=tile_n]{tile_n()}} or \link{tile_blocksize} specified in one of the
following ways:
\itemize{
\item A named function, e.g. \code{tile_blocksize} or \code{"tile_blocksize"}.
\item An anonymous function, e.g. \verb{\\(x) tile_grid(x, nrow = 2, ncol = 2)}.
}}

\item{filetype}{character. File format expressed as GDAL driver names passed
to \code{\link[terra:makeTiles]{terra::makeTiles()}}.}

\item{gdal}{character. GDAL driver specific datasource creation options
passed to \code{\link[terra:makeTiles]{terra::makeTiles()}}.}

\item{...}{additional arguments not yet used.}

\item{packages}{Character vector of packages to load right before
the target runs or the output data is reloaded for
downstream targets. Use \code{tar_option_set()} to set packages
globally for all subsequent targets you define.}

\item{library}{Character vector of library paths to try
when loading \code{packages}.}

\item{repository}{Character of length 1, remote repository for target
storage. Choices:
\itemize{
\item \code{"local"}: file system of the local machine.
\item \code{"aws"}: Amazon Web Services (AWS) S3 bucket. Can be configured
with a non-AWS S3 bucket using the \code{endpoint} argument of
\code{\link[targets:tar_resources_aws]{tar_resources_aws()}}, but versioning capabilities may be lost
in doing so.
See the cloud storage section of
\url{https://books.ropensci.org/targets/data.html}
for details for instructions.
\item \code{"gcp"}: Google Cloud Platform storage bucket.
See the cloud storage section of
\url{https://books.ropensci.org/targets/data.html}
for details for instructions.
\item A character string from \code{\link[targets:tar_repository_cas]{tar_repository_cas()}} for content-addressable
storage.
}

Note: if \code{repository} is not \code{"local"} and \code{format} is \code{"file"}
then the target should create a single output file.
That output file is uploaded to the cloud and tracked for changes
where it exists in the cloud. The local file is deleted after
the target runs.}

\item{error}{Character of length 1, what to do if the target
stops and throws an error. Options:
\itemize{
\item \code{"stop"}: the whole pipeline stops and throws an error.
\item \code{"continue"}: the whole pipeline keeps going.
\item \code{"null"}: The errored target continues and returns \code{NULL}.
The data hash is deliberately wrong so the target is not
up to date for the next run of the pipeline. In addition,
as of \code{targets} version 1.8.0.9011, a value of \code{NULL} is given
to upstream dependencies with \code{error = "null"} if loading fails.
\item \code{"abridge"}: any currently running targets keep running,
but no new targets launch after that.
\item \code{"trim"}: all currently running targets stay running. A queued
target is allowed to start if:
\enumerate{
\item It is not downstream of the error, and
\item It is not a sibling branch from the same \code{\link[targets:tar_target]{tar_target()}} call
(if the error happened in a dynamic branch).
}

The idea is to avoid starting any new work that the immediate error
impacts. \code{error = "trim"} is just like \code{error = "abridge"},
but it allows potentially healthy regions of the dependency graph
to begin running.
(Visit \url{https://books.ropensci.org/targets/debugging.html}
to learn how to debug targets using saved workspaces.)
}}

\item{memory}{Character of length 1, memory strategy. Possible values:
\itemize{
\item \code{"auto"} (default): equivalent to \code{memory = "transient"} in almost
all cases. But to avoid superfluous reads from disk,
\code{memory = "auto"} is equivalent to \code{memory = "persistent"} for
for non-dynamically-branched targets that other targets
dynamically branch over. For example: if your pipeline has
\code{tar_target(name = y, command = x, pattern = map(x))},
then \code{tar_target(name = x, command = f(), memory = "auto")}
will use persistent memory in order to avoid rereading all of \code{x}
for every branch of \code{y}.
\item \code{"transient"}: the target gets unloaded
after every new target completes.
Either way, the target gets automatically loaded into memory
whenever another target needs the value.
\item \code{"persistent"}: the target stays in memory
until the end of the pipeline (unless \code{storage} is \code{"worker"},
in which case \code{targets} unloads the value from memory
right after storing it in order to avoid sending
copious data over a network).
}

For cloud-based file targets
(e.g. \code{format = "file"} with \code{repository = "aws"}),
the \code{memory} option applies to the
temporary local copy of the file:
\code{"persistent"} means it remains until the end of the pipeline
and is then deleted,
and \code{"transient"} means it gets deleted as soon as possible.
The former conserves bandwidth,
and the latter conserves local storage.}

\item{garbage_collection}{Logical: \code{TRUE} to run \code{base::gc()}
just before the target runs, in whatever R process it is about to run
(which could be a parallel worker).
\code{FALSE} to omit garbage collection.
Numeric values get converted to \code{FALSE}.
The \code{garbage_collection} option in \code{\link[targets:tar_option_set]{tar_option_set()}}
is independent of the
argument of the same name in \code{\link[targets:tar_target]{tar_target()}}.}

\item{deployment}{Character of length 1. If \code{deployment} is
\code{"main"}, then the target will run on the central controlling R process.
Otherwise, if \code{deployment} is \code{"worker"} and you set up the pipeline
with distributed/parallel computing, then
the target runs on a parallel worker. For more on distributed/parallel
computing in \code{targets}, please visit
\url{https://books.ropensci.org/targets/crew.html}.}

\item{priority}{Deprecated on 2025-04-08 (\code{targets} version 1.10.1.9013).
\code{targets} has moved to a more efficient scheduling algorithm
(\url{https://github.com/ropensci/targets/issues/1458})
which cannot support priorities.
The \code{priority} argument of \code{\link[targets:tar_target]{tar_target()}} no longer has a reliable
effect on execution order.}

\item{resources}{Object returned by \code{tar_resources()}
with optional settings for high-performance computing
functionality, alternative data storage formats,
and other optional capabilities of \code{targets}.
See \code{tar_resources()} for details.}

\item{storage}{Character string to control when the output of the target
is saved to storage. Only relevant when using \code{targets}
with parallel workers (\url{https://books.ropensci.org/targets/crew.html}).
Must be one of the following values:
\itemize{
\item \code{"worker"} (default): the worker saves/uploads the value.
\item \code{"main"}: the target's return value is sent back to the
host machine and saved/uploaded locally.
\item \code{"none"}: \code{targets} makes no attempt to save the result
of the target to storage in the location where \code{targets}
expects it to be. Saving to storage is the responsibility
of the user. Use with caution.
}}

\item{retrieval}{Character string to control when the current target
loads its dependencies into memory before running.
(Here, a "dependency" is another target upstream that the current one
depends on.) Only relevant when using \code{targets}
with parallel workers (\url{https://books.ropensci.org/targets/crew.html}).
Must be one of the following values:
\itemize{
\item \code{"auto"} (default): equivalent to \code{retrieval = "worker"} in almost all
cases. But to avoid unnecessary reads from disk, \code{retrieval = "auto"}
is equivalent to \code{retrieval = "main"} for dynamic branches that
branch over non-dynamic targets. For example: if your pipeline has
\code{tar_target(x, command = f())}, then
\code{tar_target(y, command = x, pattern = map(x), retrieval = "auto")}
will use \code{"main"} retrieval in order to avoid rereading all of \code{x}
for every branch of \code{y}.
\item \code{"worker"}: the worker loads the target's dependencies.
\item \code{"main"}: the target's dependencies are loaded on the host machine
and sent to the worker before the target runs.
\item \code{"none"}: \code{targets} makes no attempt to load its
dependencies. With \code{retrieval = "none"}, loading dependencies
is the responsibility of the user. Use with caution.
}}

\item{cue}{An optional object from \code{tar_cue()} to customize the
rules that decide whether the target is up to date.}

\item{description}{Character of length 1, a custom free-form human-readable
text description of the target. Descriptions appear as target labels
in functions like \code{\link[targets:tar_manifest]{tar_manifest()}} and \code{\link[targets:tar_visnetwork]{tar_visnetwork()}},
and they let you select subsets of targets for the \code{names} argument of
functions like \code{\link[targets:tar_make]{tar_make()}}. For example,
\code{tar_manifest(names = tar_described_as(starts_with("survival model")))}
lists all the targets whose descriptions start with the character
string \code{"survival model"}.}
}
\value{
a list of two targets: an upstream target that creates a list of
extents and a downstream pattern that maps over these extents to create a
list of SpatRaster objects.
}
\description{
Creates two targets, a list of extents defining tiles and a downstream
pattern that maps over these extents to create a list of \code{SpatRaster} objects
that can be used with \href{https://books.ropensci.org/targets/dynamic.html}{dynamic branching}.
}
\details{
When a raster is too large or too high resolution to work on
in-memory, one possible solution is to iterate over tiles. Raster tiles can
then be operated on one at a time, or possibly in parallel if resources are
available, and then the results can be aggregated. A natural way to do this
in the context of a \code{targets} pipeline is to split the raster into multiple
raster targets with dynamic branching so that downstream targets can be
applied to each branch of the upstream target with the \code{pattern} argument
to \code{tar_terra_rast()} or \code{tar_target()}. \code{tar_terra_tiles()} facilitates
creation of such a dynamically branched target. This workflow isn't
appropriate for operations that aggregate spatially, only pixel-wise
operations (possibly aggregating across multiple layers).

This target factory is useful when a raster is too large or too high
resolution to work on in-memory. It can instead be split into tiles that can
be iterated over using dynamic branching.
}
\note{
The \code{iteration} argument is unavailable because it is hard-coded to
\code{"list"}, the only option that works currently.

When using the \code{\link[=tile_blocksize]{tile_blocksize()}} helper function, you may need to set
\code{memory = "transient"} on the upstream target provided to the \code{raster}
argument of \code{tar_terra_tiles()}.  More details are in the help file for
\code{\link[=tile_blocksize]{tile_blocksize()}}.
}
\examples{
# For CRAN. Ensures these examples run under certain conditions.
# To run this locally, run the code inside this if statement
if (Sys.getenv("TAR_LONG_EXAMPLES") == "true") {
  targets::tar_dir({
    targets::tar_script({
        library(targets)
        library(geotargets)
        library(terra)
        list(
            tar_target(
                my_file,
                system.file("ex/elev.tif", package="terra"),
                format = "file"
            ),
            tar_terra_rast(
                my_map,
                terra::rast(my_file)
            ),
            tar_terra_tiles(
                name = rast_split,
                raster = my_map,
                tile_fun = \(x) tile_grid(x, ncol = 2, nrow = 2)
            )
        )
    })
    targets::tar_manifest()
  })
}
}
\seealso{
\code{\link[=tile_n]{tile_n()}}, \code{\link[=tile_grid]{tile_grid()}}, \code{\link[=tile_blocksize]{tile_blocksize()}}, \code{\link[=tar_terra_rast]{tar_terra_rast()}}, \code{\link[=tar_terra_vrt]{tar_terra_vrt()}}
}
\author{
Eric Scott
}
