% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/geom_vector_smooth.R
\docType{data}
\name{geom_vector_smooth}
\alias{geom_vector_smooth}
\alias{stat_vector_smooth}
\alias{geom_vector_smooth2}
\alias{stat_vector_smooth2}
\alias{StatVectorSmooth}
\alias{GeomVectorSmooth}
\title{Create a Smooth Vector Plot Layer}
\format{
An object of class \code{GeomVectorSmooth} (inherits from \code{GeomSegment}, \code{Geom}, \code{ggproto}, \code{gg}) of length 6.
}
\usage{
geom_vector_smooth(
  mapping = NULL,
  data = NULL,
  stat = "vector_smooth",
  position = "identity",
  ...,
  na.rm = FALSE,
  show.legend = NA,
  inherit.aes = TRUE,
  n = c(11, 11),
  method = "lm",
  se = TRUE,
  se.circle = TRUE,
  pi_type = "ellipse",
  conf_level = c(0.95, NA),
  formula = cbind(fx, fy) ~ x * y,
  eval_points = NULL,
  arrow = grid::arrow(angle = 20, length = unit(0.015, "npc"), type = "closed")
)

GeomVectorSmooth
}
\arguments{
\item{mapping}{A set of aesthetic mappings created by \code{ggplot2::aes()}.
\strong{Required:} Must include \strong{\code{x}} and \strong{\code{y}}; vector displacements are defined by
\strong{\code{fx}} and \strong{\code{fy}}.}

\item{data}{A data frame containing the raw vector data.}

\item{stat}{The statistical transformation to use on the data (default:
\code{"vector_smooth"}).}

\item{position}{Position adjustment, either as a string or the result of a
position adjustment function.}

\item{...}{Other arguments passed to \code{ggplot2::layer()} and the underlying
geometry/stat.}

\item{n}{An integer vector specifying the number of grid points along each
axis for smoothing.}

\item{method}{Character. Specifies the smoothing method. Currently, the only
supported method is \code{"lm"}, which fits a multivariate linear model to
predict the vector displacements (\code{fx}, \code{fy}) from \code{x} and \code{y}.}

\item{se}{Logical. If \code{TRUE}, prediction (confidence) intervals are computed
and plotted.}

\item{se.circle}{Logical. If \code{TRUE}, circles are drawn around the origin of
each vector to represent the radius of the prediction interval.}

\item{pi_type}{Character. Determines the display style for prediction
intervals:
\itemize{
\item \code{"wedge"} (default): Angular wedges are drawn.
\item \code{"ellipse"}: Ellipses are used to represent the covariance of the predictions.
If \code{pi_type} is set to \code{"ellipse"} and \code{eval_points} is \code{NULL}, it will
revert to \code{"wedge"}.
}}

\item{conf_level}{Numeric. Specifies the confidence level for the prediction
intervals. Default is \code{0.95}.}

\item{formula}{A formula specifying the multivariate linear model used for
smoothing. The default is \code{cbind(fx, fy) ~ x * y}.}

\item{eval_points}{A data frame of evaluation points. If provided, these
specify the grid where the smoothing model is evaluated. If \code{NULL}, a grid
is generated based on \code{n}.}

\item{arrow}{A \code{grid::arrow()} specification for arrowheads on the smoothed
vectors.}
}
\value{
A ggplot2 layer that can be added to a plot to create a smooth vector
field visualization.
}
\description{
\code{geom_vector_smooth()} creates a ggplot2 layer that visualizes a smooth
vector field. It takes raw vector data and applies smoothing (via a
multivariate linear model) to estimate the underlying vector field. This
functionality is analogous to \code{geom_smooth()} in ggplot2 but is tailored for
vector data rather than scalar responses.
}
\section{Aesthetics}{
 \code{geom_vector_smooth()} supports the following aesthetics
(required aesthetics are in \strong{bold}):
\itemize{
\item \strong{\code{x}}: The x-coordinate of the vector's starting point.
\item \strong{\code{y}}: The y-coordinate of the vector's starting point.
\item \strong{\code{fx}}: The horizontal component of the vector displacement.
\item \strong{\code{fy}}: The vertical component of the vector displacement.
\item \code{color}: The color of the vector lines.
\item \code{linewidth}: The thickness of the vector lines.
\item \code{linetype}: The type of the vector lines (e.g., solid, dashed).
\item \code{alpha}: The transparency level of the vectors.
\item \code{arrow}: An aesthetic that can be used to modify arrowhead properties.
}
}

\section{Details}{

\strong{Multivariate Linear Model:}
The \code{"lm"} method fits a multivariate linear model to predict vector
displacements (\code{fx} and \code{fy}) based on the coordinates \code{x} and \code{y}, including
interaction terms (\code{x * y}). This model smooths the raw vector data to
provide an estimate of the underlying vector field.

\strong{Prediction Intervals:}
When \code{se = TRUE}, prediction intervals are computed for the smoothed vectors.
Two types of intervals are supported:
\itemize{
\item \strong{Ellipse:} Ellipses represent the joint uncertainty (covariance) in the predicted \code{fx} and \code{fy}.
\item \strong{Wedge:} Wedges (angular sectors) indicate the range of possible vector directions and magnitudes.
The type of interval displayed is controlled by \code{pi_type}, and the confidence
level is set via \code{conf_level}.
}
}

\examples{
# Function to generate vectors
generate_vectors <- function(v) {
  x <- v[1]
  y <- v[2]
  c(
    sin(x) + sin(y) + rnorm(1, 5, 1),
    sin(x) - sin(y) - rnorm(1, 5, 1)
  )
}

# Set seed for reproducibility
set.seed(123)

# Create sample points and compute vectors
sample_points <- data.frame(
  x = runif(30, 0, 10),
  y = runif(30, 0, 10)
)

result <- t(apply(sample_points, 1, generate_vectors))

sample_points$xend <- result[, 1]
sample_points$yend <- result[, 2]
sample_points$fx <- sample_points$xend - sample_points$x
sample_points$fy <- sample_points$yend - sample_points$y
sample_points$distance <- sqrt(sample_points$fx^2 + sample_points$fy^2)
sample_points$angle <- atan2(sample_points$fy, sample_points$fx)

# Define evaluation points
eval_points <- data.frame(
  x = c(0, 7.5),
  y = c(10, 5)
)

# Example 1:
ggplot(sample_points, aes(x = x, y = y)) +
  geom_vector(aes(fx = fx, fy = fy, color = NULL), center = FALSE, alpha = 0.2) +
  geom_vector_smooth(aes(fx = fx, fy = fy), n = 5) +
  ggtitle("Smoothed Vector Field")

# Example 2: Ellipse with eval_points
ggplot(sample_points, aes(x = x, y = y)) +
  geom_vector(aes(fx = fx, fy = fy, color = NULL), center = FALSE, alpha = 0.2) +
  geom_vector_smooth(aes(fx = fx, fy = fy), eval_points = eval_points, conf_level = c(0.9)) +
  ggtitle("Smoothed Vector Field with Ellipse Intervals")

# Example 3: Wedge with eval_points
ggplot(sample_points, aes(x = x, y = y)) +
  geom_vector(aes(fx = fx, fy = fy, color = NULL), center = FALSE, alpha = 0.2) +
  geom_vector_smooth(aes(fx = fx, fy = fy), eval_points = eval_points, pi_type = "ellipse") +
  ggtitle("Smoothed Vector Field with Wedge Intervals")

}
\keyword{datasets}
\keyword{internal}
