/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Dmodule.h" 

#include "H5private.h"   
#include "H5CXprivate.h" 
#include "H5Dpkg.h"      
#include "H5Eprivate.h"  
#include "H5FLprivate.h" 
#include "H5MMprivate.h" 

#define H5D__SCATGATH_USE_CMPD_OPT_READ(DSET_INFO, IN_PLACE_TCONV)                                           \
    ((DSET_INFO)->type_info.cmpd_subset && H5T_SUBSET_FALSE != (DSET_INFO)->type_info.cmpd_subset->subset && \
     !(IN_PLACE_TCONV))

#define H5D__SCATGATH_USE_CMPD_OPT_WRITE(DSET_INFO, IN_PLACE_TCONV)                                          \
    ((DSET_INFO)->type_info.cmpd_subset && H5T_SUBSET_DST == (DSET_INFO)->type_info.cmpd_subset->subset &&   \
     (DSET_INFO)->type_info.dst_type_size == (DSET_INFO)->type_info.cmpd_subset->copy_size &&                \
     !(IN_PLACE_TCONV))

static herr_t H5D__scatter_file(const H5D_io_info_t *io_info, const H5D_dset_io_info_t *dset_info,
                                H5S_sel_iter_t *file_iter, size_t nelmts, const void *buf);
static size_t H5D__gather_file(const H5D_io_info_t *io_info, const H5D_dset_io_info_t *dset_info,
                               H5S_sel_iter_t *file_iter, size_t nelmts, void *buf);
static herr_t H5D__compound_opt_read(size_t nelmts, H5S_sel_iter_t *iter, const H5D_type_info_t *type_info,
                                     uint8_t *tconv_buf, void *user_buf );
static herr_t H5D__compound_opt_write(size_t nelmts, const H5D_type_info_t *type_info, void *tconv_buf);

H5FL_EXTERN(H5S_sel_iter_t);

H5FL_SEQ_EXTERN(size_t);

H5FL_SEQ_EXTERN(hsize_t);

static herr_t
H5D__scatter_file(const H5D_io_info_t *_io_info, const H5D_dset_io_info_t *_dset_info, H5S_sel_iter_t *iter,
                  size_t nelmts, const void *_buf)
{
    H5D_io_info_t      tmp_io_info;           
    H5D_dset_io_info_t tmp_dset_info;         
    hsize_t           *off = NULL;            
    hsize_t            mem_off;               
    size_t             mem_curr_seq;          
    size_t             dset_curr_seq;         
    size_t            *len = NULL;            
    size_t             orig_mem_len, mem_len; 
    size_t             nseq;                  
    size_t             nelem;                 
    size_t             dxpl_vec_size;         
    size_t             vec_size;              
    herr_t             ret_value = SUCCEED;   

    FUNC_ENTER_PACKAGE

    
    assert(_io_info);
    assert(_dset_info);
    assert(_dset_info->dset);
    assert(_dset_info->store);
    assert(iter);
    assert(nelmts > 0);
    assert(_buf);

    
    H5MM_memcpy(&tmp_io_info, _io_info, sizeof(*_io_info));
    H5MM_memcpy(&tmp_dset_info, _dset_info, sizeof(*_dset_info));
    tmp_io_info.op_type    = H5D_IO_OP_WRITE;
    tmp_dset_info.buf.cvp  = _buf;
    tmp_io_info.dsets_info = &tmp_dset_info;

    
    if (H5CX_get_vec_size(&dxpl_vec_size) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't retrieve I/O vector size");

    
    if (dxpl_vec_size > H5D_IO_VECTOR_SIZE)
        vec_size = dxpl_vec_size;
    else
        vec_size = H5D_IO_VECTOR_SIZE;
    if (NULL == (len = H5FL_SEQ_MALLOC(size_t, vec_size)))
        HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL, "can't allocate I/O length vector array");
    if (NULL == (off = H5FL_SEQ_MALLOC(hsize_t, vec_size)))
        HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL, "can't allocate I/O offset vector array");

    
    while (nelmts > 0) {
        
        if (H5S_SELECT_ITER_GET_SEQ_LIST(iter, vec_size, nelmts, &nseq, &nelem, off, len) < 0)
            HGOTO_ERROR(H5E_INTERNAL, H5E_UNSUPPORTED, FAIL, "sequence length generation failed");

        
        mem_curr_seq = dset_curr_seq = 0;
        orig_mem_len = mem_len = nelem * iter->elmt_size;
        mem_off                = 0;

        
        if ((*tmp_dset_info.layout_ops.writevv)(&tmp_io_info, &tmp_dset_info, nseq, &dset_curr_seq, len, off,
                                                (size_t)1, &mem_curr_seq, &mem_len, &mem_off) < 0)
            HGOTO_ERROR(H5E_DATASPACE, H5E_WRITEERROR, FAIL, "write error");

        
        tmp_dset_info.buf.cvp = (const uint8_t *)tmp_dset_info.buf.cvp + orig_mem_len;

        
        nelmts -= nelem;
    } 

done:
    
    if (len)
        len = H5FL_SEQ_FREE(size_t, len);
    if (off)
        off = H5FL_SEQ_FREE(hsize_t, off);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static size_t
H5D__gather_file(const H5D_io_info_t *_io_info, const H5D_dset_io_info_t *_dset_info, H5S_sel_iter_t *iter,
                 size_t nelmts, void *_buf )
{
    H5D_io_info_t      tmp_io_info;           
    H5D_dset_io_info_t tmp_dset_info;         
    hsize_t           *off = NULL;            
    hsize_t            mem_off;               
    size_t             mem_curr_seq;          
    size_t             dset_curr_seq;         
    size_t            *len = NULL;            
    size_t             orig_mem_len, mem_len; 
    size_t             nseq;                  
    size_t             nelem;                 
    size_t             dxpl_vec_size;         
    size_t             vec_size;              
    size_t             ret_value = nelmts;    

    FUNC_ENTER_PACKAGE

    
    assert(_io_info);
    assert(_dset_info);
    assert(_dset_info->dset);
    assert(_dset_info->store);
    assert(iter);
    assert(nelmts > 0);
    assert(_buf);

    
    H5MM_memcpy(&tmp_io_info, _io_info, sizeof(*_io_info));
    H5MM_memcpy(&tmp_dset_info, _dset_info, sizeof(*_dset_info));
    tmp_io_info.op_type    = H5D_IO_OP_READ;
    tmp_dset_info.buf.vp   = _buf;
    tmp_io_info.dsets_info = &tmp_dset_info;

    
    if (H5CX_get_vec_size(&dxpl_vec_size) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, 0, "can't retrieve I/O vector size");

    
    if (dxpl_vec_size > H5D_IO_VECTOR_SIZE)
        vec_size = dxpl_vec_size;
    else
        vec_size = H5D_IO_VECTOR_SIZE;
    if (NULL == (len = H5FL_SEQ_MALLOC(size_t, vec_size)))
        HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, 0, "can't allocate I/O length vector array");
    if (NULL == (off = H5FL_SEQ_MALLOC(hsize_t, vec_size)))
        HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, 0, "can't allocate I/O offset vector array");

    
    while (nelmts > 0) {
        
        if (H5S_SELECT_ITER_GET_SEQ_LIST(iter, vec_size, nelmts, &nseq, &nelem, off, len) < 0)
            HGOTO_ERROR(H5E_INTERNAL, H5E_UNSUPPORTED, 0, "sequence length generation failed");

        
        mem_curr_seq = dset_curr_seq = 0;
        orig_mem_len = mem_len = nelem * iter->elmt_size;
        mem_off                = 0;

        
        if ((*tmp_dset_info.layout_ops.readvv)(&tmp_io_info, &tmp_dset_info, nseq, &dset_curr_seq, len, off,
                                               (size_t)1, &mem_curr_seq, &mem_len, &mem_off) < 0)
            HGOTO_ERROR(H5E_DATASPACE, H5E_READERROR, 0, "read error");

        
        tmp_dset_info.buf.vp = (uint8_t *)tmp_dset_info.buf.vp + orig_mem_len;

        
        nelmts -= nelem;
    } 

done:
    
    if (len)
        len = H5FL_SEQ_FREE(size_t, len);
    if (off)
        off = H5FL_SEQ_FREE(hsize_t, off);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D__scatter_mem(const void *_tscat_buf, H5S_sel_iter_t *iter, size_t nelmts, void *_buf )
{
    uint8_t       *buf       = (uint8_t *)_buf; 
    const uint8_t *tscat_buf = (const uint8_t *)_tscat_buf;
    hsize_t       *off       = NULL;    
    size_t        *len       = NULL;    
    size_t         curr_len;            
    size_t         nseq;                
    size_t         curr_seq;            
    size_t         nelem;               
    size_t         dxpl_vec_size;       
    size_t         vec_size;            
    herr_t         ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(tscat_buf);
    assert(iter);
    assert(nelmts > 0);
    assert(buf);

    
    if (H5CX_get_vec_size(&dxpl_vec_size) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't retrieve I/O vector size");

    
    if (dxpl_vec_size > H5D_IO_VECTOR_SIZE)
        vec_size = dxpl_vec_size;
    else
        vec_size = H5D_IO_VECTOR_SIZE;
    if (NULL == (len = H5FL_SEQ_MALLOC(size_t, vec_size)))
        HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL, "can't allocate I/O length vector array");
    if (NULL == (off = H5FL_SEQ_MALLOC(hsize_t, vec_size)))
        HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL, "can't allocate I/O offset vector array");

    
    while (nelmts > 0) {
        
        if (H5S_SELECT_ITER_GET_SEQ_LIST(iter, vec_size, nelmts, &nseq, &nelem, off, len) < 0)
            HGOTO_ERROR(H5E_INTERNAL, H5E_UNSUPPORTED, 0, "sequence length generation failed");

        
        for (curr_seq = 0; curr_seq < nseq; curr_seq++) {
            
            curr_len = len[curr_seq];

            H5MM_memcpy(buf + off[curr_seq], tscat_buf, curr_len);

            
            tscat_buf += curr_len;
        } 

        
        nelmts -= nelem;
    } 

done:
    
    if (len)
        len = H5FL_SEQ_FREE(size_t, len);
    if (off)
        off = H5FL_SEQ_FREE(hsize_t, off);

    FUNC_LEAVE_NOAPI(ret_value)
} 

size_t
H5D__gather_mem(const void *_buf, H5S_sel_iter_t *iter, size_t nelmts, void *_tgath_buf )
{
    const uint8_t *buf       = (const uint8_t *)_buf; 
    uint8_t       *tgath_buf = (uint8_t *)_tgath_buf;
    hsize_t       *off       = NULL;   
    size_t        *len       = NULL;   
    size_t         curr_len;           
    size_t         nseq;               
    size_t         curr_seq;           
    size_t         nelem;              
    size_t         dxpl_vec_size;      
    size_t         vec_size;           
    size_t         ret_value = nelmts; 

    FUNC_ENTER_PACKAGE

    
    assert(buf);
    assert(iter);
    assert(nelmts > 0);
    assert(tgath_buf);

    
    if (H5CX_get_vec_size(&dxpl_vec_size) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, 0, "can't retrieve I/O vector size");

    
    if (dxpl_vec_size > H5D_IO_VECTOR_SIZE)
        vec_size = dxpl_vec_size;
    else
        vec_size = H5D_IO_VECTOR_SIZE;
    if (NULL == (len = H5FL_SEQ_MALLOC(size_t, vec_size)))
        HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, 0, "can't allocate I/O length vector array");
    if (NULL == (off = H5FL_SEQ_MALLOC(hsize_t, vec_size)))
        HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, 0, "can't allocate I/O offset vector array");

    
    while (nelmts > 0) {
        
        if (H5S_SELECT_ITER_GET_SEQ_LIST(iter, vec_size, nelmts, &nseq, &nelem, off, len) < 0)
            HGOTO_ERROR(H5E_INTERNAL, H5E_UNSUPPORTED, 0, "sequence length generation failed");

        
        for (curr_seq = 0; curr_seq < nseq; curr_seq++) {
            
            curr_len = len[curr_seq];

            H5MM_memcpy(tgath_buf, buf + off[curr_seq], curr_len);

            
            tgath_buf += curr_len;
        } 

        
        nelmts -= nelem;
    } 

done:
    
    if (len)
        len = H5FL_SEQ_FREE(size_t, len);
    if (off)
        off = H5FL_SEQ_FREE(hsize_t, off);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D__scatgath_read(const H5D_io_info_t *io_info, const H5D_dset_io_info_t *dset_info)
{
    void           *buf;                    
    void           *tmp_buf;                
    H5S_sel_iter_t *mem_iter       = NULL;  
    bool            mem_iter_init  = false; 
    H5S_sel_iter_t *bkg_iter       = NULL;  
    bool            bkg_iter_init  = false; 
    H5S_sel_iter_t *file_iter      = NULL;  
    bool            file_iter_init = false; 
    hsize_t         smine_start;            
    size_t          smine_nelmts;           
    bool            in_place_tconv;         
    herr_t          ret_value = SUCCEED;    

    FUNC_ENTER_PACKAGE

    
    assert(io_info);
    assert(dset_info);
    assert(dset_info->mem_space);
    assert(dset_info->file_space);
    assert(dset_info->buf.vp);

    
    buf = dset_info->buf.vp;

    
    if (dset_info->nelmts == 0)
        HGOTO_DONE(SUCCEED);

    
    in_place_tconv = dset_info->layout_io_info.contig_piece_info &&
                     dset_info->layout_io_info.contig_piece_info->in_place_tconv;

    
    if (in_place_tconv && H5D__SCATGATH_USE_CMPD_OPT_READ(dset_info, false) &&
        (io_info->use_select_io != H5D_SELECTION_IO_MODE_ON) &&
        (dset_info->type_info.need_bkg || (dset_info->nelmts <= dset_info->type_info.request_nelmts)))
        in_place_tconv = false;

    
    if (NULL == (mem_iter = H5FL_MALLOC(H5S_sel_iter_t)))
        HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL, "can't allocate memory iterator");
    if (NULL == (bkg_iter = H5FL_MALLOC(H5S_sel_iter_t)))
        HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL, "can't allocate background iterator");
    if (NULL == (file_iter = H5FL_MALLOC(H5S_sel_iter_t)))
        HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL, "can't allocate file iterator");

    
    if (H5S_select_iter_init(file_iter, dset_info->file_space, dset_info->type_info.src_type_size,
                             H5S_SEL_ITER_GET_SEQ_LIST_SORTED) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to initialize file selection information");
    file_iter_init = true; 
    if (H5S_select_iter_init(mem_iter, dset_info->mem_space, dset_info->type_info.dst_type_size, 0) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to initialize memory selection information");
    mem_iter_init = true; 
    if (H5S_select_iter_init(bkg_iter, dset_info->mem_space, dset_info->type_info.dst_type_size, 0) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to initialize background selection information");
    bkg_iter_init = true; 

    
    for (smine_start = 0; smine_start < dset_info->nelmts; smine_start += smine_nelmts) {
        size_t n; 

        assert(H5S_SELECT_ITER_NELMTS(file_iter) == (dset_info->nelmts - smine_start));

        
        if (in_place_tconv) {
            
            assert(!H5D__SCATGATH_USE_CMPD_OPT_READ(dset_info, in_place_tconv));
            if (dset_info->type_info.need_bkg && (io_info->use_select_io != H5D_SELECTION_IO_MODE_ON))
                smine_nelmts =
                    (size_t)MIN(dset_info->type_info.request_nelmts, (dset_info->nelmts - smine_start));
            else {
                assert(smine_start == 0);
                smine_nelmts = dset_info->nelmts;
            }

            
            tmp_buf = (uint8_t *)buf + dset_info->layout_io_info.contig_piece_info->buf_off +
                      (smine_start * dset_info->type_info.dst_type_size);
        }
        else {
            
            tmp_buf = io_info->tconv_buf;

            
            smine_nelmts =
                (size_t)MIN(dset_info->type_info.request_nelmts, (dset_info->nelmts - smine_start));
        }

        

        
        if ((H5T_BKG_YES == dset_info->type_info.need_bkg) &&
            !H5D__SCATGATH_USE_CMPD_OPT_READ(dset_info, in_place_tconv)) {
            n = H5D__gather_mem(buf, bkg_iter, smine_nelmts, io_info->bkg_buf );
            if (n != smine_nelmts)
                HGOTO_ERROR(H5E_IO, H5E_READERROR, FAIL, "mem gather failed");
        }

        
        n = H5D__gather_file(io_info, dset_info, file_iter, smine_nelmts, tmp_buf );
        if (n != smine_nelmts)
            HGOTO_ERROR(H5E_IO, H5E_READERROR, FAIL, "file gather failed");

        
        if (H5D__SCATGATH_USE_CMPD_OPT_READ(dset_info, in_place_tconv)) {
            if (H5D__compound_opt_read(smine_nelmts, mem_iter, &dset_info->type_info, tmp_buf, buf ) <
                0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "datatype conversion failed");
        } 
        else {
            
            if (H5T_convert(dset_info->type_info.tpath, dset_info->type_info.src_type,
                            dset_info->type_info.dst_type, smine_nelmts, (size_t)0, (size_t)0, tmp_buf,
                            io_info->bkg_buf) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTCONVERT, FAIL, "datatype conversion failed");

            
            if (!dset_info->type_info.is_xform_noop) {
                H5Z_data_xform_t *data_transform; 

                
                if (H5CX_get_data_transform(&data_transform) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get data transform info");

                if (H5Z_xform_eval(data_transform, tmp_buf, smine_nelmts, dset_info->type_info.mem_type) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_BADVALUE, FAIL, "Error performing data transform");
            }

            
            if (!in_place_tconv)
                if (H5D__scatter_mem(tmp_buf, mem_iter, smine_nelmts, buf ) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_READERROR, FAIL, "scatter failed");
        } 
    }     

done:
    
    if (file_iter_init && H5S_SELECT_ITER_RELEASE(file_iter) < 0)
        HDONE_ERROR(H5E_DATASET, H5E_CANTFREE, FAIL, "Can't release selection iterator");
    if (file_iter)
        file_iter = H5FL_FREE(H5S_sel_iter_t, file_iter);
    if (mem_iter_init && H5S_SELECT_ITER_RELEASE(mem_iter) < 0)
        HDONE_ERROR(H5E_DATASET, H5E_CANTFREE, FAIL, "Can't release selection iterator");
    if (mem_iter)
        mem_iter = H5FL_FREE(H5S_sel_iter_t, mem_iter);
    if (bkg_iter_init && H5S_SELECT_ITER_RELEASE(bkg_iter) < 0)
        HDONE_ERROR(H5E_DATASET, H5E_CANTFREE, FAIL, "Can't release selection iterator");
    if (bkg_iter)
        bkg_iter = H5FL_FREE(H5S_sel_iter_t, bkg_iter);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D__scatgath_write(const H5D_io_info_t *io_info, const H5D_dset_io_info_t *dset_info)
{
    const void     *buf;                    
    void           *tmp_buf;                
    H5S_sel_iter_t *mem_iter       = NULL;  
    bool            mem_iter_init  = false; 
    H5S_sel_iter_t *bkg_iter       = NULL;  
    bool            bkg_iter_init  = false; 
    H5S_sel_iter_t *file_iter      = NULL;  
    bool            file_iter_init = false; 
    hsize_t         smine_start;            
    size_t          smine_nelmts;           
    bool            in_place_tconv;         
    herr_t          ret_value = SUCCEED;    

    FUNC_ENTER_PACKAGE

    
    assert(io_info);
    assert(dset_info);
    assert(dset_info->mem_space);
    assert(dset_info->file_space);
    assert(dset_info->buf.cvp);

    
    buf = dset_info->buf.cvp;

    
    if (dset_info->nelmts == 0)
        HGOTO_DONE(SUCCEED);

    
    in_place_tconv = dset_info->layout_io_info.contig_piece_info &&
                     dset_info->layout_io_info.contig_piece_info->in_place_tconv;

    
    if (in_place_tconv && H5D__SCATGATH_USE_CMPD_OPT_WRITE(dset_info, false) &&
        (io_info->use_select_io != H5D_SELECTION_IO_MODE_ON) &&
        (dset_info->type_info.need_bkg || (dset_info->nelmts <= dset_info->type_info.request_nelmts)))
        in_place_tconv = false;

    
    if (NULL == (mem_iter = H5FL_MALLOC(H5S_sel_iter_t)))
        HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL, "can't allocate memory iterator");
    if (NULL == (bkg_iter = H5FL_MALLOC(H5S_sel_iter_t)))
        HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL, "can't allocate background iterator");
    if (NULL == (file_iter = H5FL_MALLOC(H5S_sel_iter_t)))
        HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL, "can't allocate file iterator");

    
    if (H5S_select_iter_init(file_iter, dset_info->file_space, dset_info->type_info.dst_type_size,
                             H5S_SEL_ITER_GET_SEQ_LIST_SORTED) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to initialize file selection information");
    file_iter_init = true; 
    if (H5S_select_iter_init(mem_iter, dset_info->mem_space, dset_info->type_info.src_type_size, 0) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to initialize memory selection information");
    mem_iter_init = true; 
    if (H5S_select_iter_init(bkg_iter, dset_info->file_space, dset_info->type_info.dst_type_size,
                             H5S_SEL_ITER_GET_SEQ_LIST_SORTED) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to initialize background selection information");
    bkg_iter_init = true; 

    
    for (smine_start = 0; smine_start < dset_info->nelmts; smine_start += smine_nelmts) {
        size_t n; 

        assert(H5S_SELECT_ITER_NELMTS(file_iter) == (dset_info->nelmts - smine_start));

        
        if (in_place_tconv) {
            
            assert(!H5D__SCATGATH_USE_CMPD_OPT_WRITE(dset_info, in_place_tconv));
            if (dset_info->type_info.need_bkg && (io_info->use_select_io != H5D_SELECTION_IO_MODE_ON))
                smine_nelmts =
                    (size_t)MIN(dset_info->type_info.request_nelmts, (dset_info->nelmts - smine_start));
            else {
                assert(smine_start == 0);
                smine_nelmts = dset_info->nelmts;
            }

            
            
            tmp_buf = (uint8_t *)dset_info->buf.vp + dset_info->layout_io_info.contig_piece_info->buf_off +
                      (smine_start * dset_info->type_info.src_type_size);
        }
        else {
            
            tmp_buf = io_info->tconv_buf;

            
            smine_nelmts =
                (size_t)MIN(dset_info->type_info.request_nelmts, (dset_info->nelmts - smine_start));

            
            n = H5D__gather_mem(buf, mem_iter, smine_nelmts, tmp_buf );
            if (n != smine_nelmts)
                HGOTO_ERROR(H5E_IO, H5E_WRITEERROR, FAIL, "mem gather failed");
        }

        
        if (H5D__SCATGATH_USE_CMPD_OPT_WRITE(dset_info, in_place_tconv)) {
            if (H5D__compound_opt_write(smine_nelmts, &dset_info->type_info, tmp_buf) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "datatype conversion failed");

        } 
        else {
            if (H5T_BKG_YES == dset_info->type_info.need_bkg) {
                n = H5D__gather_file(io_info, dset_info, bkg_iter, smine_nelmts, io_info->bkg_buf );
                if (n != smine_nelmts)
                    HGOTO_ERROR(H5E_IO, H5E_READERROR, FAIL, "file gather failed");
            } 

            
            if (!dset_info->type_info.is_xform_noop) {
                H5Z_data_xform_t *data_transform; 

                
                if (H5CX_get_data_transform(&data_transform) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get data transform info");

                if (H5Z_xform_eval(data_transform, tmp_buf, smine_nelmts, dset_info->type_info.mem_type) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_BADVALUE, FAIL, "Error performing data transform");
            }

            
            if (H5T_convert(dset_info->type_info.tpath, dset_info->type_info.src_type,
                            dset_info->type_info.dst_type, smine_nelmts, (size_t)0, (size_t)0, tmp_buf,
                            io_info->bkg_buf) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTCONVERT, FAIL, "datatype conversion failed");
        } 

        
        if (H5D__scatter_file(io_info, dset_info, file_iter, smine_nelmts, tmp_buf) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_WRITEERROR, FAIL, "scatter failed");
    } 

done:
    
    if (file_iter_init && H5S_SELECT_ITER_RELEASE(file_iter) < 0)
        HDONE_ERROR(H5E_DATASET, H5E_CANTFREE, FAIL, "Can't release selection iterator");
    if (file_iter)
        file_iter = H5FL_FREE(H5S_sel_iter_t, file_iter);
    if (mem_iter_init && H5S_SELECT_ITER_RELEASE(mem_iter) < 0)
        HDONE_ERROR(H5E_DATASET, H5E_CANTFREE, FAIL, "Can't release selection iterator");
    if (mem_iter)
        mem_iter = H5FL_FREE(H5S_sel_iter_t, mem_iter);
    if (bkg_iter_init && H5S_SELECT_ITER_RELEASE(bkg_iter) < 0)
        HDONE_ERROR(H5E_DATASET, H5E_CANTFREE, FAIL, "Can't release selection iterator");
    if (bkg_iter)
        bkg_iter = H5FL_FREE(H5S_sel_iter_t, bkg_iter);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D__scatgath_read_select(H5D_io_info_t *io_info)
{
    H5S_t         **tmp_mem_spaces   = NULL;  
    H5S_sel_iter_t *mem_iter         = NULL;  
    bool            mem_iter_init    = false; 
    void          **tmp_bufs         = NULL;  
    void           *tmp_bkg_buf      = NULL;  
    size_t          tconv_bytes_used = 0;     
    size_t          bkg_bytes_used   = 0;     
    size_t          i;                        
    herr_t          ret_value = SUCCEED;      

    FUNC_ENTER_PACKAGE

    
    assert(io_info);
    assert(io_info->count > 0);
    assert(io_info->mem_spaces || io_info->pieces_added == 0);
    assert(io_info->file_spaces || io_info->pieces_added == 0);
    assert(io_info->addrs || io_info->pieces_added == 0);
    assert(io_info->element_sizes || io_info->pieces_added == 0);
    assert(io_info->rbufs || io_info->pieces_added == 0);

    
    if (NULL == (tmp_bufs = H5MM_malloc(io_info->pieces_added * sizeof(void *))))
        HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL, "memory allocation failed for temporary buffer list");

    
    if (NULL == (mem_iter = H5FL_MALLOC(H5S_sel_iter_t)))
        HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL, "can't allocate memory iterator");

    
    
    if (NULL == (tmp_mem_spaces = H5MM_malloc(io_info->pieces_added * sizeof(H5S_t *))))
        HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL,
                    "memory allocation failed for temporary memory space list");

    
    for (i = 0; i < io_info->pieces_added; i++) {
        H5D_dset_io_info_t *dset_info = io_info->sel_pieces[i]->dset_info;

        assert(io_info->sel_pieces[i]->piece_points > 0);

        
        if (dset_info->type_info.is_xform_noop && dset_info->type_info.is_conv_noop) {
            
            tmp_mem_spaces[i] = io_info->mem_spaces[i];
            tmp_bufs[i]       = io_info->rbufs[i];
        }
        else {
            
            if (NULL ==
                (tmp_mem_spaces[i] = H5S_create_simple(1, &io_info->sel_pieces[i]->piece_points, NULL))) {
                memset(&tmp_mem_spaces[i], 0, (io_info->pieces_added - i) * sizeof(tmp_mem_spaces[0]));
                HGOTO_ERROR(H5E_DATASET, H5E_CANTCREATE, FAIL, "unable to create simple memory dataspace");
            }

            
            if (io_info->sel_pieces[i]->in_place_tconv)
                
                tmp_bufs[i] = (uint8_t *)(io_info->rbufs[i]) + io_info->sel_pieces[i]->buf_off;
            else {
                
                tmp_bufs[i] = io_info->tconv_buf + tconv_bytes_used;
                tconv_bytes_used +=
                    io_info->sel_pieces[i]->piece_points *
                    MAX(dset_info->type_info.src_type_size, dset_info->type_info.dst_type_size);
                assert(tconv_bytes_used <= io_info->tconv_buf_size);
            }

            
            if (!H5D__SCATGATH_USE_CMPD_OPT_READ(dset_info, io_info->sel_pieces[i]->in_place_tconv)) {
                
                if (dset_info->type_info.need_bkg) {
                    assert(io_info->bkg_buf);

                    
                    tmp_bkg_buf = io_info->bkg_buf + bkg_bytes_used;
                    bkg_bytes_used +=
                        io_info->sel_pieces[i]->piece_points * dset_info->type_info.dst_type_size;
                    assert(bkg_bytes_used <= io_info->bkg_buf_size);

                    
                    if (H5T_BKG_YES == dset_info->type_info.need_bkg) {
                        
                        assert(!mem_iter_init);
                        if (H5S_select_iter_init(mem_iter, io_info->mem_spaces[i],
                                                 dset_info->type_info.dst_type_size, 0) < 0)
                            HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL,
                                        "unable to initialize memory selection information");
                        mem_iter_init = true; 

                        if ((size_t)io_info->sel_pieces[i]->piece_points !=
                            H5D__gather_mem(io_info->rbufs[i], mem_iter,
                                            (size_t)io_info->sel_pieces[i]->piece_points,
                                            tmp_bkg_buf ))
                            HGOTO_ERROR(H5E_IO, H5E_READERROR, FAIL, "mem gather failed");

                        
                        assert(mem_iter_init);
                        if (H5S_SELECT_ITER_RELEASE(mem_iter) < 0)
                            HGOTO_ERROR(H5E_DATASET, H5E_CANTFREE, FAIL, "Can't release selection iterator");
                        mem_iter_init = false;
                    }
                }
            }
        }
    }

    
    H5_CHECK_OVERFLOW(io_info->pieces_added, size_t, uint32_t);
    if (H5F_shared_select_read(io_info->f_sh, H5FD_MEM_DRAW, (uint32_t)io_info->pieces_added, tmp_mem_spaces,
                               io_info->file_spaces, io_info->addrs, io_info->element_sizes, tmp_bufs) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_READERROR, FAIL, "selection read failed");

    
    bkg_bytes_used = 0;

    
    for (i = 0; i < io_info->pieces_added; i++) {
        H5D_dset_io_info_t *dset_info = io_info->sel_pieces[i]->dset_info;

        assert(tmp_mem_spaces[i]);

        
        if (tmp_mem_spaces[i] != io_info->mem_spaces[i]) {
            H5_CHECK_OVERFLOW(io_info->sel_pieces[i]->piece_points, hsize_t, size_t);

            
            assert(!mem_iter_init);
            if (H5S_select_iter_init(mem_iter, io_info->mem_spaces[i], dset_info->type_info.dst_type_size,
                                     0) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL,
                            "unable to initialize memory selection information");
            mem_iter_init = true; 

            
            if (H5D__SCATGATH_USE_CMPD_OPT_READ(dset_info, io_info->sel_pieces[i]->in_place_tconv)) {
                if (H5D__compound_opt_read((size_t)io_info->sel_pieces[i]->piece_points, mem_iter,
                                           &dset_info->type_info, tmp_bufs[i], io_info->rbufs[i] ) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "datatype conversion failed");
            }
            else {
                
                if (dset_info->type_info.need_bkg) {
                    assert(io_info->bkg_buf);

                    
                    tmp_bkg_buf = io_info->bkg_buf + bkg_bytes_used;
                    bkg_bytes_used +=
                        io_info->sel_pieces[i]->piece_points * dset_info->type_info.dst_type_size;
                    assert(bkg_bytes_used <= io_info->bkg_buf_size);
                }

                
                if (H5T_convert(dset_info->type_info.tpath, dset_info->type_info.src_type,
                                dset_info->type_info.dst_type, (size_t)io_info->sel_pieces[i]->piece_points,
                                (size_t)0, (size_t)0, tmp_bufs[i], tmp_bkg_buf) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_CANTCONVERT, FAIL, "datatype conversion failed");

                
                if (!dset_info->type_info.is_xform_noop) {
                    H5Z_data_xform_t *data_transform; 

                    
                    if (H5CX_get_data_transform(&data_transform) < 0)
                        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get data transform info");

                    if (H5Z_xform_eval(data_transform, tmp_bufs[i],
                                       (size_t)io_info->sel_pieces[i]->piece_points,
                                       dset_info->type_info.mem_type) < 0)
                        HGOTO_ERROR(H5E_DATASET, H5E_BADVALUE, FAIL, "Error performing data transform");
                }

                
                if (!io_info->sel_pieces[i]->in_place_tconv)
                    if (H5D__scatter_mem(tmp_bufs[i], mem_iter, (size_t)io_info->sel_pieces[i]->piece_points,
                                         io_info->rbufs[i] ) < 0)
                        HGOTO_ERROR(H5E_DATASET, H5E_READERROR, FAIL, "scatter failed");
            }

            
            assert(mem_iter_init);
            if (H5S_SELECT_ITER_RELEASE(mem_iter) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTFREE, FAIL, "Can't release selection iterator");
            mem_iter_init = false;
        }
    }

done:
    
    if (mem_iter_init && H5S_SELECT_ITER_RELEASE(mem_iter) < 0)
        HDONE_ERROR(H5E_DATASET, H5E_CANTFREE, FAIL, "Can't release selection iterator");
    if (mem_iter)
        mem_iter = H5FL_FREE(H5S_sel_iter_t, mem_iter);

    
    H5MM_free(tmp_bufs);
    tmp_bufs = NULL;

    
    if (tmp_mem_spaces) {
        for (i = 0; i < io_info->pieces_added; i++)
            if (tmp_mem_spaces[i] != io_info->mem_spaces[i] && tmp_mem_spaces[i] &&
                H5S_close(tmp_mem_spaces[i]) < 0)
                HDONE_ERROR(H5E_DATASET, H5E_CLOSEERROR, FAIL, "Can't close dataspace");
        H5MM_free(tmp_mem_spaces);
        tmp_mem_spaces = NULL;
    }

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D__scatgath_write_select(H5D_io_info_t *io_info)
{
    H5S_t         **write_mem_spaces  = NULL;  
    size_t          spaces_added      = 0;     
    H5S_sel_iter_t *mem_iter          = NULL;  
    bool            mem_iter_init     = false; 
    const void    **write_bufs        = NULL;  
    size_t          tconv_bytes_used  = 0;     
    size_t          bkg_bytes_used    = 0;     
    H5S_t         **bkg_mem_spaces    = NULL;  
    H5S_t         **bkg_file_spaces   = NULL;  
    haddr_t        *bkg_addrs         = NULL;  
    size_t         *bkg_element_sizes = NULL;  
    void          **bkg_bufs   = NULL; 
    size_t          bkg_pieces = 0;    
    size_t          i;                 
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(io_info);
    assert(io_info->count > 0);
    assert(io_info->mem_spaces || io_info->pieces_added == 0);
    assert(io_info->file_spaces || io_info->pieces_added == 0);
    assert(io_info->addrs || io_info->pieces_added == 0);
    assert(io_info->element_sizes || io_info->pieces_added == 0);
    assert(io_info->wbufs || io_info->pieces_added == 0);

    
    if (NULL == (write_bufs = (const void **)H5MM_malloc(io_info->pieces_added * sizeof(const void *))))
        HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL, "memory allocation failed for temporary buffer list");

    
    if (NULL == (mem_iter = H5FL_MALLOC(H5S_sel_iter_t)))
        HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL, "can't allocate memory iterator");

    
    
    if (NULL == (write_mem_spaces = H5MM_malloc(io_info->pieces_added * sizeof(H5S_t *))))
        HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL,
                    "memory allocation failed for temporary memory space list");

    
    for (i = 0; i < io_info->pieces_added; i++) {
        H5D_dset_io_info_t *dset_info = io_info->sel_pieces[i]->dset_info;

        assert(io_info->sel_pieces[i]->piece_points > 0);

        
        if (dset_info->type_info.is_xform_noop && dset_info->type_info.is_conv_noop) {
            
            write_mem_spaces[i] = io_info->mem_spaces[i];
            spaces_added++;
            write_bufs[i] = io_info->wbufs[i];
        }
        else {
            void *tmp_write_buf; 
            void *tmp_bkg_buf = NULL;

            H5_CHECK_OVERFLOW(io_info->sel_pieces[i]->piece_points, hsize_t, size_t);

            
            assert(!mem_iter_init);
            if (H5S_select_iter_init(mem_iter, io_info->mem_spaces[i], dset_info->type_info.src_type_size,
                                     0) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL,
                            "unable to initialize memory selection information");
            mem_iter_init = true; 

            
            if (NULL ==
                (write_mem_spaces[i] = H5S_create_simple(1, &io_info->sel_pieces[i]->piece_points, NULL)))
                HGOTO_ERROR(H5E_DATASET, H5E_CANTCREATE, FAIL, "unable to create simple memory dataspace");
            spaces_added++;

            
            if (io_info->sel_pieces[i]->in_place_tconv) {
                H5_flexible_const_ptr_t flex_buf;

                
                
                flex_buf.cvp  = io_info->wbufs[i];
                tmp_write_buf = (uint8_t *)flex_buf.vp + io_info->sel_pieces[i]->buf_off;
            }
            else {
                
                tmp_write_buf = io_info->tconv_buf + tconv_bytes_used;
                tconv_bytes_used +=
                    io_info->sel_pieces[i]->piece_points *
                    MAX(dset_info->type_info.src_type_size, dset_info->type_info.dst_type_size);
                assert(tconv_bytes_used <= io_info->tconv_buf_size);

                
                if ((size_t)io_info->sel_pieces[i]->piece_points !=
                    H5D__gather_mem(io_info->wbufs[i], mem_iter, (size_t)io_info->sel_pieces[i]->piece_points,
                                    tmp_write_buf ))
                    HGOTO_ERROR(H5E_IO, H5E_WRITEERROR, FAIL, "mem gather failed");
            }

            
            write_bufs[i] = (const void *)tmp_write_buf;

            
            if (H5D__SCATGATH_USE_CMPD_OPT_WRITE(dset_info, io_info->sel_pieces[i]->in_place_tconv)) {
                if (H5D__compound_opt_write((size_t)io_info->sel_pieces[i]->piece_points,
                                            &dset_info->type_info, tmp_write_buf) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "datatype conversion failed");

            } 
            else {
                
                if (dset_info->type_info.need_bkg) {
                    assert(io_info->bkg_buf);

                    
                    tmp_bkg_buf = io_info->bkg_buf + bkg_bytes_used;
                    bkg_bytes_used +=
                        io_info->sel_pieces[i]->piece_points * dset_info->type_info.dst_type_size;
                    assert(bkg_bytes_used <= io_info->bkg_buf_size);
                }

                
                if (H5T_BKG_YES == dset_info->type_info.need_bkg) {
                    assert(io_info->must_fill_bkg);

                    
                    if (!bkg_mem_spaces) {
                        assert(!bkg_file_spaces && !bkg_addrs && !bkg_element_sizes && !bkg_bufs);
                        if (NULL == (bkg_mem_spaces = H5MM_malloc(io_info->pieces_added * sizeof(H5S_t *))))
                            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL,
                                        "memory allocation failed for memory space list");
                        if (NULL == (bkg_file_spaces = H5MM_malloc(io_info->pieces_added * sizeof(H5S_t *))))
                            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL,
                                        "memory allocation failed for file space list");
                        if (NULL == (bkg_addrs = H5MM_malloc(io_info->pieces_added * sizeof(haddr_t))))
                            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL,
                                        "memory allocation failed for piece address list");
                        if (NULL == (bkg_element_sizes = H5MM_malloc(io_info->pieces_added * sizeof(size_t))))
                            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL,
                                        "memory allocation failed for element size list");
                        if (NULL == (bkg_bufs = H5MM_malloc(io_info->pieces_added * sizeof(const void *))))
                            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL,
                                        "memory allocation failed for write buffer list");
                    }

                    
                    assert(bkg_mem_spaces && bkg_file_spaces && bkg_addrs && bkg_element_sizes && bkg_bufs);
                    bkg_mem_spaces[bkg_pieces]    = write_mem_spaces[i];
                    bkg_file_spaces[bkg_pieces]   = io_info->file_spaces[i];
                    bkg_addrs[bkg_pieces]         = io_info->addrs[i];
                    bkg_element_sizes[bkg_pieces] = io_info->element_sizes[i];

                    
                    bkg_bufs[bkg_pieces] = tmp_bkg_buf;

                    
                    bkg_pieces++;
                }
                else {
                    
                    
                    if (!dset_info->type_info.is_xform_noop) {
                        H5Z_data_xform_t *data_transform; 

                        
                        if (H5CX_get_data_transform(&data_transform) < 0)
                            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get data transform info");

                        if (H5Z_xform_eval(data_transform, tmp_write_buf,
                                           (size_t)io_info->sel_pieces[i]->piece_points,
                                           dset_info->type_info.mem_type) < 0)
                            HGOTO_ERROR(H5E_DATASET, H5E_BADVALUE, FAIL, "Error performing data transform");
                    }

                    
                    if (H5T_convert(dset_info->type_info.tpath, dset_info->type_info.src_type,
                                    dset_info->type_info.dst_type,
                                    (size_t)io_info->sel_pieces[i]->piece_points, (size_t)0, (size_t)0,
                                    tmp_write_buf, tmp_bkg_buf) < 0)
                        HGOTO_ERROR(H5E_DATASET, H5E_CANTCONVERT, FAIL, "datatype conversion failed");
                }
            }

            
            assert(mem_iter_init);
            if (H5S_SELECT_ITER_RELEASE(mem_iter) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTFREE, FAIL, "Can't release selection iterator");
            mem_iter_init = false;
        }
    }

    assert(spaces_added == io_info->pieces_added);

    
    if (io_info->must_fill_bkg) {
        size_t j = 0; 

        
        H5_CHECK_OVERFLOW(bkg_pieces, size_t, uint32_t);
        if (H5F_shared_select_read(io_info->f_sh, H5FD_MEM_DRAW, (uint32_t)bkg_pieces, bkg_mem_spaces,
                                   bkg_file_spaces, bkg_addrs, bkg_element_sizes, bkg_bufs) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_READERROR, FAIL, "selection read to background buffer failed");

        
        for (i = 0; i < io_info->pieces_added; i++) {
            H5D_dset_io_info_t *dset_info = io_info->sel_pieces[i]->dset_info;

            if ((H5T_BKG_YES == dset_info->type_info.need_bkg) &&
                !H5D__SCATGATH_USE_CMPD_OPT_WRITE(dset_info, io_info->sel_pieces[i]->in_place_tconv)) {
                
                void *tmp_write_buf =
                    (void *)((uint8_t *)io_info->tconv_buf +
                             ((const uint8_t *)write_bufs[i] - (const uint8_t *)io_info->tconv_buf));

                
                if (!dset_info->type_info.is_xform_noop) {
                    H5Z_data_xform_t *data_transform; 

                    
                    if (H5CX_get_data_transform(&data_transform) < 0)
                        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get data transform info");

                    if (H5Z_xform_eval(data_transform, tmp_write_buf,
                                       (size_t)io_info->sel_pieces[i]->piece_points,
                                       dset_info->type_info.mem_type) < 0)
                        HGOTO_ERROR(H5E_DATASET, H5E_BADVALUE, FAIL, "Error performing data transform");
                }

                
                assert(j < bkg_pieces);
                if (H5T_convert(dset_info->type_info.tpath, dset_info->type_info.src_type,
                                dset_info->type_info.dst_type, (size_t)io_info->sel_pieces[i]->piece_points,
                                (size_t)0, (size_t)0, tmp_write_buf, bkg_bufs[j]) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_CANTCONVERT, FAIL, "datatype conversion failed");

                
                j++;
            }
        }

        assert(j == bkg_pieces);
    }

    
    H5_CHECK_OVERFLOW(io_info->pieces_added, size_t, uint32_t);
    if (H5F_shared_select_write(io_info->f_sh, H5FD_MEM_DRAW, (uint32_t)io_info->pieces_added,
                                write_mem_spaces, io_info->file_spaces, io_info->addrs,
                                io_info->element_sizes, write_bufs) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_WRITEERROR, FAIL, "selection write failed");

done:
    
    if (mem_iter_init && H5S_SELECT_ITER_RELEASE(mem_iter) < 0)
        HDONE_ERROR(H5E_DATASET, H5E_CANTFREE, FAIL, "Can't release selection iterator");
    if (mem_iter)
        mem_iter = H5FL_FREE(H5S_sel_iter_t, mem_iter);

    
    H5MM_free(write_bufs);
    write_bufs = NULL;

    
    if (write_mem_spaces) {
        for (i = 0; i < spaces_added; i++) {
            assert(write_mem_spaces[i]);
            if (write_mem_spaces[i] != io_info->mem_spaces[i] && H5S_close(write_mem_spaces[i]) < 0)
                HDONE_ERROR(H5E_DATASET, H5E_CLOSEERROR, FAIL, "Can't close dataspace");
        }
        H5MM_free(write_mem_spaces);
        write_mem_spaces = NULL;
    }

    
    H5MM_free(bkg_mem_spaces);
    bkg_mem_spaces = NULL;
    H5MM_free(bkg_file_spaces);
    bkg_file_spaces = NULL;
    H5MM_free(bkg_addrs);
    bkg_addrs = NULL;
    H5MM_free(bkg_element_sizes);
    bkg_element_sizes = NULL;
    H5MM_free(bkg_bufs);
    bkg_bufs = NULL;

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__compound_opt_read(size_t nelmts, H5S_sel_iter_t *iter, const H5D_type_info_t *type_info,
                       uint8_t *tconv_buf, void *user_buf )
{
    uint8_t *ubuf = (uint8_t *)user_buf; 
    uint8_t *xdbuf;                      
    hsize_t *off = NULL;                 
    size_t  *len = NULL;                 
    size_t   src_stride, dst_stride, copy_size;
    size_t   dxpl_vec_size;       
    size_t   vec_size;            
    herr_t   ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(nelmts > 0);
    assert(iter);
    assert(type_info);
    assert(type_info->cmpd_subset);
    assert(H5T_SUBSET_SRC == type_info->cmpd_subset->subset ||
           H5T_SUBSET_DST == type_info->cmpd_subset->subset);
    assert(user_buf);

    
    if (H5CX_get_vec_size(&dxpl_vec_size) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't retrieve I/O vector size");

    
    if (dxpl_vec_size > H5D_IO_VECTOR_SIZE)
        vec_size = dxpl_vec_size;
    else
        vec_size = H5D_IO_VECTOR_SIZE;
    if (NULL == (len = H5FL_SEQ_MALLOC(size_t, vec_size)))
        HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL, "can't allocate I/O length vector array");
    if (NULL == (off = H5FL_SEQ_MALLOC(hsize_t, vec_size)))
        HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL, "can't allocate I/O offset vector array");

    
    src_stride = type_info->src_type_size;
    dst_stride = type_info->dst_type_size;

    
    copy_size = type_info->cmpd_subset->copy_size;

    
    xdbuf = tconv_buf;
    while (nelmts > 0) {
        size_t nseq;     
        size_t curr_seq; 
        size_t elmtno;   

        
        if (H5S_SELECT_ITER_GET_SEQ_LIST(iter, vec_size, nelmts, &nseq, &elmtno, off, len) < 0)
            HGOTO_ERROR(H5E_INTERNAL, H5E_UNSUPPORTED, 0, "sequence length generation failed");

        
        for (curr_seq = 0; curr_seq < nseq; curr_seq++) {
            size_t   curr_off;    
            size_t   curr_len;    
            size_t   curr_nelmts; 
            uint8_t *xubuf;
            size_t   i; 

            
            curr_len = len[curr_seq];
            H5_CHECK_OVERFLOW(off[curr_seq], hsize_t, size_t);
            curr_off = (size_t)off[curr_seq];

            
            curr_nelmts = curr_len / dst_stride;
            xubuf       = ubuf + curr_off;

            
            for (i = 0; i < curr_nelmts; i++) {
                memmove(xubuf, xdbuf, copy_size);

                
                xdbuf += src_stride;
                xubuf += dst_stride;
            } 
        }     

        
        nelmts -= elmtno;
    } 

done:
    
    if (len)
        len = H5FL_SEQ_FREE(size_t, len);
    if (off)
        off = H5FL_SEQ_FREE(hsize_t, off);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__compound_opt_write(size_t nelmts, const H5D_type_info_t *type_info, void *tconv_buf)
{
    uint8_t *xsbuf, *xdbuf;          
    size_t   src_stride, dst_stride; 
    size_t   i;                      

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(nelmts > 0);
    assert(type_info);

    
    src_stride = type_info->src_type_size;
    dst_stride = type_info->dst_type_size;

    
    xsbuf = tconv_buf;
    xdbuf = tconv_buf;
    for (i = 0; i < nelmts; i++) {
        memmove(xdbuf, xsbuf, dst_stride);

        
        xsbuf += src_stride;
        xdbuf += dst_stride;
    } 

    FUNC_LEAVE_NOAPI(SUCCEED)
} 
