/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Omodule.h" 
#define H5T_FRIEND     

#include "H5private.h"   
#include "H5CXprivate.h" 
#include "H5Dprivate.h"  
#include "H5Eprivate.h"  
#include "H5Fprivate.h"  
#include "H5Gprivate.h"  
#include "H5Iprivate.h"  
#include "H5Opkg.h"      
#include "H5Tpkg.h"      

static herr_t H5O__oh_tag(const H5O_loc_t *oloc, haddr_t *tag);
static herr_t H5O__refresh_metadata_close(H5O_loc_t *oloc, H5G_loc_t *obj_loc, hid_t oid);

herr_t
H5O_flush(H5O_loc_t *oloc, hid_t obj_id)
{
    void                  *obj_ptr;             
    const H5O_obj_class_t *obj_class;           
    herr_t                 ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    if (H5F_HAS_FEATURE(oloc->file, H5FD_FEAT_HAS_MPI))
        HGOTO_ERROR(H5E_ARGS, H5E_UNSUPPORTED, FAIL, "H5Oflush isn't supported for parallel");

    
    if (NULL == (obj_ptr = H5VL_object(obj_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid object identifier");

    
    if (NULL == (obj_class = H5O__obj_class(oloc)))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTINIT, FAIL, "unable to determine object class");

    
    if (obj_class->flush && obj_class->flush(obj_ptr) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTFLUSH, FAIL, "unable to flush object");

    
    if (H5O_flush_common(oloc, obj_id) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTFLUSH, FAIL, "unable to flush object and object flush callback");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5O_flush_common(H5O_loc_t *oloc, hid_t obj_id)
{
    haddr_t tag       = 0;
    herr_t  ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    if (H5O__oh_tag(oloc, &tag) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTFLUSH, FAIL, "unable to flush object metadata");

    
    if (H5F_flush_tagged_metadata(oloc->file, tag) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTFLUSH, FAIL, "unable to flush tagged metadata");

    
    if (H5F_object_flush_cb(oloc->file, obj_id) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTFLUSH, FAIL, "unable to do object flush callback");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__oh_tag(const H5O_loc_t *oloc, haddr_t *tag)
{
    H5O_t *oh        = NULL;    
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(oloc);

    
    if (NULL == (oh = H5O_protect(oloc, H5AC__READ_ONLY_FLAG, false)))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTPROTECT, FAIL, "unable to protect object's object header");

    
    if (HADDR_UNDEF == (*tag = H5O_OH_GET_ADDR(oh)))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, FAIL, "unable to get address of object header");

done:
    
    if (oh && H5O_unprotect(oloc, oh, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_OHDR, H5E_CANTUNPROTECT, FAIL, "unable to release object header");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5O_refresh_metadata(H5O_loc_t *oloc, hid_t oid)
{
    H5VL_object_t *vol_obj   = NULL;  
    bool           objs_incr = false; 
    H5F_t         *file      = NULL;
    herr_t         ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    if (!(H5F_INTENT(oloc->file) & H5F_ACC_RDWR)) {
        H5G_loc_t         obj_loc;
        H5O_loc_t         obj_oloc;
        H5G_name_t        obj_path;
        H5O_shared_t      cached_H5O_shared;
        H5VL_connector_t *connector = NULL;

        
        file = oloc->file;

        
        obj_loc.oloc = &obj_oloc;
        obj_loc.path = &obj_path;
        H5G_loc_reset(&obj_loc);

        
        H5F_incr_nopen_objs(oloc->file);
        objs_incr = true;

        
        if (H5I_get_type(oid) == H5I_DATATYPE)
            if (H5T_save_refresh_state(oid, &cached_H5O_shared) < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTOPENOBJ, FAIL, "unable to save datatype state");

        
        if (NULL == (vol_obj = H5VL_vol_object(oid)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid object identifier");
        connector = H5VL_OBJ_CONNECTOR(vol_obj);

        
        H5VL_conn_inc_rc(connector);

        
        if (H5O__refresh_metadata_close(oloc, &obj_loc, oid) < 0) {
            H5VL_conn_dec_rc(connector);
            HGOTO_ERROR(H5E_OHDR, H5E_CANTLOAD, FAIL, "unable to refresh object");
        }

        
        if (H5O_refresh_metadata_reopen(oid, H5P_DEFAULT, &obj_loc, connector, false) < 0) {
            H5VL_conn_dec_rc(connector);
            HGOTO_ERROR(H5E_OHDR, H5E_CANTLOAD, FAIL, "unable to refresh object");
        }

        
        if (H5VL_conn_dec_rc(connector) < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_CANTDEC, FAIL, "can't decrement reference count for connector");

        
        if (H5I_get_type(oid) == H5I_DATATYPE)
            if (H5T_restore_refresh_state(oid, &cached_H5O_shared) < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTOPENOBJ, FAIL, "unable to restore datatype state");

    } 

done:
    if (objs_incr && file)
        H5F_decr_nopen_objs(file);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__refresh_metadata_close(H5O_loc_t *oloc, H5G_loc_t *obj_loc, hid_t oid)
{
    H5F_t  *file;                
    haddr_t tag       = 0;       
    bool    corked    = false;   
    herr_t  ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    if (obj_loc) {
        H5G_loc_t tmp_loc;

        H5G_loc(oid, &tmp_loc);
        H5G_loc_copy(obj_loc, &tmp_loc, H5_COPY_DEEP);
    } 

    
    if (H5I_get_type(oid) == H5I_DATASET)
        if (H5D_mult_refresh_close(oid) < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_CANTOPENOBJ, FAIL, "unable to prepare refresh for dataset");

    
    if (H5O__oh_tag(oloc, &tag) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTFLUSH, FAIL, "unable to get object header address");

    
    if (H5AC_cork(oloc->file, tag, H5AC__GET_CORKED, &corked) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_SYSTEM, FAIL, "unable to retrieve an object's cork status");

    
    file = oloc->file;
    
    if (H5I_dec_ref(oid) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTINIT, FAIL, "unable to close object");

    
    if (H5F_flush_tagged_metadata(file, tag) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTFLUSH, FAIL, "unable to flush tagged metadata");

    
    if (H5AC_evict_tagged_metadata(file, tag, true) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTFLUSH, FAIL, "unable to evict metadata");

    
    if (corked)
        if (H5AC_cork(file, tag, H5AC__SET_CORK, &corked) < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_SYSTEM, FAIL, "unable to cork the object");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5O_refresh_metadata_reopen(hid_t oid, hid_t apl_id, H5G_loc_t *obj_loc, H5VL_connector_t *vol_connector,
                            bool start_swmr)
{
    void      *object = NULL;       
    H5I_type_t type;                
    herr_t     ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(obj_loc);
    assert(vol_connector);

    
    type = H5I_get_type(oid);

    switch (type) {
        case H5I_GROUP:
            
            if (NULL == (object = H5G_open(obj_loc)))
                HGOTO_ERROR(H5E_OHDR, H5E_CANTOPENOBJ, FAIL, "unable to open group");
            break;

        case H5I_DATATYPE:
            
            if (NULL == (object = H5T_open(obj_loc)))
                HGOTO_ERROR(H5E_OHDR, H5E_CANTOPENOBJ, FAIL, "unable to open named datatype");
            break;

        case H5I_DATASET:
            
            if (H5CX_set_apl(&apl_id, H5P_CLS_DACC, oid, true) < 0)
                HGOTO_ERROR(H5E_OHDR, H5E_CANTSET, FAIL, "can't set access property list info");

            
            if (NULL ==
                (object = H5D_open(obj_loc, apl_id == H5P_DEFAULT ? H5P_DATASET_ACCESS_DEFAULT : apl_id)))
                HGOTO_ERROR(H5E_DATASET, H5E_CANTOPENOBJ, FAIL, "unable to open dataset");
            if (!start_swmr) 
                if (H5D_mult_refresh_reopen((H5D_t *)object) < 0)
                    HGOTO_ERROR(H5E_OHDR, H5E_CANTOPENOBJ, FAIL, "unable to finish refresh for dataset");
            break;

        case H5I_MAP:
            HGOTO_ERROR(H5E_OHDR, H5E_BADTYPE, FAIL, "maps not supported in native VOL connector");

        case H5I_UNINIT:
        case H5I_BADID:
        case H5I_FILE:
        case H5I_DATASPACE:
        case H5I_ATTR:
        case H5I_VFL:
        case H5I_VOL:
        case H5I_GENPROP_CLS:
        case H5I_GENPROP_LST:
        case H5I_ERROR_CLASS:
        case H5I_ERROR_MSG:
        case H5I_ERROR_STACK:
        case H5I_SPACE_SEL_ITER:
        case H5I_EVENTSET:
        case H5I_NTYPES:
        default:
            HGOTO_ERROR(H5E_OHDR, H5E_BADTYPE, FAIL,
                        "not a valid file object ID (dataset, group, or datatype)");
            break;
    } 

    
    if ((H5VL_register_using_existing_id(type, object, vol_connector, true, oid)) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTREGISTER, FAIL, "unable to re-register object ID after refresh");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
