#' Simulation-based (indirect) fitting of parametric models
#'
#' Fits an untractable parametric model by simulation
#'
#' @param tobs A vector containing the observed summary statistics.
#' @param tsim A function implementing the model to be simulated. It
#'        must take as arguments a vector of model parameter values and
#'        it must return a vector of summary statistics.
#' @param l,u Two numeric vectors of length equal to the number of the
#'        parameters containing the lower and upper bounds on the parameters.
#' @param cluster if not null, the model will be simulated in parallel;
#'        the argument can be either a cluster object (as defined in package
#'        `parallel`) or a positive integer (in this case the cluster will be created and
#'        deleted at the end by `ifit`).
#' @param export a vector giving the names of the global objects
#'        (tipically needed by function `tsim`) which should be exported to the
#'        cluster; not used if `cluster` is null.
#' @param trace An integer value; if greater than zero, results will be
#'        printed (approximately) every `trace` model simulations.
#' @param Ntotal The maximum number of allowed model simulations.
#' @param NTotglobal The maximum number of simulations performed during
#'        the global search phase.
#' @param
#' Ninit,Nelite,Aelite,Tolglobal,Tollocal,Tolmodel,NFitlocal,NAddglobal,NAddlocal,Rhomax,Lambda
#' Constants affecting the details of the algorithm (see the vignette
#' describing the algorithm)
#' @returns
#'   An object of class `ifit` for which a suitable `print` method
#'   exists and whose elements can be accessed using the
#'   functions described in [ifit-methods]
#' @references
#' Guido Masarotto (2015) 'Simulation-Based Fitting of Intractable
#' Models  via Sequential Sampling and Local Smoothing',
#' arXiv eprint 2511.08180, pp. 1-23, \doi{10.48550/arXiv.2511.08180} 
#' @examples
#' # A toy model
#' set.seed(1)
#' n <- 3
#' y <- rnorm(n, 1)
#' tobs <- mean(y)
#' tsim <- function(theta) mean(rnorm(n, theta[1]))
#' m <- ifit(tobs, tsim, l = -3, u = 3)
#' m
#' coef(m)
#' confint(m)
#' globalIFIT(m)
#' numsimIFIT(m)
#' vcov(m, "parameters")
#' vcov(m, "statistics")
#' jacobianIFIT(m)
#' diagIFIT(m, plot = FALSE)
#' estfunIFIT(m)
#'
#' \donttest{
#' # Logit model 
#' # It takes some time to run
#' n <- 100
#' X <- seq(-1, 1, length = n)
#' Z <- rnorm(n)
#' X <- cbind(1, X, Z, Z + rnorm(n))
#' logitSim <- function(theta) rbinom(n, 1, 1 / (1 + exp(-X %*% theta)))
#' logitStat <- function(y) as.numeric(crossprod(X, y))
#' theta <- c(-1, 1, 0.5, -0.5)
#' y <- logitSim(theta)
#' m <- ifit(
#'     tobs = logitStat(y), tsim = function(t) logitStat(logitSim(t)),
#'     l = rep(-5, 4), u = rep(5, 4), trace = 1000
#' )
#' m
#' g <- glm(y ~ X - 1, family = binomial)
#' rbind(
#'     ifit.estimates = coef(m),
#'     glm.estimates = coef(g),
#'     ifit.se = sqrt(diag(vcov(m))),
#'     glm.se = sqrt(diag(vcov(g)))
#' )
#' }
#' @keywords inference
#' @export
ifit <- function(tobs, tsim, l, u, cluster = NULL, export = NULL,
                 trace = 0, Ntotal = 1000000, NTotglobal = 20000,
                 Ninit = 1000, Nelite = 100, Aelite = 0.5,
                 Tolglobal = 0.1, Tollocal = 1, Tolmodel = 1.5,
                 NFitlocal = 4000, NAddglobal = 100, NAddlocal = 10,
                 Lambda = 0.1, Rhomax = 0.1) {
    ## initialization
    this.call <- match.call()
    p <- length(l)
    q <- length(tobs)
    ctrl <- checkIFIT(
        p, q, l, u,
        trace, Ntotal, NTotglobal, Ninit, Nelite, Aelite,
        Tolglobal, Tollocal, Tolmodel,
        NFitlocal, NAddglobal, NAddlocal, Rhomax, Lambda
    )
    data <- new.env()
    data$n <- Ninit
    data$p <- p
    data$q <- q
    data$par <- matrix(NA, p, Ntotal)
    data$stat <- matrix(NA, q, Ntotal)
    if (!is.null(cluster)) {
        if (is.integer(cluster)) {
            data$cl <- parallel::makeCluster(cluster)
        } else {
            if (inherits(cluster, "cluster")) {
                data$cl <- cluster
            } else {
                stop("argument `cluster` should be an integer or a `parallel` cluster")
            }
        }
        parallel::clusterExport(data$cl, c("tobs", "tsim", export))
        parallel::clusterSetRNGStream(data$cl, floor(stats::runif(1, 2, 10^6)))
    }
    rsim <- function(x) tobs - tsim(x)
    ## initial random latin square
    idx <- seq.int(Ninit)
    data$par[, idx] <- l + (u - l) *
        (replicate(p, order(stats::runif(Ninit))) + matrix(stats::runif(Ninit * p), Ninit) - 1) / Ninit
    data$stat[, idx] <-
        if (is.null(data$cl)) {
            apply(data$par[, idx, drop=FALSE], 2, rsim)
        } else {
            parallel::parCapply(data$cl, data$par[, idx, drop=FALSE], rsim)
        }
    ## ready...go...
    guess <- nnglobalsearch(rsim, l, u, data, ctrl)
    nglobal <- data$n
    ans <- nnlocalsearch(rsim, l, u, guess, data, ctrl)
    ## ...done...return
    ans$call <- this.call
    idx <- seq_len(data$n)
    data$par <- data$par[, idx]
    data$stat <- data$stat[, idx]
    if (!is.null(cluster) && is.integer(cluster)) parallel::stopCluster(data$cl)
    data$cl <- NULL
    ans$data <- data
    ans$nsim <- c(global = nglobal, local = data$n - nglobal)
    ans$global <- guess
    ans$model <- list(tobs = tobs, tsim = tsim, l = l, u = u)
    ctrl$trace <- NULL
    ans$ctrl <- ctrl
    np <- paste0("theta", seq.int(p))
    ns <- paste0("stat", seq.int(q))
    names(ans$guess) <- np
    names(ans$grad) <- np
    names(ans$grad.se) <- np
    colnames(ans$cov) <- rownames(ans$cov) <- np
    names(ans$a) <- ns
    colnames(ans$V) <- rownames(ans$V) <- ns
    colnames(ans$B) <- np
    rownames(ans$B) <- ns
    class(ans) <- "ifit"
    ans
}
