// [[Rcpp::depends(BH)]]

#include <Rcpp.h>
#include <boost/container/flat_set.hpp>
using namespace Rcpp;

typedef boost::container::flat_set<int> fs;
typedef boost::container::flat_set<int>::iterator fsi;

#define UpdateOrbits( delta_add_mat, delta_rem_mat, nodes, x, mod0, mod1) {\
  int pos0 = x + mod0;\
  int pos1 = x + mod1;\
  for (int i = 0; i < lookup_lengths[pos0]; ++i) {\
    delta_rem_mat(nodes[i], lookup_orbits[lookup_position[pos0]+i]) += 1;\
  }\
  for (int i = 0; i < lookup_lengths[pos1]; ++i) {\
    delta_add_mat(nodes[i], lookup_orbits[lookup_position[pos1]+i]) += 1;\
  }\
}

#define W( i0, i1, z ) (containsInt(i0, i1)*z)
static int adj_chunk = 8*sizeof(int);

static int lookup_lengths[] = {0,2,0,3,0,3,3,3,0,0,0,4,0,4,4,4,0,0,0,4,0,4,4,4,0,0,4,4,4,4,4,4,0,0,0,4,0,4,4,4,0,4,0,4,4,4,4,4,0,4,4,4,0,4,4,4,
                               4,4,4,4,4,4,4,4,0,0,0,0,0,0,0,0,0,0,0,5,0,5,5,5,0,0,0,5,0,5,5,5,0,0,5,5,5,5,5,5,0,0,0,5,0,5,5,5,0,5,0,5,5,5,5,5,
                               0,5,5,5,0,5,5,5,5,5,5,5,5,5,5,5,0,0,0,0,0,0,0,0,0,0,0,5,0,5,5,5,0,0,0,5,0,5,5,5,0,0,5,5,5,5,5,5,0,0,0,5,0,5,5,5,
                               0,5,0,5,5,5,5,5,0,5,5,5,0,5,5,5,5,5,5,5,5,5,5,5,0,0,0,0,0,0,0,0,0,0,5,5,5,5,5,5,0,0,5,5,5,5,5,5,0,0,5,5,5,5,5,5,
                               0,0,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,0,0,0,0,0,0,0,0,0,0,0,5,0,5,5,5,0,0,0,5,0,5,5,5,
                               0,0,5,5,5,5,5,5,0,0,0,5,0,5,5,5,0,5,0,5,5,5,5,5,0,5,5,5,0,5,5,5,5,5,5,5,5,5,5,5,0,0,0,0,0,0,0,0,0,5,0,5,5,5,5,5,
                               0,5,0,5,5,5,5,5,5,5,5,5,5,5,5,5,0,5,0,5,5,5,5,5,0,5,0,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,0,0,0,0,0,0,0,0,
                               0,5,5,5,0,5,5,5,0,5,5,5,0,5,5,5,5,5,5,5,5,5,5,5,0,5,5,5,0,5,5,5,5,5,5,5,5,5,5,5,0,5,5,5,0,5,5,5,5,5,5,5,5,5,5,5,
                               0,0,0,0,0,0,0,0,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
                               5,5,5,5,5,5,5,5,0,0,0,0,0,0,0,0,0,0,0,5,0,5,5,5,0,0,0,5,0,5,5,5,0,0,5,5,5,5,5,5,0,0,0,5,0,5,5,5,0,5,0,5,5,5,5,5,
                               0,5,5,5,0,5,5,5,5,5,5,5,5,5,5,5,0,0,0,5,0,5,5,5,0,0,0,5,0,5,5,5,0,0,5,5,5,5,5,5,0,0,5,5,5,5,5,5,0,5,0,5,5,5,5,5,
                               0,5,0,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,0,0,0,5,0,5,5,5,0,0,5,5,5,5,5,5,0,0,0,5,0,5,5,5,0,0,5,5,5,5,5,5,
                               0,5,5,5,0,5,5,5,5,5,5,5,5,5,5,5,0,5,5,5,0,5,5,5,5,5,5,5,5,5,5,5,0,0,5,5,5,5,5,5,0,0,5,5,5,5,5,5,0,0,5,5,5,5,5,5,
                               0,0,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,0,0,0,5,0,5,5,5,0,5,0,5,5,5,5,5,
                               0,5,5,5,0,5,5,5,5,5,5,5,5,5,5,5,0,0,0,5,0,5,5,5,0,5,0,5,5,5,5,5,0,5,5,5,0,5,5,5,5,5,5,5,5,5,5,5,0,5,0,5,5,5,5,5,
                               0,5,0,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,0,5,0,5,5,5,5,5,0,5,0,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
                               0,5,5,5,0,5,5,5,5,5,5,5,5,5,5,5,0,5,5,5,0,5,5,5,5,5,5,5,5,5,5,5,0,5,5,5,0,5,5,5,5,5,5,5,5,5,5,5,0,5,5,5,0,5,5,5,
                               5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
                               5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5};
static int lookup_position[] = {0,0,2,2,5,5,8,11,14,14,14,14,18,18,22,26,30,30,30,30,34,34,38,42,46,46,46,50,54,58,62,66,70,70,70,70,74,74,78,
                                82,86,86,90,90,94,98,102,106,110,110,114,118,122,122,126,130,134,138,142,146,150,154,158,162,166,166,166,166,
                                166,166,166,166,166,166,166,166,171,171,176,181,186,186,186,186,191,191,196,201,206,206,206,211,216,221,226,
                                231,236,236,236,236,241,241,246,251,256,256,261,261,266,271,276,281,286,286,291,296,301,301,306,311,316,321,
                                326,331,336,341,346,351,356,356,356,356,356,356,356,356,356,356,356,356,361,361,366,371,376,376,376,376,381,
                                381,386,391,396,396,396,401,406,411,416,421,426,426,426,426,431,431,436,441,446,446,451,451,456,461,466,471,
                                476,476,481,486,491,491,496,501,506,511,516,521,526,531,536,541,546,546,546,546,546,546,546,546,546,546,546,
                                551,556,561,566,571,576,576,576,581,586,591,596,601,606,606,606,611,616,621,626,631,636,636,636,641,646,651,
                                656,661,666,671,676,681,686,691,696,701,706,711,716,721,726,731,736,741,746,751,756,761,766,771,776,781,786,
                                786,786,786,786,786,786,786,786,786,786,786,791,791,796,801,806,806,806,806,811,811,816,821,826,826,826,831,
                                836,841,846,851,856,856,856,856,861,861,866,871,876,876,881,881,886,891,896,901,906,906,911,916,921,921,926,
                                931,936,941,946,951,956,961,966,971,976,976,976,976,976,976,976,976,976,976,981,981,986,991,996,1001,1006,1006,
                                1011,1011,1016,1021,1026,1031,1036,1041,1046,1051,1056,1061,1066,1071,1076,1076,1081,1081,1086,1091,1096,1101,
                                1106,1106,1111,1111,1116,1121,1126,1131,1136,1141,1146,1151,1156,1161,1166,1171,1176,1181,1186,1191,1196,1201,
                                1206,1211,1216,1216,1216,1216,1216,1216,1216,1216,1216,1216,1221,1226,1231,1231,1236,1241,1246,1246,1251,1256,
                                1261,1261,1266,1271,1276,1281,1286,1291,1296,1301,1306,1311,1316,1316,1321,1326,1331,1331,1336,1341,1346,1351,
                                1356,1361,1366,1371,1376,1381,1386,1386,1391,1396,1401,1401,1406,1411,1416,1421,1426,1431,1436,1441,1446,1451,
                                1456,1456,1456,1456,1456,1456,1456,1456,1456,1461,1466,1471,1476,1481,1486,1491,1496,1501,1506,1511,1516,1521,
                                1526,1531,1536,1541,1546,1551,1556,1561,1566,1571,1576,1581,1586,1591,1596,1601,1606,1611,1616,1621,1626,1631,
                                1636,1641,1646,1651,1656,1661,1666,1671,1676,1681,1686,1691,1696,1701,1706,1711,1716,1721,1726,1731,1736,1736,
                                1736,1736,1736,1736,1736,1736,1736,1736,1736,1736,1741,1741,1746,1751,1756,1756,1756,1756,1761,1761,1766,1771,
                                1776,1776,1776,1781,1786,1791,1796,1801,1806,1806,1806,1806,1811,1811,1816,1821,1826,1826,1831,1831,1836,1841,
                                1846,1851,1856,1856,1861,1866,1871,1871,1876,1881,1886,1891,1896,1901,1906,1911,1916,1921,1926,1926,1926,1926,
                                1931,1931,1936,1941,1946,1946,1946,1946,1951,1951,1956,1961,1966,1966,1966,1971,1976,1981,1986,1991,1996,1996,
                                1996,2001,2006,2011,2016,2021,2026,2026,2031,2031,2036,2041,2046,2051,2056,2056,2061,2061,2066,2071,2076,2081,
                                2086,2091,2096,2101,2106,2111,2116,2121,2126,2131,2136,2141,2146,2151,2156,2161,2166,2166,2166,2166,2171,2171,
                                2176,2181,2186,2186,2186,2191,2196,2201,2206,2211,2216,2216,2216,2216,2221,2221,2226,2231,2236,2236,2236,2241,
                                2246,2251,2256,2261,2266,2266,2271,2276,2281,2281,2286,2291,2296,2301,2306,2311,2316,2321,2326,2331,2336,2336,
                                2341,2346,2351,2351,2356,2361,2366,2371,2376,2381,2386,2391,2396,2401,2406,2406,2406,2411,2416,2421,2426,2431,
                                2436,2436,2436,2441,2446,2451,2456,2461,2466,2466,2466,2471,2476,2481,2486,2491,2496,2496,2496,2501,2506,2511,
                                2516,2521,2526,2531,2536,2541,2546,2551,2556,2561,2566,2571,2576,2581,2586,2591,2596,2601,2606,2611,2616,2621,
                                2626,2631,2636,2641,2646,2651,2656,2661,2666,2671,2676,2681,2686,2686,2686,2686,2691,2691,2696,2701,2706,2706,
                                2711,2711,2716,2721,2726,2731,2736,2736,2741,2746,2751,2751,2756,2761,2766,2771,2776,2781,2786,2791,2796,2801,
                                2806,2806,2806,2806,2811,2811,2816,2821,2826,2826,2831,2831,2836,2841,2846,2851,2856,2856,2861,2866,2871,2871,
                                2876,2881,2886,2891,2896,2901,2906,2911,2916,2921,2926,2926,2931,2931,2936,2941,2946,2951,2956,2956,2961,2961,
                                2966,2971,2976,2981,2986,2991,2996,3001,3006,3011,3016,3021,3026,3031,3036,3041,3046,3051,3056,3061,3066,3066,
                                3071,3071,3076,3081,3086,3091,3096,3096,3101,3101,3106,3111,3116,3121,3126,3131,3136,3141,3146,3151,3156,3161,
                                3166,3171,3176,3181,3186,3191,3196,3201,3206,3206,3211,3216,3221,3221,3226,3231,3236,3241,3246,3251,3256,3261,
                                3266,3271,3276,3276,3281,3286,3291,3291,3296,3301,3306,3311,3316,3321,3326,3331,3336,3341,3346,3346,3351,3356,
                                3361,3361,3366,3371,3376,3381,3386,3391,3396,3401,3406,3411,3416,3416,3421,3426,3431,3431,3436,3441,3446,3451,
                                3456,3461,3466,3471,3476,3481,3486,3491,3496,3501,3506,3511,3516,3521,3526,3531,3536,3541,3546,3551,3556,3561,
                                3566,3571,3576,3581,3586,3591,3596,3601,3606,3611,3616,3621,3626,3631,3636,3641,3646,3651,3656,3661,3666,3671,
                                3676,3681,3686,3691,3696,3701,3706,3711,3716,3721,3726,3731,3736,3741,3746,3751,3756,3761,3766,3771,3776,3781,
                                3786,3791,3796,3801};
static int lookup_orbits[] = {0,0,2,1,1,1,2,1,1,1,2,3,3,3,7,6,6,6,5,5,4,4,5,4,5,4,11,10,10,9,5,5,4,4,6,7,6,6,4,5,5,4,10,11,10,9,5,4,4,5,11,10,
                              9,10,4,5,4,5,10,11,9,10,8,8,8,8,13,13,12,12,5,4,5,4,4,5,5,4,6,6,7,6,10,10,11,9,5,4,4,5,11,9,10,10,4,4,5,5,8,8,8,
                              8,10,9,11,10,13,12,13,12,4,5,4,5,4,4,5,5,8,8,8,8,9,11,10,10,9,10,11,10,12,13,13,12,6,6,6,7,10,10,9,11,10,9,10,11,
                              13,12,12,13,9,10,10,11,12,13,12,13,12,12,13,13,14,14,14,14,23,22,22,22,22,21,20,18,19,19,21,18,20,19,19,33,32,32,
                              31,31,21,20,19,18,19,20,21,19,19,18,16,16,17,15,15,26,26,25,24,24,21,18,19,20,19,33,32,31,32,31,16,16,15,17,15,
                              26,26,24,25,24,38,36,37,37,35,42,41,40,40,39,21,19,20,18,19,16,17,16,15,15,20,19,21,19,18,26,25,26,24,24,21,19,
                              18,20,19,33,31,32,32,31,16,15,16,17,15,38,37,36,37,35,26,24,26,25,24,42,40,41,40,39,16,17,15,16,15,16,15,17,16,
                              15,38,37,37,36,35,28,30,29,29,27,28,29,30,29,27,47,48,48,46,45,20,19,19,21,18,26,25,24,26,24,26,24,25,26,24,42,
                              40,40,41,39,28,29,29,30,27,47,48,46,48,45,47,46,48,48,45,58,57,57,57,56,21,20,19,19,18,20,21,19,18,19,16,16,17,
                              15,15,26,26,25,24,24,20,21,18,19,19,22,23,22,22,22,18,21,20,19,19,32,33,32,31,31,16,16,15,17,15,26,26,24,25,24,
                              18,21,19,20,19,32,33,31,32,31,36,38,37,37,35,41,42,40,40,39,17,16,16,15,15,19,21,20,18,19,19,20,21,19,18,25,26,
                              26,24,24,17,16,15,16,15,30,28,29,29,27,15,16,17,16,15,37,38,37,36,35,29,28,30,29,27,48,47,48,46,45,19,21,18,20,
                              19,15,16,16,17,15,37,38,36,37,35,31,33,32,32,31,24,26,26,25,24,40,42,41,40,39,19,20,19,21,18,25,26,24,26,24,29,
                              28,29,30,27,48,47,46,48,45,24,26,25,26,24,40,42,40,41,39,46,47,48,48,45,57,58,57,57,56,21,18,19,19,20,33,32,31,
                              31,32,16,16,15,15,17,26,26,24,24,25,38,36,37,35,37,42,41,40,39,40,16,16,15,15,17,26,26,24,24,25,18,21,19,19,20,
                              32,33,31,31,32,36,38,37,35,37,41,42,40,39,40,38,36,35,37,37,42,41,39,40,40,36,38,35,37,37,41,42,39,40,40,50,50,
                              49,49,49,55,55,54,54,54,17,15,16,15,16,30,29,28,27,29,15,17,16,15,16,29,30,28,27,29,37,37,38,35,36,48,48,47,45,
                              46,17,15,15,16,16,30,29,27,28,29,30,27,29,29,28,44,43,43,43,43,34,34,34,34,34,53,53,51,51,52,53,51,53,52,51,61,
                              60,60,59,59,15,17,15,16,16,29,30,27,28,29,34,34,34,34,34,53,53,51,51,52,27,30,29,29,28,43,44,43,43,43,51,53,53,
                              52,51,60,61,60,59,59,37,37,35,38,36,48,48,45,47,46,53,51,52,53,51,61,60,59,60,59,51,53,52,53,51,60,61,59,60,59,
                              63,63,64,64,62,67,67,66,66,65,21,19,20,19,18,16,17,16,15,15,20,19,21,18,19,26,25,26,24,24,17,16,16,15,15,19,21,
                              20,19,18,19,20,21,18,19,25,26,26,24,24,17,15,16,16,15,30,29,28,29,27,15,17,16,16,15,29,30,28,29,27,37,37,38,36,
                              35,48,48,47,46,45,20,18,21,19,19,18,20,21,19,19,22,22,23,22,22,32,32,33,31,31,16,15,16,17,15,26,24,26,25,24,18,
                              19,21,20,19,36,37,38,37,35,32,31,33,32,31,41,40,42,40,39,15,16,16,17,15,19,18,21,20,19,37,36,38,37,35,24,26,26,
                              25,24,31,32,33,32,31,40,41,42,40,39,19,19,20,21,18,29,29,28,30,27,25,24,26,26,24,48,46,47,48,45,24,25,26,26,24,
                              46,48,47,48,45,40,40,42,41,39,57,57,58,57,56,21,19,18,19,20,33,31,32,31,32,16,15,16,15,17,38,37,36,35,37,26,24,
                              26,24,25,42,40,41,39,40,17,16,15,15,16,30,28,29,27,29,15,16,17,15,16,37,38,37,35,36,29,28,30,27,29,48,47,48,45,
                              46,17,15,15,16,16,30,29,27,29,28,30,27,29,28,29,44,43,43,43,43,34,34,34,34,34,53,53,51,52,51,53,51,53,51,52,61,
                              60,60,59,59,16,15,16,15,17,26,24,26,24,25,18,19,21,19,20,36,37,38,35,37,32,31,33,31,32,41,40,42,39,40,38,35,36,
                              37,37,42,39,41,40,40,36,35,38,37,37,50,49,50,49,49,41,39,42,40,40,55,54,55,54,54,15,15,17,16,16,34,34,34,34,34,
                              29,27,30,28,29,53,51,53,51,52,27,29,30,29,28,51,53,53,52,51,43,43,44,43,43,60,60,61,59,59,37,35,37,38,36,53,52,
                              51,53,51,48,45,48,47,46,61,59,60,60,59,51,52,53,53,51,63,64,63,64,62,60,59,61,60,59,67,66,67,66,65,16,17,15,15,
                              16,16,15,17,15,16,38,37,37,35,36,28,30,29,27,29,28,29,30,27,29,47,48,48,45,46,19,21,18,19,20,15,16,16,15,17,37,
                              38,36,35,37,31,33,32,31,32,24,26,26,24,25,40,42,41,39,40,15,17,15,16,16,29,30,27,29,28,34,34,34,34,34,53,53,51,
                              52,51,27,30,29,28,29,43,44,43,43,43,51,53,53,51,52,60,61,60,59,59,15,16,16,15,17,19,18,21,19,20,37,36,38,35,37,
                              24,26,26,24,25,31,32,33,31,32,40,41,42,39,40,15,15,17,16,16,34,34,34,34,34,29,27,30,29,28,53,51,53,52,51,27,29,
                              30,28,29,51,53,53,51,52,43,43,44,43,43,60,60,61,59,59,35,38,36,37,37,35,36,38,37,37,49,50,50,49,49,39,42,41,40,
                              40,39,41,42,40,40,54,55,55,54,54,35,37,37,38,36,52,53,51,53,51,52,51,53,53,51,64,63,63,64,62,45,48,48,47,46,59,
                              61,60,60,59,59,60,61,60,59,66,67,67,66,65,20,19,19,18,21,26,25,24,24,26,26,24,25,24,26,42,40,40,39,41,28,29,29,
                              27,30,47,48,46,45,48,47,46,48,45,48,58,57,57,56,57,19,20,19,18,21,25,26,24,24,26,29,28,29,27,30,48,47,46,45,48,
                              24,26,25,24,26,40,42,40,39,41,46,47,48,45,48,57,58,57,56,57,37,37,35,36,38,48,48,45,46,47,53,51,52,51,53,61,60,
                              59,59,60,51,53,52,51,53,60,61,59,59,60,63,63,64,62,64,67,67,66,65,66,19,19,20,18,21,29,29,28,27,30,25,24,26,24,
                              26,48,46,47,45,48,24,25,26,24,26,46,48,47,45,48,40,40,42,39,41,57,57,58,56,57,37,35,37,36,38,53,52,51,51,53,48,
                              45,48,46,47,61,59,60,59,60,51,52,53,51,53,63,64,63,62,64,60,59,61,59,60,67,66,67,65,66,35,37,37,36,38,52,53,51,
                              51,53,52,51,53,51,53,64,63,63,62,64,45,48,48,46,47,59,61,60,59,60,59,60,61,59,60,66,67,67,65,66,49,49,49,50,50,
                              64,64,62,63,63,64,62,64,63,63,69,68,68,68,68,62,64,64,63,63,68,69,68,68,68,68,68,69,68,68,71,71,71,70,70,21,19,
                              19,20,18,17,16,15,16,15,17,15,16,16,15,30,29,29,28,27,16,17,15,16,15,19,21,19,20,18,15,17,16,16,15,29,30,29,28,
                              27,20,19,18,21,19,26,25,24,26,24,19,20,18,21,19,25,26,24,26,24,37,37,36,38,35,48,48,46,47,45,16,15,17,16,15,15,
                              16,17,16,15,19,19,21,20,18,29,29,30,28,27,20,18,19,21,19,26,24,25,26,24,19,18,20,21,19,37,36,37,38,35,25,24,26,
                              26,24,48,46,48,47,45,18,20,19,21,19,18,19,20,21,19,36,37,37,38,35,24,26,25,26,24,24,25,26,26,24,46,48,48,47,45,
                              22,22,22,23,22,32,32,31,33,31,32,31,32,33,31,41,40,40,42,39,31,32,32,33,31,40,41,40,42,39,40,40,41,42,39,57,57,
                              57,58,56,21,19,19,18,20,17,16,15,15,16,17,15,16,15,16,30,29,29,27,28,33,31,31,32,32,30,28,27,29,29,30,27,28,29,
                              29,44,43,43,43,43,16,15,15,16,17,38,37,35,36,37,15,16,15,17,16,37,38,35,37,36,34,34,34,34,34,53,53,52,51,51,26,
                              24,24,26,25,42,40,39,41,40,29,28,27,30,29,48,47,45,48,46,53,51,51,53,52,61,60,59,60,59,16,15,15,16,17,38,35,37,
                              36,37,15,15,16,17,16,34,34,34,34,34,37,35,38,37,36,53,52,53,51,51,26,24,24,26,25,42,39,40,41,40,29,27,28,30,29,
                              53,51,51,53,52,48,45,47,48,46,61,59,60,60,59,18,19,19,21,20,36,37,35,38,37,36,35,37,38,37,50,49,49,50,49,27,29,
                              29,30,28,51,53,52,53,51,51,52,53,53,51,63,64,64,63,62,32,31,31,33,32,41,40,39,42,40,41,39,40,42,40,55,54,54,55,
                              54,43,43,43,44,43,60,60,59,61,59,60,59,60,61,59,67,66,66,67,65,16,17,15,15,16,19,21,19,18,20,15,17,16,15,16,29,
                              30,29,27,28,16,15,15,17,16,38,37,35,37,36,15,16,15,16,17,37,38,35,36,37,34,34,34,34,34,53,53,52,51,51,28,30,27,
                              29,29,31,33,31,32,32,27,30,28,29,29,43,44,43,43,43,28,29,27,30,29,47,48,45,48,46,24,26,24,26,25,40,42,39,41,40,
                              51,53,51,53,52,60,61,59,60,59,15,16,15,16,17,15,15,16,17,16,34,34,34,34,34,35,38,37,36,37,35,37,38,37,36,52,53,
                              53,51,51,19,18,19,21,20,37,36,35,38,37,29,27,29,30,28,53,51,52,53,51,35,36,37,38,37,49,50,49,50,49,52,51,53,53,
                              51,64,63,64,63,62,24,26,24,26,25,27,29,28,30,29,51,53,51,53,52,39,42,40,41,40,45,48,47,48,46,59,61,60,60,59,31,
                              32,31,33,32,40,41,39,42,40,43,43,43,44,43,60,60,59,61,59,39,41,40,42,40,54,55,54,55,54,59,60,60,61,59,66,67,66,
                              67,65,20,19,18,19,21,26,25,24,24,26,19,20,18,19,21,25,26,24,24,26,37,37,36,35,38,48,48,46,45,47,26,24,24,25,26,
                              42,40,39,40,41,29,28,27,29,30,48,47,45,46,48,53,51,51,52,53,61,60,59,59,60,28,29,27,29,30,47,48,45,46,48,24,26,
                              24,25,26,40,42,39,40,41,51,53,51,52,53,60,61,59,59,60,47,46,45,48,48,58,57,56,57,57,46,47,45,48,48,57,58,56,57,
                              57,63,63,62,64,64,67,67,65,66,66,19,19,18,20,21,29,29,27,28,30,37,35,36,37,38,53,52,51,51,53,35,37,36,37,38,52,
                              53,51,51,53,49,49,50,49,50,64,64,63,62,63,25,24,24,26,26,48,46,45,47,48,48,45,46,48,47,61,59,59,60,60,52,51,51,
                              53,53,64,63,62,63,64,64,62,63,64,63,69,68,68,68,68,24,25,24,26,26,46,48,45,47,48,51,52,51,53,53,63,64,62,63,64,
                              45,48,46,48,47,59,61,59,60,60,62,64,63,64,63,68,69,68,68,68,40,40,39,42,41,57,57,56,58,57,60,59,59,61,60,67,66,
                              65,67,66,59,60,59,61,60,66,67,65,67,66,68,68,68,69,68,71,71,70,71,70,16,15,17,15,16,15,16,17,15,16,19,19,21,18,
                              20,29,29,30,27,28,16,15,15,17,16,38,35,37,37,36,15,15,16,16,17,34,34,34,34,34,37,35,38,36,37,53,52,53,51,51,15,
                              16,15,17,16,15,15,16,16,17,34,34,34,34,34,35,38,37,37,36,35,37,38,36,37,52,53,53,51,51,19,19,18,21,20,29,29,27,
                              30,28,37,35,36,38,37,53,52,51,53,51,35,37,36,38,37,52,53,51,53,51,49,49,50,50,49,64,64,63,63,62,28,27,30,29,29,
                              27,28,30,29,29,31,31,33,32,32,43,43,44,43,43,28,27,29,30,29,47,45,48,48,46,24,24,26,26,25,51,51,53,53,52,40,39,
                              42,41,40,60,59,61,60,59,27,28,29,30,29,24,24,26,26,25,51,51,53,53,52,45,47,48,48,46,39,40,42,41,40,59,60,61,60,
                              59,31,31,32,33,32,43,43,43,44,43,40,39,41,42,40,60,59,60,61,59,39,40,41,42,40,59,60,60,61,59,54,54,55,55,54,66,
                              66,67,67,65,20,18,19,19,21,26,24,25,24,26,19,18,20,19,21,37,36,37,35,38,25,24,26,24,26,48,46,48,45,47,26,24,24,
                              25,26,42,39,40,40,41,29,27,28,29,30,53,51,51,52,53,48,45,47,46,48,61,59,60,59,60,19,18,19,20,21,37,36,35,37,38,
                              29,27,29,28,30,53,51,52,51,53,35,36,37,37,38,49,50,49,49,50,52,51,53,51,53,64,63,64,62,63,25,24,24,26,26,48,46,
                              45,48,47,48,45,46,47,48,61,59,59,60,60,52,51,51,53,53,64,63,62,64,63,64,62,63,63,64,69,68,68,68,68,28,27,29,29,
                              30,47,45,48,46,48,24,24,26,25,26,51,51,53,52,53,40,39,42,40,41,60,59,61,59,60,47,45,46,48,48,58,56,57,57,57,46,
                              45,47,48,48,63,62,63,64,64,57,56,58,57,57,67,65,67,66,66,24,24,25,26,26,51,51,52,53,53,46,45,48,47,48,63,62,64,
                              63,64,45,46,48,48,47,62,63,64,64,63,59,59,61,60,60,68,68,69,68,68,40,39,40,42,41,60,59,59,61,60,57,56,57,58,57,
                              67,65,66,67,66,59,59,60,61,60,68,68,68,69,68,66,65,67,67,66,71,70,71,71,70,18,20,19,19,21,18,19,20,19,21,36,37,
                              37,35,38,24,26,25,24,26,24,25,26,24,26,46,48,48,45,47,18,19,19,20,21,36,37,35,37,38,36,35,37,37,38,50,49,49,49,
                              50,27,29,29,28,30,51,53,52,51,53,51,52,53,51,53,63,64,64,62,63,24,26,24,25,26,27,29,28,29,30,51,53,51,52,53,39,
                              42,40,40,41,45,48,47,46,48,59,61,60,59,60,24,25,24,26,26,46,48,45,48,47,51,52,51,53,53,63,64,62,64,63,45,48,46,
                              47,48,59,61,59,60,60,62,64,63,63,64,68,69,68,68,68,27,28,29,29,30,24,24,26,25,26,51,51,53,52,53,45,47,48,46,48,
                              39,40,42,40,41,59,60,61,59,60,24,24,25,26,26,51,51,52,53,53,46,45,48,48,47,63,62,64,64,63,45,46,48,47,48,62,63,
                              64,63,64,59,59,61,60,60,68,68,69,68,68,45,47,46,48,48,45,46,47,48,48,62,63,63,64,64,56,58,57,57,57,56,57,58,57,
                              57,65,67,67,66,66,39,40,40,42,41,59,60,59,61,60,59,59,60,61,60,68,68,68,69,68,56,57,57,58,57,65,67,66,67,66,65,
                              66,67,67,66,70,71,71,71,70,22,22,22,22,23,32,32,31,31,33,32,31,32,31,33,41,40,40,39,42,31,32,32,31,33,40,41,40,
                              39,42,40,40,41,39,42,57,57,57,56,58,32,31,31,32,33,41,40,39,40,42,41,39,40,40,42,55,54,54,54,55,43,43,43,43,44,
                              60,60,59,59,61,60,59,60,59,61,67,66,66,65,67,31,32,31,32,33,40,41,39,40,42,43,43,43,43,44,60,60,59,59,61,39,41,
                              40,40,42,54,55,54,54,55,59,60,60,59,61,66,67,66,65,67,40,40,39,41,42,57,57,56,57,58,60,59,59,60,61,67,66,65,66,
                              67,59,60,59,60,61,66,67,65,66,67,68,68,68,68,69,71,71,70,70,71,31,31,32,32,33,43,43,43,43,44,40,39,41,40,42,60,
                              59,60,59,61,39,40,41,40,42,59,60,60,59,61,54,54,55,54,55,66,66,67,65,67,40,39,40,41,42,60,59,59,60,61,57,56,57,
                              57,58,67,65,66,66,67,59,59,60,60,61,68,68,68,68,69,66,65,67,66,67,71,70,71,70,71,39,40,40,41,42,59,60,59,60,61,
                              59,59,60,60,61,68,68,68,68,69,56,57,57,57,58,65,67,66,66,67,65,66,67,66,67,70,71,71,70,71,54,54,54,55,55,66,66,
                              65,67,67,66,65,66,67,67,71,70,70,71,71,65,66,66,67,67,70,71,70,71,71,70,70,71,71,71,72,72,72,72,72};

class IncGraphNetwork {
public:
  IncGraphNetwork(int amntNodes): amntNodes(amntNodes) {
    neighbours = NULL;
    neighbours_matrix = NULL;
    reset();
  };
  ~IncGraphNetwork() {
    if (neighbours != NULL) {
      delete[] neighbours;
      neighbours = NULL;
    }
    if (neighbours_matrix != NULL) {
      delete[] neighbours_matrix;
      neighbours_matrix = NULL;
    }
  }

  void reset() {
    if (neighbours != NULL) {
      delete[] neighbours;
      neighbours = NULL;
    }
    if (neighbours_matrix != NULL) {
      delete[] neighbours_matrix;
      neighbours_matrix = NULL;
    }
    neighbours = new fs[amntNodes]();
    if ((long long)amntNodes*amntNodes < 1024LL*1024*1024*8) {
      neighbours_matrix = new int[amntNodes*amntNodes/adj_chunk+1]();

    }
  }

  void setNetwork(IntegerMatrix linkMatrix) {
    reset();
    for (int i = 0; i < linkMatrix.nrow(); ++i) {
      int x = linkMatrix(i, 0);
      int y = linkMatrix(i, 1);
      neighbours[x].insert(y);
      neighbours[y].insert(x);
    }
    if ((long long)amntNodes*amntNodes < 1024LL*1024*1024*8) {
      for (int i = 0; i < linkMatrix.nrow(); ++i) {
        int x = linkMatrix(i, 0);
        int y = linkMatrix(i, 1);
    		neighbours_matrix[(x*amntNodes+y)/adj_chunk]|=(1<<((x*amntNodes+y)%adj_chunk));
    		neighbours_matrix[(y*amntNodes+x)/adj_chunk]|=(1<<((y*amntNodes+x)%adj_chunk));
      }
    }
  }

  bool contains(int x, int y) {
    if (neighbours_matrix != NULL)
      return neighbours_matrix[(x*amntNodes+y)/adj_chunk]>>((x*amntNodes+y)%adj_chunk) & 1;
    else
      return neighbours[x].find(y) != neighbours[x].end();
  }

  bool containsInt(int x, int y) {
    if (neighbours_matrix != NULL)
      return neighbours_matrix[(x*amntNodes+y)/adj_chunk]>>((x*amntNodes+y)%adj_chunk) & 1;
    else
      return neighbours[x].find(y) != neighbours[x].end()?1:0;
  }

  IntegerVector getNeighbours(int x) {
    return wrap(neighbours[x]);
  }

  void flip(int x, int y) {
    if (contains(x, y)) {
      fs::iterator itx = neighbours[x].find(y);
      neighbours[x].erase(itx);
      fs::iterator ity = neighbours[y].find(x);
      neighbours[y].erase(ity);
    } else {
      neighbours[x].insert(y);
      neighbours[y].insert(x);
    }

    if (neighbours_matrix != NULL) {
      neighbours_matrix[(x*amntNodes+y)/adj_chunk] ^= (1<<((x*amntNodes+y)%adj_chunk));
      neighbours_matrix[(y*amntNodes+x)/adj_chunk] ^= (1<<((y*amntNodes+x)%adj_chunk));
    }
  }

  List calculateDelta(int i0, int i1) {
    IntegerMatrix delta_add_mat(amntNodes, 73);
    IntegerMatrix delta_rem_mat(amntNodes, 73);

    int *nodes = new int[5]();
    bool *taken0 = new bool[amntNodes]();
    bool *taken1 = new bool[amntNodes]();
    bool *taken2 = new bool[amntNodes]();

    int mod0, mod1;
    if (!contains(i0, i1)) {
      mod0 = 1;
      mod1 = 0;
    } else {
      mod0 = 0;
      mod1 = 1;
    }

    nodes[0] = i0;
    nodes[1] = i1;

    UpdateOrbits(delta_add_mat, delta_rem_mat, nodes, 0, mod0, mod1);

    for (int i = 0; i < amntNodes; ++i) {
      taken0[i] = false;
    }

    taken0[i0] = true;
    taken0[i1] = true;

    for (int j0 = 0; j0 < 2; ++j0) {
      fs nn2 = neighbours[nodes[j0]];
      for (fsi iter2 = nn2.begin(); iter2 != nn2.end(); ++iter2) {
        int i2 = *iter2;
        if (!taken0[i2]) {
          nodes[2] = i2;
          taken0[i2] = true;

          int x1 =
            W(i0, i2, 2) +
            W(i1, i2, 4);

          UpdateOrbits(delta_add_mat, delta_rem_mat, nodes, x1, mod0, mod1);
          memcpy(taken1, taken0, sizeof(bool) * amntNodes);

          for (int j1 = 0; j1 < 3; ++j1) {
            fs nn3 = neighbours[nodes[j1]];
            for (fsi iter3 = nn3.begin(); iter3 != nn3.end(); ++iter3) {
              int i3 = *iter3;
              if (!taken1[i3]) {
                nodes[3] = i3;
                taken1[i3] = true;

                int x2 = x1 +
                  W(i0, i3, 8) +
                  W(i1, i3, 16) +
                  W(i2, i3, 32);

                UpdateOrbits(delta_add_mat, delta_rem_mat, nodes, x2, mod0, mod1);
                memcpy(taken2, taken1, sizeof(bool) * amntNodes);

                for (int j2 = 0; j2 < 4; ++j2) {
                  fs nn4 = neighbours[nodes[j2]];
                  for (fsi iter4 = nn4.begin(); iter4 != nn4.end(); ++iter4) {
                    int i4 = *iter4;
                    if (!taken2[i4]) {
                      nodes[4] = i4;
                      taken2[i4] = true;

                      int x3 = x2 +
                        W(i0, i4, 64) +
                        W(i1, i4, 128) +
                        W(i2, i4, 256) +
                        W(i3, i4, 512);

                      UpdateOrbits(delta_add_mat, delta_rem_mat, nodes, x3, mod0, mod1);
                    }
                  }
                }
              }
            }
          }
        }
      }
    }

    delete[] nodes;
    delete[] taken0;
    delete[] taken1;
    delete[] taken2;

    return List::create(
      _["add"] = delta_add_mat,
      _["rem"] = delta_rem_mat
    );
  }

  int amntNodes;
private:
  fs *neighbours;
  int *neighbours_matrix; // compressed adjacency matrix

  int w(int i0, int i1, int w) {
    return containsInt(i0, i1)*w;
  }
};


RCPP_EXPOSED_CLASS(IncGraphModule)
RCPP_MODULE(IncGraphModule) {
  class_<IncGraphNetwork>( "incgraph.network" )

  .constructor<int>( "Create a new IncGraph network" )

  .field( "amnt.nodes", &IncGraphNetwork::amntNodes )

  .method ( "reset", &IncGraphNetwork::reset, "Reinitialise all data structures" )
  .method ( "set.network", &IncGraphNetwork::setNetwork, "Load a given network into the data structures" )
  .method ( "calculate.delta", &IncGraphNetwork::calculateDelta, "Calculate the delta matrix for a given edge modification" )
  .method ( "flip", &IncGraphNetwork::flip, "Add or remove a specific edge in the network" )
  .method ( "contains", &IncGraphNetwork::contains, "Returns true if the network contains a given edge" )
  .method ( "get.neighbours", &IncGraphNetwork::getNeighbours, "Return the neighbours of a node" )
  ;
}
